<?php

namespace App\Http\Controllers;

use App\Models\Book;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\Classroom;
use App\Models\Exam;
use Illuminate\Support\Str;
use App\Models\Downloadsetings;
use App\Models\Downloadfile;
use App\Models\Downloadcenter;
use PDF;
use Illuminate\Support\Carbon;
use App\Models\Noticeboard;
use App\Models\PaymentConfirmation;



class ApiController extends Controller
{
    public function login(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'student_id' => 'required',
            'password' => 'required',
        ]);

        // Check if the validation fails
        if ($validator->fails()) {
            // Return validation errors
            return response()->json([
                'status' => 'false',
                'errors' => $validator->errors(),
            ], 422); // HTTP status code 422 for Unprocessable Entity
        }
        if (Auth::attempt(['student_id' => $request->student_id, 'password' => $request->password])) {
            $user = Auth::user();
            $token = $user->createToken('API Token')->accessToken;

            return response()->json(['token' => $token], 200);


            // $randomToken = Str::random(32);
            // $user = User::where('student_id',$request->student_id)->first();
            // $user->token = $randomToken;
            // $user->save();
            // return response()->json([
            //     'status' => 'true',
            //     'token' => $user->token,
            // ], 200);
        } else {
            return response()->json([
                'status' => 'false',
                'error' => 'Unauthorized. Invalid student id or password.'
            ], 401);
        }
    }


    public function examData(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }
        // Fetch exams with the related subject data using eager loading
        $examsQuery = Exam::with('subject') // Eager load the subject relationship
            ->where('student_id', $user->id);

        // Check if year is set in the request
        $examsQuery->when(isset($request['year']), function ($query) use ($request) {
            $year = $request['year'];
            return $query->where(function ($query) use ($year) {
                $query->where('year', $year)
                    ->orWhere('created_at', 'LIKE', '%' . $year . '%');
            });
        });


        // Execute the query
        $exams = $examsQuery->get();

        // Check if exams were found
        if ($exams->isEmpty()) {
            return response()->json([
                'status' => 'true',
                'exam' => 'No data found'
            ], 200);
        }

        // Group exams by trimester and subject
        $groupedExams = $exams->groupBy(['trimester_id', 'subject_id']);

        // Prepare the response data
        $responseData = [];

        foreach ($groupedExams as $trimesterId => $subjects) {
            $trimesterData = [];
            foreach ($subjects as $subjectId => $examGroup) {
                $subjectData = [
                    'subject_id' => $subjectId,
                    'subject_name' => $examGroup->first()->subject->name, // Get the subject name from the first exam in the group
                    'exams' => []
                ];

                // Add each exam in the subject group to the subject data
                foreach ($examGroup as $exam) {
                    $subjectData['exams'][] = [
                        'id' => $exam->id,
                        'ACS1' => $exam->ACS1a,
                        'ACS2' => $exam->ACS2a,
                        'MACS' => $exam->MACS,
                        'AT' => $exam->AT,
                        'MT' => $exam->MT,
                        'NE' => $exam->NE,
                        'is_active' => $exam->is_active,
                        'created_at' => $exam->created_at,
                        'updated_at' => $exam->updated_at,
                        'year' => $exam->year,
                        // Add other fields as necessary
                    ];
                }

                // Add the subject data to the trimester data
                $trimesterData[] = $subjectData;
            }

            // Add the trimester data to the response data
            $responseData[] = [
                'trimester_id' => $trimesterId,
                'subjects' => $trimesterData
            ];
        }

        // Return the grouped exam data in the response
        return response()->json([
            'status' => 'true',
            'exams' => $responseData
        ], 200);
    }


    public function listarMatrizes(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        } else {

            if ($user->role === 0) {
                // Admin user can access all downloads
                $downloads = Downloadsetings::orderByDesc('created_at')->get(); // Remove paginate() here
            } else {
                // Non-admin user's access based on role or permissions
                $downloads = Downloadsetings::where('type', 'all')
                    ->orWhere('class_id', $user->classroom_id)
                    ->orderByDesc('created_at')->get(); // Remove paginate() here
            }

            if ($downloads->isEmpty()) {
                return response()->json([
                    'status' => 'true',
                    'downloads' => "No data found"
                ], 200);
            }

            $data = [];
            foreach ($downloads as $model) {
                $downloadCenter = Downloadcenter::where('slug', $model->download_id)->first();
                $downloadFile = Downloadfile::where('download_id', $model->download_id)->first();
                $item = [
                    'id' => $model->id,
                    'type' => $model->type,
                    'class_id' => $model->class_id,
                    'student_id' => $model->student_id,
                    'active' => $model->active,
                    'download_id' => $model->download_id,
                    'media_name' => $downloadFile ? $downloadFile->media_name : null,
                    'media_url' => $downloadFile ? $downloadFile->media_url : null,
                    'author' => $downloadCenter ? $downloadCenter->author : null,
                    'title' => $downloadCenter ? $downloadCenter->title : null,
                    'description' => $downloadCenter ? $downloadCenter->description : null,
                    'created_at' => $model->created_at,
                    'updated_at' => $model->updated_at,
                    // Add more fields as needed
                ];
                $data[] = $item;
            }

            return response()->json([
                'status' => 'success',
                'downloads' => $data
            ], 200);
        }
    }
    


    public function generatePdfReport(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);

        } else {
            // Get the current date
            $currentYear = Carbon::now()->year;

            // Calculate the last year
            $lastYear = $currentYear - 1;

            // Fetch exams based on the selected criteria, where created_at date is in the last year
            $exams = Exam::query()
                ->where('student_id', $user->id)
                ->whereYear('created_at', $lastYear) // Filter by the last year
                ->get();

            $data = [
                'title' => 'COLÉGIO POLITÉCNICO DE MOÇAMBIQUE',
                'date' => date('d/m/Y'),
                'models' => $exams,
                'name' => $user->name,
                'id' => $user->student_id,
            ];

            // Generate the PDF
            $pdf = PDF::loadView('exports.userreport', $data);

            // Set options for the PDF (e.g., paper size, orientation, etc.)
            $pdf->setPaper('A4', 'portrait');

            // Add a page number
            $pdf->setOptions(['isPhpEnabled' => true]);

            // Return the PDF file as a response
            return $pdf->stream();
        }
    }



    public function listarComunicados(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        } else {


            $comunicados = Noticeboard::where('type', 'all')
                ->orWhere('class_id', $user->classroom_id)
                ->orWhere('student_id', $user->id)
                ->orderByDesc('created_at')
                ->get();

            if ($comunicados->isEmpty()) {
                return response()->json([
                    'status' => 'true',
                    'comunicados' => "No data found"
                ], 200);
            } else {
                return response()->json([
                    'status' => 'true',
                    'comunicados' => $comunicados
                ], 200);
            }
        }
    }

    public function ListarConfirmacoesPagamento(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        } else {
            // Fetch payment confirmations based on the user's role or permissions
            if ($user->role === 0) {
                // Admin user can access all payment confirmations
                $confirmations = PaymentConfirmation::orderByDesc('created_at')->get();

            } else {
                // Non-admin user's access based on role or permissions
                $confirmations = PaymentConfirmation::where('student_id', $user->id)
                    ->orWhere('student_id', $user->id)
                    ->orderByDesc('created_at')
                    ->get();
            }

            if ($confirmations->isEmpty()) {
                return response()->json([
                    'status' => 'true',
                    'confirmations' => "No data found"
                ], 200);
            }

            $data = [];
            foreach ($confirmations as $confirmation) {
                $item = [
                    'id' => $confirmation->id,
                    'uuid' => $confirmation->uuid,
                    'title' => $confirmation->title,
                    'type' => $confirmation->type,
                    'class_id' => $confirmation->class_id,
                    'student_id' => $confirmation->student_id,
                    'slug' => $confirmation->slug,
                    'media_name' => $confirmation->media_name,
                    'media_url' => $confirmation->media_url,
                    'created_at' => $confirmation->created_at,
                    'updated_at' => $confirmation->updated_at,
                    // Add more fields as needed
                ];
                $data[] = $item;
            }

            return response()->json([
                'status' => 'success',
                'confirmations' => $data
            ], 200);
        }
    }


    public function getUserClassName(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        // Get the user's classroom_id
        $classroomId = $user->classroom_id;

        // Fetch the classroom information based on the classroom_id
        $classroom = Classroom::find($classroomId);

        // Check if the classroom was found
        if (!$classroom) {
            return response()->json([
                'status' => '404',
                'message' => 'Classroom not found!'
            ], 404);
        }

        // Return the user's class name as JSON response
        return response()->json([
            'status' => 'success',
            'class' => $classroom->class,
            'turma' => $classroom->name
        ], 200);
    }


    public function getUserInfo(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }
        // Return the user information as a JSON response
        return response()->json([
            'status' => 'success',
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone_number' => $user->phone_number,
                'avatar' => $user->avatar,
                // Add more fields as needed (e.g., address, date of birth, etc.)
            ]
        ], 200);
    }

    public function listBooks(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        // Fetch books that are active
        $books = Book::where('is_active', 1)
            ->orderByDesc('created_at')
            ->get();

        // Check if there are no books
        if ($books->isEmpty()) {
            return response()->json([
                'status' => 'success',
                'books' => "No data found",
            ], 200);
        }

        // Prepare data to return
        $data = [];
        foreach ($books as $book) {
            $item = [
                'id' => $book->id,
                'title' => $book->title,
                'author' => $book->author,
                'description' => $book->description,
                'category_id' => $book->category_id,
                'slug' => $book->slug,
                'media_name' => $book->media_name,
                'media_url' => $book->media_url, // Cover image URL
                'pdf_name' => $book->pdf_name,
                'pdf_url' => $book->pdf_url, // PDF URL
                'created_at' => $book->created_at,
                'updated_at' => $book->updated_at,
                // Add more fields as needed
            ];
            $data[] = $item;
        }

        // Return JSON response with book data
        return response()->json([
            'status' => 'success',
            'books' => $data,
        ], 200);
    }



    public function storeDeviceToken(Request $request, $userId)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'device_token' => 'required|string',
        ]);

        // Check if the validation fails
        if ($validator->fails()) {
            // Return validation errors
            return response()->json([
                'error' => $validator->errors()->first(),
            ], 422);
        }

        // Find the user by ID
        $user = User::where('student_id', $userId)->first();

        // Check if the user exists
        if (!$user) {
            // Return an error response
            return response()->json([
                'error' => 'User not found',
            ], 404);
        }

        // Save the device token for the user
        $user->device_token = $request->device_token;
        $user->save();

        // Return a success response
        return response()->json([
            'message' => 'Device token saved successfully',
        ], 200);
    }


    public function removeDeviceToken(Request $request, $identifier)
    {
        // Find the user by email
        $user = User::where('email', $identifier)->first();

        // If user is not found by email, try searching by student ID
        if (!$user && ctype_alpha(substr($identifier, 0, 1))) {
            $user = User::where('student_id', $identifier)->first();
        }

        // Check if the user exists
        if (!$user) {
            // Return an error response
            return response()->json([
                'error' => 'User not found',
            ], 404);
        }

        // Remove the device token for the user
        $user->device_token = null; // or $user->device_token = ''; depending on your database setup
        $user->save();

        // Return a success response
        return response()->json([
            'message' => 'Device token removed successfully',
        ], 200);
    }


    //------------------------------------------------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------------------------------------------------

    protected $perPage = 30;

    /**
     * Authinticate the connection for pusher
     *
     * @param Request $request
     * @return void
     */
    public function pusherAuth(Request $request)
    {
        return Chatify::pusherAuth(
            $request->user(),
            Auth::user(),
            $request['channel_name'],
            $request['socket_id']
        );
    }

    /**
     * Fetch data by id for (user/group)
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function idFetchData(Request $request)
    {
        return auth()->user();
        // Favorite
        $favorite = Chatify::inFavorite($request['id']);

        // User data
        if ($request['type'] == 'user') {
            $fetch = User::where('id', $request['id'])->first();
            if ($fetch) {
                $userAvatar = Chatify::getUserWithAvatar($fetch)->avatar;
            }
        }

        // send the response
        return Response::json([
            'favorite' => $favorite,
            'fetch' => $fetch ?? null,
            'user_avatar' => $userAvatar ?? null,
        ]);
    }

    /**
     * This method to make a links for the attachments
     * to be downloadable.
     *
     * @param string $fileName
     * @return \Illuminate\Http\JsonResponse
     */
    public function download($fileName)
    {
        $path = config('chatify.attachments.folder') . '/' . $fileName;
        if (Chatify::storage()->exists($path)) {
            return response()->json([
                'file_name' => $fileName,
                'download_path' => Chatify::storage()->url($path)
            ], 200);
        } else {
            return response()->json([
                'message' => "Sorry, File does not exist in our server or may have been deleted!"
            ], 404);
        }
    }

    /**
     * Send a message to database
     *
     * @param Request $request
     * @return JSON response
     */
    public function send(Request $request)
    {
        // default variables
        $error = (object) [
            'status' => 0,
            'message' => null
        ];
        $attachment = null;
        $attachment_title = null;

        // if there is attachment [file]
        if ($request->hasFile('file')) {
            // allowed extensions
            $allowed_images = Chatify::getAllowedImages();
            $allowed_files = Chatify::getAllowedFiles();
            $allowed = array_merge($allowed_images, $allowed_files);

            $file = $request->file('file');
            // check file size
            if ($file->getSize() < Chatify::getMaxUploadSize()) {
                if (in_array(strtolower($file->extension()), $allowed)) {
                    // get attachment name
                    $attachment_title = $file->getClientOriginalName();
                    // upload attachment and store the new name
                    $attachment = Str::uuid() . "." . $file->extension();
                    $file->storeAs(config('chatify.attachments.folder'), $attachment, config('chatify.storage_disk_name'));
                } else {
                    $error->status = 1;
                    $error->message = "File extension not allowed!";
                }
            } else {
                $error->status = 1;
                $error->message = "File size you are trying to upload is too large!";
            }
        }

        if (!$error->status) {
            // send to database
            $message = Chatify::newMessage([
                'type' => $request['type'],
                'from_id' => Auth::user()->id,
                'to_id' => $request['id'],
                'body' => htmlentities(trim($request['message']), ENT_QUOTES, 'UTF-8'),
                'attachment' => ($attachment) ? json_encode((object) [
                    'new_name' => $attachment,
                    'old_name' => htmlentities(trim($attachment_title), ENT_QUOTES, 'UTF-8'),
                ]) : null,
            ]);

            // fetch message to send it with the response
            $messageData = Chatify::parseMessage($message);

            // send to user using pusher
            if (Auth::user()->id != $request['id']) {
                Chatify::push("private-chatify." . $request['id'], 'messaging', [
                    'from_id' => Auth::user()->id,
                    'to_id' => $request['id'],
                    'message' => $messageData
                ]);
            }
        }

        // send the response
        return Response::json([
            'status' => '200',
            'error' => $error,
            'message' => $messageData ?? [],
            'tempID' => $request['temporaryMsgId'],
        ]);
    }

    /**
     * fetch [user/group] messages from database
     *
     * @param Request $request
     * @return JSON response
     */
    public function fetch(Request $request)
    {
        $query = Chatify::fetchMessagesQuery($request['id'])->latest();
        $messages = $query->paginate($request->per_page ?? $this->perPage);
        $totalMessages = $messages->total();
        $lastPage = $messages->lastPage();
        $response = [
            'total' => $totalMessages,
            'last_page' => $lastPage,
            'last_message_id' => collect($messages->items())->last()->id ?? null,
            'messages' => $messages->items(),
        ];
        return Response::json($response);
    }

    /**
     * Make messages as seen
     *
     * @param Request $request
     * @return void
     */
    public function seen(Request $request)
    {
        // make as seen
        $seen = Chatify::makeSeen($request['id']);
        // send the response
        return Response::json([
            'status' => $seen,
        ], 200);
    }

    /**
     * Get contacts list
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse response
     */
    public function getContacts(Request $request)
    {
        // get all users that received/sent message from/to [Auth user]
        $users = Message::join('users', function ($join) {
            $join->on('ch_messages.from_id', '=', 'users.id')
                ->orOn('ch_messages.to_id', '=', 'users.id');
        })
            ->where(function ($q) {
                $q->where('ch_messages.from_id', Auth::user()->id)
                    ->orWhere('ch_messages.to_id', Auth::user()->id);
            })
            ->where('users.id', '!=', Auth::user()->id)
            ->select('users.*', DB::raw('MAX(ch_messages.created_at) max_created_at'))
            ->orderBy('max_created_at', 'desc')
            ->groupBy('users.id')
            ->paginate($request->per_page ?? $this->perPage);

        return response()->json([
            'contacts' => $users->items(),
            'total' => $users->total() ?? 0,
            'last_page' => $users->lastPage() ?? 1,
        ], 200);
    }

    /**
     * Put a user in the favorites list
     *
     * @param Request $request
     * @return void
     */
    public function favorite(Request $request)
    {
        $userId = $request['user_id'];
        // check action [star/unstar]
        $favoriteStatus = Chatify::inFavorite($userId) ? 0 : 1;
        Chatify::makeInFavorite($userId, $favoriteStatus);

        // send the response
        return Response::json([
            'status' => @$favoriteStatus,
        ], 200);
    }

    /**
     * Get favorites list
     *
     * @param Request $request
     * @return void
     */
    public function getFavorites(Request $request)
    {
        $favorites = Favorite::where('user_id', Auth::user()->id)->get();
        foreach ($favorites as $favorite) {
            $favorite->user = User::where('id', $favorite->favorite_id)->first();
        }
        return Response::json([
            'total' => count($favorites),
            'favorites' => $favorites ?? [],
        ], 200);
    }

    /**
     * Search in messenger
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function search(Request $request)
    {
        $input = trim(filter_var($request['input']));
        $records = User::where('id', '!=', Auth::user()->id)
            ->where('name', 'LIKE', "%{$input}%")
            ->paginate($request->per_page ?? $this->perPage);

        foreach ($records->items() as $index => $record) {
            $records[$index] += Chatify::getUserWithAvatar($record);
        }

        return Response::json([
            'records' => $records->items(),
            'total' => $records->total(),
            'last_page' => $records->lastPage()
        ], 200);
    }

    /**
     * Get shared photos
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function sharedPhotos(Request $request)
    {
        $images = Chatify::getSharedPhotos($request['user_id']);

        foreach ($images as $image) {
            $image = asset(config('chatify.attachments.folder') . $image);
        }
        // send the response
        return Response::json([
            'shared' => $images ?? [],
        ], 200);
    }

    /**
     * Delete conversation
     *
     * @param Request $request
     * @return void
     */
    public function deleteConversation(Request $request)
    {
        // delete
        $delete = Chatify::deleteConversation($request['id']);

        // send the response
        return Response::json([
            'deleted' => $delete ? 1 : 0,
        ], 200);
    }

    public function updateSettings(Request $request)
    {
        $msg = null;
        $error = $success = 0;

        // dark mode
        if ($request['dark_mode']) {
            $request['dark_mode'] == "dark"
                ? User::where('id', Auth::user()->id)->update(['dark_mode' => 1])  // Make Dark
                : User::where('id', Auth::user()->id)->update(['dark_mode' => 0]); // Make Light
        }

        // If messenger color selected
        if ($request['messengerColor']) {
            $messenger_color = trim(filter_var($request['messengerColor']));
            User::where('id', Auth::user()->id)
                ->update(['messenger_color' => $messenger_color]);
        }
        // if there is a [file]
        if ($request->hasFile('avatar')) {
            // allowed extensions
            $allowed_images = Chatify::getAllowedImages();

            $file = $request->file('avatar');
            // check file size
            if ($file->getSize() < Chatify::getMaxUploadSize()) {
                if (in_array(strtolower($file->extension()), $allowed_images)) {
                    // delete the older one
                    if (Auth::user()->avatar != config('chatify.user_avatar.default')) {
                        $path = Chatify::getUserAvatarUrl(Auth::user()->avatar);
                        if (Chatify::storage()->exists($path)) {
                            Chatify::storage()->delete($path);
                        }
                    }
                    // upload
                    $avatar = Str::uuid() . "." . $file->extension();
                    $update = User::where('id', Auth::user()->id)->update(['avatar' => $avatar]);
                    $file->storeAs(config('chatify.user_avatar.folder'), $avatar, config('chatify.storage_disk_name'));
                    $success = $update ? 1 : 0;
                } else {
                    $msg = "File extension not allowed!";
                    $error = 1;
                }
            } else {
                $msg = "File size you are trying to upload is too large!";
                $error = 1;
            }
        }

        // send the response
        return Response::json([
            'status' => $success ? 1 : 0,
            'error' => $error ? 1 : 0,
            'message' => $error ? $msg : 0,
        ], 200);
    }

    /**
     * Set user's active status
     *
     * @param Request $request
     * @return void
     */
    public function setActiveStatus(Request $request)
    {
        $activeStatus = $request['status'] > 0 ? 1 : 0;
        $status = User::where('id', Auth::user()->id)->update(['active_status' => $activeStatus]);
        return Response::json([
            'status' => $status,
        ], 200);
    }



}







