<?php

namespace App\Console\Commands;

use App\Models\Exam;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class AnalyzeDuplicateExams extends Command
{
    protected $signature = 'exams:analyze-duplicates {--year=} {--student-id=} {--code=} {--subject-id=} {--trimester-id=} {--show-details} {--raw}';
    protected $description = 'Analyze duplicate exam records and show which one has the most complete marks';

    public function handle()
    {
        $year = $this->option('year') ?: date('Y');
        $studentId = $this->option('student-id');
        $studentCode = $this->option('code');
        $subjectId = $this->option('subject-id');
        $trimesterId = $this->option('trimester-id');
        $showDetails = $this->option('show-details');
        $rawMode = $this->option('raw');

        // Se foi fornecido código do estudante, buscar o UUID
        if ($studentCode) {
            $student = DB::table('users')->where('student_id', $studentCode)->first();
            if (!$student) {
                $this->error("Student with code '{$studentCode}' not found!");
                return 1;
            }
            $studentId = $student->id;
            $this->info("Found student: {$student->name} (ID: {$studentId})");
        }

        $this->info("Analyzing duplicate exam records for year: {$year}");
        $this->line("Grouping by: student_id + subject_id + trimester_id + year (regardless of classroom)");
        $this->newLine();

        // Se usar --raw, busca duplicados directamente via SQL
        if ($rawMode) {
            return $this->analyzeRaw($year, $trimesterId, $studentId);
        }

        // Build the query - SoftDeletes trait already filters deleted_at automatically
        $query = Exam::where('year', $year);

        if ($studentId) {
            $query->where('student_id', $studentId);
        }
        if ($subjectId) {
            $query->where('subject_id', $subjectId);
        }
        if ($trimesterId) {
            $query->where('trimester_id', $trimesterId);
        }

        $exams = $query->with(['student', 'subject'])->get();

        $this->info("Total exam records found: " . $exams->count());

        if ($exams->isEmpty()) {
            $this->warn("No exam records found for year {$year}");

            // Show count from raw query to debug
            $rawCount = DB::table('exams')->where('year', $year)->count();
            $this->line("Raw count in database (including soft deleted): {$rawCount}");

            $rawCountNotDeleted = DB::table('exams')->where('year', $year)->whereNull('deleted_at')->count();
            $this->line("Raw count without soft deleted: {$rawCountNotDeleted}");

            return 0;
        }

        // Group by student + subject + trimester + year (ignore classroom)
        $duplicateGroups = $exams->groupBy(fn($exam) => "{$exam->student_id}_{$exam->subject_id}_{$exam->trimester_id}_{$exam->year}");

        $totalDuplicateGroups = 0;
        $totalDuplicateRecords = 0;
        $analysisResults = [];

        foreach ($duplicateGroups as $group) {
            if ($group->count() > 1) {
                $totalDuplicateGroups++;
                $totalDuplicateRecords += $group->count() - 1; // -1 because we keep one

                $analysis = $this->analyzeGroup($group);
                $analysisResults[] = $analysis;

                if ($showDetails) {
                    $this->displayGroupAnalysis($analysis);
                }
            }
        }

        $this->info("Total unique groups: " . $duplicateGroups->count());

        if ($totalDuplicateGroups === 0) {
            $this->info("✓ No duplicate exam records found!");
            $this->newLine();
            return 0;
        }

        // Summary
        $this->displaySummary($totalDuplicateGroups, $totalDuplicateRecords, $analysisResults);

        return 0;
    }

    private function analyzeGroup($group)
    {
        $firstExam = $group->first();
        $analysis = [
            'group_key' => $firstExam->student_id . '_' . $firstExam->subject_id . '_' . $firstExam->trimester_id . '_' . $firstExam->year,
            'student_name' => $firstExam->student->name ?? 'Unknown',
            'subject_name' => $firstExam->subject->name ?? 'Unknown',
            'trimester_id' => $firstExam->trimester_id,
            'year' => $firstExam->year,
            'total_records' => $group->count(),
            'records' => [],
            'recommended_action' => '',
            'best_record_id' => null,
            'reasoning' => ''
        ];

        foreach ($group as $exam) {
            $markCount = 0;
            $marks = [];

            // Count non-zero marks
            if ($exam->ACS1a > 0) {
                $markCount++;
                $marks['ACS1a'] = $exam->ACS1a;
            }
            if ($exam->ACS2a > 0) {
                $markCount++;
                $marks['ACS2a'] = $exam->ACS2a;
            }
            if ($exam->AT > 0) {
                $markCount++;
                $marks['AT'] = $exam->AT;
            }
            if ($exam->NE > 0) {
                $markCount++;
                $marks['NE'] = $exam->NE;
            }
            if ($exam->MACS > 0) {
                $marks['MACS'] = $exam->MACS;
            }
            if ($exam->MT > 0) {
                $marks['MT'] = $exam->MT;
            }

            $analysis['records'][] = [
                'id' => $exam->id,
                'classroom_id' => $exam->classroom_id,
                'mark_count' => $markCount,
                'marks' => $marks,
                'created_at' => $exam->created_at,
                'updated_at' => $exam->updated_at,
                'is_latest' => false // Will be set later
            ];
        }

        // Sort records by mark count (descending), then by updated_at (descending)
        usort($analysis['records'], function ($a, $b) {
            if ($a['mark_count'] == $b['mark_count']) {
                return $b['updated_at'] <=> $a['updated_at'];
            }
            return $b['mark_count'] <=> $a['mark_count'];
        });

        // Mark the latest record
        $latestRecord = collect($analysis['records'])->sortByDesc('updated_at')->first();
        foreach ($analysis['records'] as &$record) {
            if ($record['id'] == $latestRecord['id']) {
                $record['is_latest'] = true;
                break;
            }
        }

        // Determine best record and recommendation
        $bestRecord = $analysis['records'][0]; // Record with most marks
        $analysis['best_record_id'] = $bestRecord['id'];

        if ($bestRecord['mark_count'] > $latestRecord['mark_count']) {
            $analysis['recommended_action'] = 'KEEP_MOST_COMPLETE';
            $analysis['reasoning'] = "Record ID {$bestRecord['id']} has {$bestRecord['mark_count']} marks vs latest record ({$latestRecord['id']}) with {$latestRecord['mark_count']} marks";
        } elseif ($bestRecord['mark_count'] == $latestRecord['mark_count'] && $bestRecord['id'] == $latestRecord['id']) {
            $analysis['recommended_action'] = 'KEEP_LATEST';
            $analysis['reasoning'] = "Latest record also has the most complete marks ({$bestRecord['mark_count']} marks)";
        } else {
            $analysis['recommended_action'] = 'REVIEW_MANUALLY';
            $analysis['reasoning'] = "Multiple records with equal mark counts ({$bestRecord['mark_count']} marks) - manual review needed";
        }

        return $analysis;
    }

    private function displayGroupAnalysis($analysis)
    {
        $this->line("<fg=yellow>━━━ Duplicate Group Analysis ━━━</>");
        $this->line("Student: <fg=cyan>{$analysis['student_name']}</>");
        $this->line("Subject: <fg=cyan>{$analysis['subject_name']}</>");
        $this->line("Trimester: <fg=cyan>{$analysis['trimester_id']}</> | Year: <fg=cyan>{$analysis['year']}</>");
        $this->line("Total Records: <fg=red>{$analysis['total_records']}</>");
        $this->newLine();

        $this->line("<fg=magenta>Records Details:</>");
        $headers = ['ID', 'Classroom', 'Marks Count', 'Marks Detail', 'Created', 'Updated', 'Latest?'];
        $rows = [];

        foreach ($analysis['records'] as $record) {
            $marksDetail = [];
            foreach ($record['marks'] as $type => $value) {
                $marksDetail[] = "{$type}:{$value}";
            }

            $rows[] = [
                $record['id'],
                $record['classroom_id'],
                $record['mark_count'],
                implode(', ', $marksDetail),
                $record['created_at']->format('Y-m-d H:i'),
                $record['updated_at']->format('Y-m-d H:i'),
                $record['is_latest'] ? '✓' : ''
            ];
        }

        $this->table($headers, $rows);

        $this->line("<fg=green>Recommendation: {$analysis['recommended_action']}</>");
        $this->line("<fg=white>Reasoning: {$analysis['reasoning']}</>");
        $this->line("<fg=blue>Best Record ID: {$analysis['best_record_id']}</>");

        $this->newLine();
        $this->line("<fg=gray>━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━</>");
        $this->newLine();
    }

    private function displaySummary($totalGroups, $totalDuplicates, $analysisResults)
    {
        $this->newLine();
        $this->line("<fg=green>━━━ SUMMARY ━━━</>");
        $this->line("Total Duplicate Groups: <fg=red>{$totalGroups}</>");
        $this->line("Total Duplicate Records to Clean: <fg=red>{$totalDuplicates}</>");
        $this->newLine();

        // Count recommendations
        $recommendations = collect($analysisResults)->groupBy('recommended_action');

        $this->line("<fg=yellow>Cleanup Recommendations:</>");
        foreach ($recommendations as $action => $groups) {
            $count = $groups->count();
            $this->line("  {$action}: <fg=cyan>{$count}</> groups");
        }

        $this->newLine();

        // Show potential issues
        $manualReviewNeeded = $recommendations->get('REVIEW_MANUALLY', collect())->count();
        $keepMostComplete = $recommendations->get('KEEP_MOST_COMPLETE', collect())->count();

        if ($manualReviewNeeded > 0) {
            $this->warn("⚠️  {$manualReviewNeeded} groups need manual review due to equal mark counts");
        }

        if ($keepMostComplete > 0) {
            $this->info("ℹ️  {$keepMostComplete} groups should keep the most complete record (not the latest)");
        }

        $this->newLine();
        $this->line("<fg=blue>Suggested Commands:</>");
        $this->line("  • Run with --show-details to see detailed analysis");
        $this->line("  • Filter by student code: --code=20240001");
        $this->line("  • Filter by student UUID: --student-id=STUDENT_ID");
        $this->line("  • Filter by specific subject: --subject-id=SUBJECT_ID");
        $this->line("  • Filter by specific trimester: --trimester-id=TRIMESTER_ID");
        $this->line("  • Use --raw for direct SQL analysis");
    }

    /**
     * Análise directa via SQL - mais rápido e mostra duplicados directamente
     */
    private function analyzeRaw(int $year, ?string $trimesterId, ?string $studentId = null): int
    {
        $this->info("Using RAW SQL mode...");
        $this->newLine();

        // Query para encontrar duplicados directamente
        $query = DB::table('exams')
            ->select([
                'student_id',
                'subject_id',
                'trimester_id',
                'year',
                DB::raw('COUNT(*) as total_records'),
                DB::raw('GROUP_CONCAT(id ORDER BY updated_at DESC) as exam_ids'),
                DB::raw('GROUP_CONCAT(classroom_id ORDER BY updated_at DESC) as classroom_ids'),
            ])
            ->where('year', $year)
            ->whereNull('deleted_at')
            ->groupBy(['student_id', 'subject_id', 'trimester_id', 'year'])
            ->having(DB::raw('COUNT(*)'), '>', 1);

        if ($trimesterId) {
            $query->where('trimester_id', $trimesterId);
        }

        if ($studentId) {
            $query->where('student_id', $studentId);
        }

        $duplicates = $query->get();

        if ($duplicates->isEmpty()) {
            $this->info("✓ No duplicate exam records found!");

            // Mostrar estatísticas gerais
            $totalRecords = DB::table('exams')->where('year', $year)->whereNull('deleted_at')->count();
            $this->line("Total exam records for {$year}: {$totalRecords}");

            return 0;
        }

        $this->error("Found {$duplicates->count()} duplicate groups!");
        $this->newLine();

        // Mostrar tabela de duplicados
        $headers = ['Student ID', 'Subject ID', 'Trimester', 'Year', 'Count', 'Exam IDs', 'Classrooms'];
        $rows = [];

        foreach ($duplicates as $dup) {
            // Obter nomes do estudante e disciplina
            $student = DB::table('users')->where('id', $dup->student_id)->first();
            $subject = DB::table('subjects')->where('id', $dup->subject_id)->first();

            $studentName = $student->name ?? 'Unknown';
            $subjectName = $subject->name ?? 'Unknown';

            // Truncar IDs se muito longos
            $examIds = strlen($dup->exam_ids) > 40
                ? substr($dup->exam_ids, 0, 37) . '...'
                : $dup->exam_ids;

            $rows[] = [
                substr($studentName, 0, 25),
                substr($subjectName, 0, 20),
                $dup->trimester_id,
                $dup->year,
                $dup->total_records,
                $examIds,
                $dup->classroom_ids,
            ];
        }

        $this->table($headers, $rows);

        // Resumo
        $totalDuplicateRecords = $duplicates->sum('total_records') - $duplicates->count();
        $this->newLine();
        $this->line("<fg=yellow>━━━ SUMMARY ━━━</>");
        $this->line("Total Duplicate Groups: <fg=red>{$duplicates->count()}</>");
        $this->line("Total Records to Remove: <fg=red>{$totalDuplicateRecords}</>");
        $this->newLine();

        // Mostrar SQL para investigar um específico
        $this->line("<fg=blue>To investigate a specific student:</>");
        $this->line("  php artisan exams:analyze-duplicates --year={$year} --code=STUDENT_CODE --show-details");
        $this->line("  php artisan exams:analyze-duplicates --year={$year} --code=STUDENT_CODE --raw");

        return 0;
    }
}
