<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Carbon\Carbon;

class CleanBciFiles extends Command
{
    protected $signature = 'bci:clean
                            {--days=30 : Apagar ficheiros com mais de X dias}
                            {--dry-run : Mostrar o que seria apagado sem apagar}';

    protected $description = 'Limpar ficheiros BCI antigos das pastas processed, failed e reports';

    public function handle()
    {
        $days = (int) $this->option('days');
        $dryRun = $this->option('dry-run');

        $baseDir = public_path('bci/bmeps');
        $folders = ['processed', 'failed', 'reports'];

        $totalDeleted = 0;
        $totalSize = 0;

        $this->info("A procurar ficheiros com mais de {$days} dias...");
        if ($dryRun) {
            $this->warn('MODO DRY-RUN: Nenhum ficheiro será apagado');
        }
        $this->newLine();

        foreach ($folders as $folder) {
            $path = $baseDir . '/' . $folder;

            if (!is_dir($path)) {
                $this->line("  Pasta não existe: {$folder}/");
                continue;
            }

            $files = File::files($path);
            $oldFiles = [];
            $folderSize = 0;

            foreach ($files as $file) {
                $modifiedTime = Carbon::createFromTimestamp($file->getMTime());

                if ($modifiedTime->diffInDays(now()) >= $days) {
                    $oldFiles[] = $file;
                    $folderSize += $file->getSize();
                }
            }

            $count = count($oldFiles);
            $sizeFormatted = $this->formatBytes($folderSize);

            if ($count > 0) {
                $this->info("  {$folder}/: {$count} ficheiros ({$sizeFormatted})");

                if (!$dryRun) {
                    foreach ($oldFiles as $file) {
                        @unlink($file->getPathname());
                    }
                    $totalDeleted += $count;
                    $totalSize += $folderSize;
                }
            } else {
                $this->line("  {$folder}/: 0 ficheiros antigos");
            }
        }

        $this->newLine();

        if ($dryRun) {
            $this->info("Total que seria apagado: {$totalDeleted} ficheiros");
        } else {
            $sizeFormatted = $this->formatBytes($totalSize);
            $this->info("Total apagado: {$totalDeleted} ficheiros ({$sizeFormatted})");
        }

        return 0;
    }

    private function formatBytes($bytes): string
    {
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        }
        return $bytes . ' bytes';
    }
}
