<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Exception;

class ClearPaymentsAndFees extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'payments:clear
                            {--payments : Clear only payments (fee_assigns, payment_references, payment_reconciliations)}
                            {--fees : Clear only fee structures/plans}
                            {--force : Skip confirmation prompt}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clear all payments and/or fee plans from the system';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->warn('========================================');
        $this->warn('⚠️  CLEAR PAYMENTS AND FEES');
        $this->warn('========================================');

        $clearPayments = $this->option('payments');
        $clearFees = $this->option('fees');

        // If no specific option is set, clear both
        if (!$clearPayments && !$clearFees) {
            $clearPayments = true;
            $clearFees = true;
        }

        // Show what will be cleared
        $this->info('This will clear the following:');
        if ($clearPayments) {
            $this->line('  - Payment assignments (fee_assigns)');
            $this->line('  - Payment references (payment_references)');
            $this->line('  - Payment reconciliations (payment_reconciliations)');
        }
        if ($clearFees) {
            $this->line('  - Fee structures (fee_structures)');
            $this->line('  - Fee structure items (fee_structure_items)');
        }

        // Get counts before clearing
        $counts = $this->getRecordCounts($clearPayments, $clearFees);
        $this->info('');
        $this->info('Current Records:');

        if ($clearPayments) {
            $this->line("  - Fee Assigns: {$counts['fee_assigns']}");
            $this->line("  - Payment References: {$counts['payment_references']}");
            $this->line("  - Payment Reconciliations: {$counts['payment_reconciliations']}");
        }
        if ($clearFees) {
            $this->line("  - Fee Structures: {$counts['fee_structures']}");
            $this->line("  - Fee Structure Items: {$counts['fee_structure_items']}");
        }

        // Confirmation
        if (!$this->option('force')) {
            $this->warn('');
            $this->warn('⚠️  WARNING: This action cannot be undone!');

            if (!$this->confirm('Are you sure you want to proceed?')) {
                $this->info('Operation cancelled.');
                return Command::SUCCESS;
            }

            // Double confirmation for safety
            if (!$this->confirm('This is your last chance. Proceed with deletion?')) {
                $this->info('Operation cancelled.');
                return Command::SUCCESS;
            }
        }

        try {
            $startTime = Carbon::now();

            DB::beginTransaction();

            $deleted = [];

            if ($clearPayments) {
                $this->info('');
                $this->info('Clearing payments...');

                // Clear payment reconciliations first (no foreign key dependencies)
                $deleted['payment_reconciliations'] = DB::table('payment_reconciliations')->delete();
                $this->line("✓ Deleted {$deleted['payment_reconciliations']} payment reconciliations");

                // Clear fee assigns
                $deleted['fee_assigns'] = DB::table('fee_assigns')->delete();
                $this->line("✓ Deleted {$deleted['fee_assigns']} fee assigns");

                // Clear payment references
                $deleted['payment_references'] = DB::table('payment_references')->delete();
                $this->line("✓ Deleted {$deleted['payment_references']} payment references");
            }

            if ($clearFees) {
                $this->info('');
                $this->info('Clearing fee structures...');

                // Clear fee structure items first (has foreign key to fee_structures)
                $deleted['fee_structure_items'] = DB::table('fee_structure_items')->delete();
                $this->line("✓ Deleted {$deleted['fee_structure_items']} fee structure items");

                // Clear fee structures
                $deleted['fee_structures'] = DB::table('fee_structures')->delete();
                $this->line("✓ Deleted {$deleted['fee_structures']} fee structures");
            }

            DB::commit();

            $endTime = Carbon::now();
            $duration = $endTime->diffInSeconds($startTime);

            // Display results
            $this->info('');
            $this->info('========================================');
            $this->info('✅ Successfully Cleared');
            $this->info('========================================');

            $totalDeleted = array_sum($deleted);
            $this->info("Total records deleted: {$totalDeleted}");
            $this->info("Processing time: {$duration} seconds");
            $this->info('Completed at: ' . $endTime->format('Y-m-d H:i:s'));

            // Log the operation
            Log::warning('Payments and fees cleared', [
                'cleared_payments' => $clearPayments,
                'cleared_fees' => $clearFees,
                'deleted_counts' => $deleted,
                'executed_at' => $endTime->toDateTimeString()
            ]);

            return Command::SUCCESS;

        } catch (Exception $e) {
            DB::rollBack();

            $this->error('');
            $this->error('Failed to clear records: ' . $e->getMessage());

            if ($this->option('verbose')) {
                $this->error('Stack trace:');
                $this->error($e->getTraceAsString());
            }

            Log::error('Failed to clear payments and fees', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return Command::FAILURE;
        }
    }

    /**
     * Get current record counts
     */
    private function getRecordCounts($clearPayments, $clearFees)
    {
        $counts = [];

        if ($clearPayments) {
            $counts['fee_assigns'] = DB::table('fee_assigns')->count();
            $counts['payment_references'] = DB::table('payment_references')->count();
            $counts['payment_reconciliations'] = DB::table('payment_reconciliations')->count();
        }

        if ($clearFees) {
            $counts['fee_structures'] = DB::table('fee_structures')->count();
            $counts['fee_structure_items'] = DB::table('fee_structure_items')->count();
        }

        return $counts;
    }
}
