<?php

namespace App\Console\Commands;

use App\Models\SystemSetting;
use App\Services\SmsService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class MonitorBmeps extends Command
{
    protected $signature = 'bmeps:monitor';

    protected $description = 'Monitoriza a chegada de ficheiros BMEPS do banco e envia alerta SMS se parar';

    public function handle(): int
    {
        $incomingPath = public_path('bci/bmeps/incoming');
        $maxHoursWithoutFiles = 2;
        $problems = [];

        // 1. Verificar conectividade SFTP ao servidor (testa se config e utilizador existem)
        try {
            $host = config('filesystems.disks.sftp.host');
            $port = config('filesystems.disks.sftp.port', 22);
            $username = config('filesystems.disks.sftp.username');
            $password = config('filesystems.disks.sftp.password');

            $sftp = new \phpseclib3\Net\SFTP($host, $port, 10);
            if (!$sftp->login($username, $password)) {
                $problems[] = 'SFTP login falhou - verificar utilizador copmoz-bci ou config sshd no servidor';
            } else {
                $this->line('SFTP: Conexao OK');
                $sftp->disconnect();
            }
        } catch (\Exception $e) {
            $problems[] = 'SFTP inacessivel - servidor pode estar em baixo (' . $e->getMessage() . ')';
        }

        // 2. Verificar se pasta incoming existe e tem permissões
        if (!is_dir($incomingPath)) {
            $problems[] = "Pasta incoming nao existe: {$incomingPath}";
        } elseif (!is_writable($incomingPath)) {
            $problems[] = 'Pasta incoming sem permissao de escrita';
        }

        // 3. Verificar se chegaram ficheiros recentes do banco
        if (is_dir($incomingPath)) {
            $files = glob($incomingPath . '/*.txt') ?: [];
            $files = array_merge($files, glob($incomingPath . '/*.TXT') ?: []);

            if (empty($files)) {
                // Sem ficheiros no incoming — verificar processed/ para ver último ficheiro
                $processedPath = public_path('bci/bmeps/processed');
                $processedFiles = is_dir($processedPath) ? (glob($processedPath . '/*.txt') ?: []) : [];
                $processedFiles = array_merge($processedFiles, is_dir($processedPath) ? (glob($processedPath . '/*.TXT') ?: []) : []);

                $latestTime = 0;
                foreach ($processedFiles as $file) {
                    $mtime = filemtime($file);
                    if ($mtime > $latestTime) {
                        $latestTime = $mtime;
                    }
                }

                if ($latestTime > 0) {
                    $hoursSince = (time() - $latestTime) / 3600;
                    if ($hoursSince > $maxHoursWithoutFiles) {
                        $problems[] = 'Sem ficheiros do banco ha ' . round($hoursSince) . ' horas (ultimo: ' . date('d/m/Y H:i', $latestTime) . ')';
                    }
                } else {
                    $problems[] = 'Nenhum ficheiro do banco encontrado (incoming e processed vazios)';
                }
            } else {
                // Verificar idade do ficheiro mais recente no incoming
                $latestTime = 0;
                foreach ($files as $file) {
                    $mtime = filemtime($file);
                    if ($mtime > $latestTime) {
                        $latestTime = $mtime;
                    }
                }

                $hoursSince = (time() - $latestTime) / 3600;
                if ($hoursSince > $maxHoursWithoutFiles) {
                    $problems[] = 'Ficheiro mais recente no incoming tem ' . round($hoursSince) . ' horas (de ' . date('d/m/Y H:i', $latestTime) . ')';
                }
            }
        }

        // Se não há problemas, tudo ok
        if (empty($problems)) {
            $this->info('BMEPS Monitor: Tudo OK - ficheiros a chegar normalmente.');
            Log::info('BMEPS Monitor: OK');
            return 0;
        }

        // Há problemas — registar e enviar SMS
        $message = "ALERTA SIGA COPMOZ\n\n"
            . "Problemas detectados no SFTP bancario:\n\n"
            . implode("\n", array_map(fn($p) => "- {$p}", $problems))
            . "\n\nVerifique o servidor urgentemente.";

        Log::warning('BMEPS Monitor: Problemas detectados', ['problems' => $problems]);
        $this->warn('BMEPS Monitor: Problemas detectados!');
        foreach ($problems as $p) {
            $this->line("  - {$p}");
        }

        // Enviar SMS ao admin
        $adminPhone = $this->getAdminPhone();
        if ($adminPhone) {
            try {
                $smsService = app(SmsService::class);
                $result = $smsService->send($adminPhone, $message);

                if ($result['success']) {
                    $this->info("SMS de alerta enviado para {$adminPhone}");
                    Log::info('BMEPS Monitor: SMS alerta enviado', ['phone' => $adminPhone]);
                } else {
                    $this->error("Falha ao enviar SMS: " . ($result['error'] ?? 'Erro desconhecido'));
                    Log::error('BMEPS Monitor: Falha SMS', ['error' => $result['error'] ?? null]);
                }
            } catch (\Exception $e) {
                $this->error("Erro ao enviar SMS: " . $e->getMessage());
                Log::error('BMEPS Monitor: Erro SMS', ['error' => $e->getMessage()]);
            }
        } else {
            $this->warn('Numero de admin nao configurado. Configure em System Settings > bmeps_alert_phone');
        }

        return 1;
    }

    private function getAdminPhone(): ?string
    {
        try {
            $phone = SystemSetting::get('bmeps_alert_phone', '258849285587', 'bmeps');
            return !empty($phone) ? $phone : null;
        } catch (\Exception $e) {
            return '258849285587';
        }
    }
}
