<?php

namespace App\Console\Commands;

use App\Services\PaymentReconciliationService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class ProcessBmepsFiles extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'bmeps:process
                            {--force : Force processing even if no files found}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Processa automaticamente arquivos BMEPS/MEPS pendentes para reconciliação de pagamentos';

    protected $reconciliationService;

    /**
     * Create a new command instance.
     */
    public function __construct(PaymentReconciliationService $reconciliationService)
    {
        parent::__construct();
        $this->reconciliationService = $reconciliationService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔄 Iniciando processamento automático de arquivos BMEPS/MEPS...');

        // Auto-processing started

        try {
            $results = $this->reconciliationService->processAllPendingFiles();

            if ($results['total_files'] === 0) {
                $this->warn('⚠️  Nenhum arquivo encontrado para processar.');
                // No files found

                if (!$this->option('force')) {
                    return 0;
                }
            }

            // Exibir estatísticas
            $this->newLine();
            $this->info('✅ Processamento concluído!');
            $this->newLine();

            $this->table(
                ['Estatística', 'Valor'],
                [
                    ['Arquivos Processados', $results['total_files'] ?? 0],
                    ['Total de Registros', $results['total_records'] ?? 0],
                    ['Reconciliados', $results['reconciled'] ?? 0],
                    ['Duplicados', $results['duplicates'] ?? 0],
                    ['Não Encontrados', $results['not_found'] ?? 0],
                    ['Falharam', $results['failed'] ?? 0],
                ]
            );

            Log::info('BMEPS completed', ['reconciled' => $results['reconciled'] ?? 0, 'failed' => $results['failed'] ?? 0]);

            // Verificar se houve erros
            if (($results['failed'] ?? 0) > 0) {
                $this->error('⚠️  Alguns pagamentos falharam. Verifique os logs para mais detalhes.');
                return 1;
            }

            return 0;

        } catch (\Exception $e) {
            $this->error('❌ Erro durante o processamento: ' . $e->getMessage());

            Log::error('BMEPS Auto-processing failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return 1;
        }
    }
}
