<?php

namespace App\Console\Commands;

use App\Models\Guardian;
use App\Models\PaymentReference;
use App\Models\PaymentReminderSetting;
use App\Services\SmsService;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SendPaymentReminders extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'payment:send-reminders {--dry-run : Run without actually sending SMS}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Envia lembretes SMS configuráveis de pagamento';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $isDryRun = $this->option('dry-run');
        $today = now()->format('Y-m-d');
        $currentTime = now()->format('H:i');
        $currentHour = now()->format('H');

        // Obter configurações
        $frequency = cache('payment_reminders_frequency', 1); // 1 ou 2 vezes por dia
        $sendTime1 = cache('payment_reminders_send_time', '09:00');
        $sendTime2 = cache('payment_reminders_send_time_2', '14:00');

        // Obter execuções já realizadas hoje
        $runsToday = cache('payment_reminders_runs_today', []);
        if (!is_array($runsToday) || (isset($runsToday['date']) && $runsToday['date'] !== $today)) {
            // Reset se mudou o dia
            $runsToday = ['date' => $today, 'runs' => []];
        }

        // Determinar qual horário verificar
        $sendTime1Hour = explode(':', $sendTime1)[0];
        $sendTime2Hour = explode(':', $sendTime2)[0];

        $shouldRunNow = false;
        $runNumber = null;

        if ($frequency == 1) {
            // 1x por dia - verificar primeiro horário
            if ($currentHour >= $sendTime1Hour && !in_array(1, $runsToday['runs'] ?? [])) {
                $shouldRunNow = true;
                $runNumber = 1;
            }
        } else {
            // 2x por dia - verificar ambos os horários
            if ($currentHour >= $sendTime1Hour && !in_array(1, $runsToday['runs'] ?? [])) {
                $shouldRunNow = true;
                $runNumber = 1;
            } elseif ($currentHour >= $sendTime2Hour && !in_array(2, $runsToday['runs'] ?? [])) {
                $shouldRunNow = true;
                $runNumber = 2;
            }
        }

        if (!$isDryRun && !$shouldRunNow) {
            $completedRuns = count($runsToday['runs'] ?? []);
            $maxRuns = $frequency;

            if ($completedRuns >= $maxRuns) {
                $this->info("✅ Todos os envios de hoje já foram realizados ({$completedRuns}/{$maxRuns}).");
            } else {
                $nextTime = $runNumber === null ? ($completedRuns == 0 ? $sendTime1 : $sendTime2) : '';
                $this->info("⏰ Ainda não é hora de enviar. Próximo horário: " . ($completedRuns == 0 ? $sendTime1 : $sendTime2) . ", Hora atual: {$currentTime}");
            }
            return 0;
        }

        if ($isDryRun) {
            $this->warn('🔍 DRY RUN MODE - Nenhum SMS será enviado');
            $runNumber = 1; // Para dry run
        }

        $this->info('🚀 Iniciando envio de lembretes de pagamento...');
        $this->info('Data/Hora: ' . now()->format('d/m/Y H:i:s'));
        $this->info("⏰ Configuração: {$frequency}x por dia");
        if ($frequency == 1) {
            $this->info("   Horário: {$sendTime1}");
        } else {
            $this->info("   Horários: {$sendTime1} e {$sendTime2}");
            $this->info("   Executando: " . ($runNumber == 1 ? "1º envio ({$sendTime1})" : "2º envio ({$sendTime2})"));
        }
        $this->newLine();

        // Buscar configurações de lembretes ativas
        $reminderSettings = PaymentReminderSetting::enabled()
            ->ordered()
            ->get();

        if ($reminderSettings->isEmpty()) {
            $this->warn('⚠️  Nenhuma configuração de lembrete ativa encontrada.');
            return 0;
        }

        $this->info("📋 Encontradas {$reminderSettings->count()} configurações de lembrete ativas.");
        $this->newLine();

        $totalSent = 0;
        $totalFailed = 0;
        $totalSkipped = 0;

        foreach ($reminderSettings as $setting) {
            $this->info("📅 Processando: {$setting->name} ({$setting->days_before} dias antes)...");

            // Calcular data alvo
            $targetDate = Carbon::now()->addDays($setting->days_before)->startOfDay();
            $endOfTargetDate = $targetDate->copy()->endOfDay();

            // Determinar qual campo de tracking usar baseado em days_before
            $trackingField = $this->getTrackingField($setting->days_before);

            // Buscar referências que precisam receber este lembrete
            $references = $this->getReferencesForReminder($setting, $targetDate, $endOfTargetDate, $trackingField);

            if ($references->isEmpty()) {
                $this->info("   ✅ Nenhuma referência encontrada.");
            } else {
                $this->info("   📋 Encontradas {$references->count()} referências.");

                foreach ($references as $reference) {
                    $result = $this->sendReminder($reference, $setting, $trackingField, $isDryRun);
                    if ($result === true) {
                        $totalSent++;
                    } elseif ($result === false) {
                        $totalFailed++;
                    } else {
                        $totalSkipped++;
                    }
                }
            }

            $this->newLine();
        }

        $this->info('📊 Resumo:');
        $this->info("   ✅ Enviados: {$totalSent}");
        $this->info("   ⏭️  Ignorados: {$totalSkipped}");
        $this->info("   ❌ Falhas: {$totalFailed}");

        // Marcar esta execução como concluída
        if (!$isDryRun) {
            $runsToday = cache('payment_reminders_runs_today', ['date' => $today, 'runs' => []]);
            if (!is_array($runsToday) || !isset($runsToday['runs'])) {
                $runsToday = ['date' => $today, 'runs' => []];
            }
            if (!in_array($runNumber, $runsToday['runs'])) {
                $runsToday['runs'][] = $runNumber;
            }
            $runsToday['date'] = $today;
            cache()->put('payment_reminders_runs_today', $runsToday, now()->endOfDay());
        }

        Log::info('Payment reminders completed', [
            'sent' => $totalSent,
            'skipped' => $totalSkipped,
            'failed' => $totalFailed,
            'run_number' => $runNumber,
            'frequency' => $frequency ?? 1,
            'dry_run' => $isDryRun,
        ]);

        return 0;
    }

    /**
     * Retorna o campo de tracking apropriado baseado em days_before
     */
    private function getTrackingField(int $daysBefore): string
    {
        return match ($daysBefore) {
            0 => 'due_date_reminder_sent_at',
            2 => 'first_reminder_sent_at',
            default => 'first_reminder_sent_at', // Fallback
        };
    }

    /**
     * Busca referências que precisam receber o lembrete
     */
    private function getReferencesForReminder($setting, $targetDate, $endOfTargetDate, $trackingField)
    {
        $query = PaymentReference::with('student')
            ->where('approval_status', 'approved')
            ->where('status', 'pending')
            ->whereNull($trackingField)  // Ainda não recebeu este lembrete
            ->whereNotNull('due_date')
            ->whereBetween('due_date', [$targetDate, $endOfTargetDate]);

        // Se for o lembrete do dia de vencimento (days_before = 0),
        // garantir que já recebeu o primeiro lembrete
        if ($setting->days_before == 0) {
            $query->whereNotNull('first_reminder_sent_at');
        }

        return $query->get();
    }

    /**
     * Envia lembrete SMS usando o template configurado
     */
    private function sendReminder(PaymentReference $reference, PaymentReminderSetting $setting, string $trackingField, bool $isDryRun)
    {
        try {
            if (!$reference->student || !$reference->student->mobile) {
                $this->warn("   ⏭️  Ref {$reference->reference_number}: Sem número de telemóvel");
                return null;
            }

            $monthName = $this->getPortugueseMonth($reference->fee_month);
            $dueDate = Carbon::parse($reference->due_date)->format('d/m/Y');

            // Preparar dados para o template
            $templateData = [
                'student_name' => $reference->student->name,
                'fee_month' => $monthName,
                'fee_year' => $reference->fee_year,
                'entity_code' => $reference->entity_code,
                'reference_number' => $reference->reference_number,
                'amount' => number_format($reference->amount, 2),
                'due_date' => $dueDate,
                'days_left' => $setting->days_before,
            ];

            // Processar template
            $message = $setting->processTemplate($templateData);

            $this->line("📌 Processando: {$reference->reference_number}");
            $this->line("   Estudante: {$reference->student->name}");
            $this->line("   Tipo: {$setting->name}");
            $this->line("   Vencimento: {$dueDate}");

            if (!$isDryRun) {
                DB::beginTransaction();

                try {
                    $smsService = new SmsService();

                    // Enviar SMS ao estudante
                    $result = $smsService->send($reference->student->mobile, $message);

                    if (!$result['success']) {
                        throw new \Exception($result['error'] ?? 'Erro ao enviar SMS');
                    }

                    // Enviar SMS aos encarregados do estudante
                    $guardians = Guardian::whereHas('students', function ($q) use ($reference) {
                        $q->where('users.id', $reference->student_id);
                    })->get();

                    foreach ($guardians as $guardian) {
                        if (!empty($guardian->phone)) {
                            try {
                                $smsService->send($guardian->phone, $message);
                                $this->line("   📱 SMS enviado ao encarregado: {$guardian->name}");
                            } catch (\Exception $e) {
                                Log::warning('Failed to send reminder SMS to guardian', [
                                    'guardian_id' => $guardian->id,
                                    'guardian_name' => $guardian->name,
                                    'error' => $e->getMessage(),
                                ]);
                            }
                        }
                    }

                    // Atualizar campo de tracking
                    $reference->update([
                        $trackingField => now(),
                    ]);

                    DB::commit();

                    $this->info("   ✅ SMS enviado com sucesso!");

                    Log::info('Payment reminder SMS sent', [
                        'reference_id' => $reference->id,
                        'reference_number' => $reference->reference_number,
                        'student_id' => $reference->student_id,
                        'reminder_setting_id' => $setting->id,
                        'reminder_name' => $setting->name,
                        'days_before' => $setting->days_before,
                        'gateway' => $result['gateway'],
                        'guardians_notified' => $guardians->count(),
                    ]);

                    return true;

                } catch (\Exception $e) {
                    DB::rollback();
                    throw $e;
                }
            } else {
                // Dry run - also show guardian info
                $guardians = Guardian::whereHas('students', function ($q) use ($reference) {
                    $q->where('users.id', $reference->student_id);
                })->get();
                $this->info("   ✅ [DRY RUN] SMS seria enviado ao estudante + {$guardians->count()} encarregado(s)");
                return true;
            }

        } catch (\Exception $e) {
            $this->error("   ❌ Erro: {$e->getMessage()}");
            Log::error('Error sending reminder SMS', [
                'reference_id' => $reference->id,
                'reference_number' => $reference->reference_number,
                'reminder_setting_id' => $setting->id,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Converte mês para português
     */
    private function getPortugueseMonth($monthName): string
    {
        // Se já é string (nome do mês), tentar mapear
        if (is_string($monthName) && !is_numeric($monthName)) {
            $portugueseMonths = [
                'janeiro' => 'Janeiro', 'fevereiro' => 'Fevereiro', 'março' => 'Março',
                'abril' => 'Abril', 'maio' => 'Maio', 'junho' => 'Junho',
                'julho' => 'Julho', 'agosto' => 'Agosto', 'setembro' => 'Setembro',
                'outubro' => 'Outubro', 'novembro' => 'Novembro', 'dezembro' => 'Dezembro',
            ];

            $englishToPortuguese = [
                'january' => 'Janeiro', 'february' => 'Fevereiro', 'march' => 'Março',
                'april' => 'Abril', 'may' => 'Maio', 'june' => 'Junho',
                'july' => 'Julho', 'august' => 'Agosto', 'september' => 'Setembro',
                'october' => 'Outubro', 'november' => 'Novembro', 'december' => 'Dezembro',
            ];

            $monthLower = strtolower($monthName);

            if (isset($portugueseMonths[$monthLower])) {
                return $portugueseMonths[$monthLower];
            }

            if (isset($englishToPortuguese[$monthLower])) {
                return $englishToPortuguese[$monthLower];
            }

            return ucfirst($monthName);
        }

        // Se é número, mapear diretamente
        $months = [
            1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
            5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
            9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
        ];

        return $months[$monthName] ?? $monthName;
    }
}
