<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use Carbon\Carbon;

class FeeCollectionExport implements FromCollection, WithHeadings, WithMapping, WithStyles, ShouldAutoSize, WithEvents
{
    protected $fees;
    protected $totalAmount = 0;
    protected $totalDiscount = 0;
    protected $totalFine = 0;
    protected $totalNet = 0;

    public function __construct($fees)
    {
        $this->fees = $fees;
        $this->calculateTotals();
    }

    protected function calculateTotals()
    {
        foreach ($this->fees as $fee) {
            $this->totalAmount += $fee->amount ?? 0;
            $this->totalDiscount += $fee->discount ?? 0;
            $this->totalFine += $fee->fine ?? 0;
            $this->totalNet += ($fee->amount ?? 0) - ($fee->discount ?? 0) + ($fee->fine ?? 0);
        }
    }

    public function collection()
    {
        return $this->fees;
    }

    public function headings(): array
    {
        return [
            'ID Pagamento',
            'Data Pagamento',
            'ID Estudante',
            'Nome do Estudante',
            'Classe',
            'Tipo de Taxa',
            'Valor (MZN)',
            'Desconto (MZN)',
            'Multa (MZN)',
            'Total (MZN)',
            'Método de Pagamento',
            'Status',
            'ID Transação',
            'Mês',
            'Ano'
        ];
    }

    public function map($fee): array
    {
        return [
            $fee->id,
            $fee->pay_day ? Carbon::parse($fee->pay_day)->format('d/m/Y') : '-',
            $fee->student_id,
            $fee->user->name ?? 'N/A',
            $fee->feeStructure->grade ?? '-',
            $fee->feeStructure->fee_name ?? '-',
            number_format($fee->amount ?? 0, 2, ',', '.'),
            number_format($fee->discount ?? 0, 2, ',', '.'),
            number_format($fee->fine ?? 0, 2, ',', '.'),
            number_format(($fee->amount ?? 0) - ($fee->discount ?? 0) + ($fee->fine ?? 0), 2, ',', '.'),
            $this->translatePaymentMethod($fee->pay_type),
            $this->translateStatus($fee->status),
            $fee->pay_id ?? '-',
            $fee->pay_month ?? '-',
            $fee->pay_year ?? '-'
        ];
    }

    protected function translatePaymentMethod($method)
    {
        $translations = [
            'Cash' => 'Dinheiro',
            'Bank' => 'Transferência Bancária',
            'Online' => 'Online',
            'Mobile' => 'Mobile Money'
        ];

        return $translations[$method] ?? $method ?? '-';
    }

    protected function translateStatus($status)
    {
        $translations = [
            'paid' => 'Pago',
            'unpaid' => 'Pendente',
            'partial' => 'Parcial',
            'overdue' => 'Atrasado'
        ];

        return $translations[$status] ?? $status ?? '-';
    }

    public function styles(Worksheet $sheet)
    {
        $lastRow = $sheet->getHighestRow();
        $lastColumn = $sheet->getHighestColumn();

        // Estilo do cabeçalho
        $sheet->getStyle('A1:' . $lastColumn . '1')->applyFromArray([
            'font' => [
                'bold' => true,
                'color' => ['rgb' => 'FFFFFF'],
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '2C3E50'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
            ],
        ]);

        // Bordas para toda a tabela
        $sheet->getStyle('A1:' . $lastColumn . $lastRow)->applyFromArray([
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => ['rgb' => 'CCCCCC'],
                ],
            ],
        ]);

        // Altura da linha do cabeçalho
        $sheet->getRowDimension(1)->setRowHeight(30);

        // Zebra striping (linhas alternadas)
        for ($i = 2; $i <= $lastRow; $i++) {
            if ($i % 2 == 0) {
                $sheet->getStyle('A' . $i . ':' . $lastColumn . $i)->applyFromArray([
                    'fill' => [
                        'fillType' => Fill::FILL_SOLID,
                        'startColor' => ['rgb' => 'F5F5F5'],
                    ],
                ]);
            }
        }

        // Alinhamento das colunas de valores
        $sheet->getStyle('G:J')->applyFromArray([
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_RIGHT,
            ],
        ]);

        // Alinhamento do centro para Status e Método
        $sheet->getStyle('K:L')->applyFromArray([
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ]);

        return [];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();
                $lastRow = $sheet->getHighestRow() + 2;

                // Adicionar linha de totais
                $sheet->setCellValue('A' . $lastRow, 'TOTAIS:');
                $sheet->setCellValue('G' . $lastRow, 'MZN ' . number_format($this->totalAmount, 2, ',', '.'));
                $sheet->setCellValue('H' . $lastRow, 'MZN ' . number_format($this->totalDiscount, 2, ',', '.'));
                $sheet->setCellValue('I' . $lastRow, 'MZN ' . number_format($this->totalFine, 2, ',', '.'));
                $sheet->setCellValue('J' . $lastRow, 'MZN ' . number_format($this->totalNet, 2, ',', '.'));

                // Estilo da linha de totais
                $sheet->getStyle('A' . $lastRow . ':O' . $lastRow)->applyFromArray([
                    'font' => [
                        'bold' => true,
                        'size' => 12,
                    ],
                    'fill' => [
                        'fillType' => Fill::FILL_SOLID,
                        'startColor' => ['rgb' => 'E8F4F8'],
                    ],
                    'borders' => [
                        'top' => [
                            'borderStyle' => Border::BORDER_DOUBLE,
                            'color' => ['rgb' => '000000'],
                        ],
                    ],
                ]);

                // Adicionar informações do relatório
                $infoRow = $lastRow + 2;
                $sheet->setCellValue('A' . $infoRow, 'Relatório gerado em: ' . Carbon::now()->format('d/m/Y H:i:s'));
                $sheet->getStyle('A' . $infoRow)->applyFromArray([
                    'font' => [
                        'italic' => true,
                        'size' => 10,
                        'color' => ['rgb' => '666666'],
                    ],
                ]);

                // Congelar painel (cabeçalho)
                $sheet->freezePane('A2');

                // Configurar impressão
                $sheet->getPageSetup()->setOrientation(\PhpOffice\PhpSpreadsheet\Worksheet\PageSetup::ORIENTATION_LANDSCAPE);
                $sheet->getPageSetup()->setPaperSize(\PhpOffice\PhpSpreadsheet\Worksheet\PageSetup::PAPERSIZE_A4);
                $sheet->getPageSetup()->setFitToWidth(1);
                $sheet->getPageSetup()->setFitToHeight(0);

                // Configurar margens
                $sheet->getPageMargins()->setTop(0.75);
                $sheet->getPageMargins()->setRight(0.25);
                $sheet->getPageMargins()->setLeft(0.25);
                $sheet->getPageMargins()->setBottom(0.75);

                // Adicionar cabeçalho e rodapé para impressão
                $sheet->getHeaderFooter()->setOddHeader('&C&B&16RELATÓRIO DE COLETA DE TAXAS');
                $sheet->getHeaderFooter()->setOddFooter('&L&B' . config('app.name') . '&RPágina &P de &N');
            },
        ];
    }
}
