<?php

namespace App\Exports;

use App\Models\PaymentReference;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use Carbon\Carbon;

class PaymentReferencesExport implements FromQuery, WithHeadings, WithMapping, WithStyles, ShouldAutoSize
{
    protected $filters;

    public function __construct($filters = [])
    {
        $this->filters = $filters;
    }

    /**
     * Query para buscar os dados
     */
    public function query()
    {
        $query = PaymentReference::query()->with('student.classroom');

        // Aplicar filtros
        if (!empty($this->filters['status'])) {
            if ($this->filters['status'] === 'expired') {
                $query->where('status', 'pending')
                      ->where('expires_at', '<', now());
            } else {
                $query->where('status', $this->filters['status']);
            }
        }

        if (!empty($this->filters['academic_year'])) {
            $query->where('academic_year', $this->filters['academic_year']);
        }

        if (!empty($this->filters['classe'])) {
            $query->where('classe', $this->filters['classe']);
        }

        if (!empty($this->filters['turma'])) {
            $query->where('turma', $this->filters['turma']);
        }

        if (!empty($this->filters['search'])) {
            $query->where(function($q) {
                $q->where('reference_number', 'like', '%' . $this->filters['search'] . '%')
                  ->orWhere('entity_code', 'like', '%' . $this->filters['search'] . '%')
                  ->orWhereHas('student', function($sq) {
                      $sq->where('name', 'like', '%' . $this->filters['search'] . '%')
                        ->orWhere('student_id', 'like', '%' . $this->filters['search'] . '%');
                  });
            });
        }

        return $query->orderBy('created_at', 'desc');
    }

    /**
     * Cabeçalhos das colunas
     */
    public function headings(): array
    {
        return [
            'ID',
            'Código do Estudante',
            'Nome do Estudante',
            'Classe',
            'Turma',
            'Ano Acadêmico',
            'Entidade',
            'Referência',
            'Valor (MT)',
            'Mês',
            'Ano',
            'Válido De',
            'Válido Até',
            'Status',
            'Data de Pagamento',
            'Gerada em',
        ];
    }

    /**
     * Mapear cada linha
     */
    public function map($reference): array
    {
        $isExpired = $reference->status === 'pending' && now()->isAfter($reference->expires_at);
        $status = $isExpired ? 'Expirada' : ucfirst($reference->status);

        if ($status === 'Pending') $status = 'Pendente';
        if ($status === 'Paid') $status = 'Paga';

        return [
            $reference->id,
            $reference->student->student_id ?? 'N/A',
            $reference->student->name ?? 'N/A',
            $reference->classe ?? '-',
            $reference->turma ?? '-',
            $reference->academic_year ?? '-',
            $reference->entity_code,
            $reference->reference_number,
            number_format($reference->amount, 2, '.', ''),
            ucfirst($reference->fee_month),
            $reference->fee_year,
            $reference->valid_from ? Carbon::parse($reference->valid_from)->format('d/m/Y') : '-',
            $reference->expires_at ? Carbon::parse($reference->expires_at)->format('d/m/Y') : '-',
            $status,
            $reference->paid_at ? Carbon::parse($reference->paid_at)->format('d/m/Y H:i') : '-',
            $reference->created_at ? Carbon::parse($reference->created_at)->format('d/m/Y H:i') : '-',
        ];
    }

    /**
     * Estilos para o Excel
     */
    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true, 'size' => 12]],
        ];
    }
}
