<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\Classroom;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use PDF;

class AttendancePdfController extends Controller
{
    /**
     * PDF para estudante (user guard)
     */
    public function studentPdf(Request $request)
    {
        $user = Auth::user();
        if (!$user) {
            abort(403);
        }

        $month = $request->query('month', now()->month);
        $year = $request->query('year', now()->year);
        $view = $request->query('view', 'month');

        $classroom = Classroom::find($user->classroom_id);

        $query = Attendance::where('student_id', $user->id);

        if ($view === 'month') {
            $query->whereMonth('date', $month)->whereYear('date', $year);
            $period = Carbon::createFromDate($year, $month, 1)->translatedFormat('F Y');
        } elseif ($view === 'year') {
            $query->whereYear('date', $year);
            $period = "Ano {$year}";
        } else {
            $query->whereMonth('date', $month)->whereYear('date', $year);
            $period = Carbon::createFromDate($year, $month, 1)->translatedFormat('F Y');
        }

        $records = $query->orderBy('date')->orderBy('created_at')->get();

        $total = $records->count();
        $present = $records->where('status', 'present')->count();
        $absent = $records->where('status', 'absent')->count();
        $late = $records->where('status', 'late')->count();
        $justified = $records->where('status', 'justified')->count();
        $rate = $total > 0 ? round(($present + $late + $justified) / $total * 100, 1) : 0;

        $summary = compact('total', 'present', 'absent', 'late', 'justified', 'rate');

        $pdf = PDF::loadView('pdf.attendance-report', [
            'student' => $user,
            'classroom' => $classroom,
            'records' => $records,
            'summary' => $summary,
            'period' => $period,
            'isAdmin' => false,
        ]);

        $pdf->setPaper('A4', 'portrait');

        return $pdf->download("Presencas_{$user->student_id}_{$period}.pdf");
    }

    /**
     * PDF para admin (admin guard)
     */
    public function adminPdf(Request $request)
    {
        $admin = Auth::guard('admin')->user();
        if (!$admin) {
            abort(403);
        }

        $classroomId = $request->query('classroom_id');
        $dateFrom = $request->query('from', date('Y-m-01'));
        $dateTo = $request->query('to', date('Y-m-d'));
        $year = $request->query('year', date('Y'));

        $classroom = Classroom::findOrFail($classroomId);

        $records = Attendance::where('classroom_id', $classroomId)
            ->where('year', $year)
            ->where('date', '>=', $dateFrom)
            ->where('date', '<=', $dateTo)
            ->orderBy('date')
            ->get();

        $students = User::where('classroom_id', $classroomId)
            ->where('year_edu', $year)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $studentBreakdown = $students->map(function ($student) use ($records) {
            $sr = $records->where('student_id', $student->id);
            $total = $sr->count();
            $present = $sr->where('status', 'present')->count();
            $absent = $sr->where('status', 'absent')->count();
            $late = $sr->where('status', 'late')->count();
            $justified = $sr->where('status', 'justified')->count();

            return [
                'name' => $student->name,
                'student_id' => $student->student_id,
                'total' => $total,
                'present' => $present,
                'absent' => $absent,
                'late' => $late,
                'justified' => $justified,
                'rate' => $total > 0 ? round(($present + $late + $justified) / $total * 100, 1) : 0,
            ];
        });

        $total = $records->count();
        $summary = [
            'total' => $total,
            'present' => $records->where('status', 'present')->count(),
            'absent' => $records->where('status', 'absent')->count(),
            'late' => $records->where('status', 'late')->count(),
            'justified' => $records->where('status', 'justified')->count(),
            'rate' => $total > 0 ? round(($records->where('status', 'present')->count() + $records->where('status', 'late')->count() + $records->where('status', 'justified')->count()) / $total * 100, 1) : 0,
        ];

        $period = Carbon::parse($dateFrom)->format('d/m/Y') . ' - ' . Carbon::parse($dateTo)->format('d/m/Y');

        $pdf = PDF::loadView('pdf.attendance-report', [
            'classroom' => $classroom,
            'records' => $records,
            'summary' => $summary,
            'studentBreakdown' => $studentBreakdown,
            'period' => $period,
            'isAdmin' => true,
        ]);

        $pdf->setPaper('A4', 'portrait');

        return $pdf->download("Presencas_{$classroom->class}a_{$classroom->name}_{$period}.pdf");
    }
}
