<?php

namespace App\Http\Controllers\Admin\Students;

use App\Http\Controllers\Controller;
use App\Models\DocumentScan;
use App\Services\GoogleVisionOCRService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CameraScanController extends Controller
{
    /**
     * Processar upload de foto da câmera
     */
    public function uploadCameraPhoto(Request $request)
    {
        try {
            $request->validate([
                'photo' => 'required|file|image|max:10240', // máx 10MB
            ]);

            $photo = $request->file('photo');
            $isBackSide = $request->boolean('is_back_side');
            $documentId = $request->input('document_id');

            // Se é verso do BI
            if ($isBackSide && $documentId) {
                return $this->processBackSide($photo, $documentId);
            }

            // Processar frente do BI
            return $this->processFrontSide($photo);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Processar foto da frente do BI
     */
    protected function processFrontSide($photo)
    {
        $ocrService = new GoogleVisionOCRService();

        // Salvar arquivo
        $filename = time() . '_camera_capture.jpg';
        $path = $photo->storeAs('document-scans', $filename, 'public');

        // Criar registro
        $scan = DocumentScan::create([
            'image_path' => $path,
            'image_filename' => $filename,
            'image_size' => $photo->getSize(),
            'status' => 'processing',
            'uploaded_by' => auth()->id(),
        ]);

        // Processar com OCR
        $fullPath = storage_path('app/public/' . $path);
        $result = $ocrService->processDocument($fullPath);

        if ($result['success']) {
            // Extrair dados do BI
            $extractedData = $ocrService->extractBIData($result['text']);

            // Atualizar registro
            $scan->update([
                'status' => 'processed',
                'raw_ocr_text' => $result['text'],
                'ocr_data' => $result['annotations'] ?? null,
                'confidence_score' => $result['confidence'],
                'document_type' => $extractedData['document_type'],
                'document_id' => $extractedData['document_id'],
                'full_name' => $extractedData['full_name'],
                'dob' => $extractedData['dob'],
                'father_name' => $extractedData['father_name'],
                'mother_name' => $extractedData['mother_name'],
                'birth_place' => $extractedData['birth_place'],
                'birth_district' => $extractedData['birth_district'],
                'birth_province' => $extractedData['birth_province'],
                'sex' => $extractedData['sex'],
                'requires_review' => $result['confidence'] < 90,
            ]);

            return response()->json([
                'success' => true,
                'document_id' => $scan->id,
                'message' => 'Documento processado com sucesso!',
            ]);
        } else {
            $scan->markAsFailed($result['error']);

            return response()->json([
                'success' => false,
                'error' => 'Erro no OCR: ' . $result['error'],
            ], 400);
        }
    }

    /**
     * Processar foto do verso do BI
     */
    protected function processBackSide($photo, $documentId)
    {
        $scan = DocumentScan::find($documentId);

        if (!$scan) {
            return response()->json([
                'success' => false,
                'error' => 'Documento não encontrado',
            ], 404);
        }

        $ocrService = new GoogleVisionOCRService();

        // Salvar arquivo do verso
        $filename = time() . '_back_' . $scan->image_filename;
        $path = $photo->storeAs('document-scans', $filename, 'public');

        // Processar com OCR
        $fullPath = storage_path('app/public/' . $path);
        $result = $ocrService->processDocument($fullPath);

        if ($result['success']) {
            // Extrair dados do verso (pai, mãe, etc)
            $extractedData = $ocrService->extractBIData($result['text']);

            // Atualizar registro com dados do verso + dados extraídos
            $scan->update([
                'document_side' => 'ambos',
                'back_image_path' => $path,
                'back_image_filename' => $filename,
                'back_image_size' => $photo->getSize(),
                'back_raw_ocr_text' => $result['text'],
                'back_ocr_data' => $result['annotations'] ?? null,
                // Atualizar pai e mãe extraídos do verso
                'father_name' => $extractedData['father_name'] ?? $scan->father_name,
                'mother_name' => $extractedData['mother_name'] ?? $scan->mother_name,
            ]);

            return response()->json([
                'success' => true,
                'document_id' => $scan->id,
                'message' => 'Verso do BI processado com sucesso!',
            ]);
        } else {
            return response()->json([
                'success' => false,
                'error' => 'Erro no OCR do verso: ' . $result['error'],
            ], 400);
        }
    }
}
