<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\CertificateService;
use Illuminate\Http\JsonResponse;

class CertificateVerificationController extends Controller
{
    public function __construct(private CertificateService $certificateService)
    {
    }

    /**
     * Verify a certificate by its verification code
     *
     * @param string $code
     * @return JsonResponse
     */
    public function verify(string $code): JsonResponse
    {
        if (empty(trim($code))) {
            return response()->json([
                'valid' => false,
                'message' => 'Codigo de verificacao nao fornecido',
            ], 400);
        }

        $result = $this->certificateService->verifyCertificate(trim($code));

        if (!$result) {
            return response()->json([
                'valid' => false,
                'message' => 'Certificado nao encontrado',
            ], 404);
        }

        if (!$result['valid']) {
            $message = match($result['status']) {
                'revoked' => 'Certificado revogado',
                default => 'Certificado invalido - falha na verificacao de integridade',
            };

            return response()->json([
                'valid' => false,
                'message' => $message,
                'status' => $result['status'],
                'integrity_verified' => $result['integrity_verified'],
            ], 200);
        }

        $cert = $result['certificate'];

        return response()->json([
            'valid' => true,
            'message' => 'Certificado valido e autentico',
            'certificate' => [
                'certificate_number' => $cert->certificate_number,
                'student_name' => $cert->student_name,
                'student_id' => $cert->student_id_number,
                'student_dob' => $cert->student_dob ? $cert->student_dob->format('d/m/Y') : null,
                'class_level' => $cert->class_level,
                'turma' => $cert->turma_name,
                'academic_year' => $cert->academic_year,
                'final_average' => (float) $cert->final_average,
                'subject_grades' => $cert->subject_grades,
                'issued_at' => $cert->issued_at->format('d/m/Y'),
                'status' => $cert->status,
            ],
            'integrity_verified' => $result['integrity_verified'],
        ], 200);
    }
}
