<?php

namespace App\Http\Livewire\Admin;

use App\Charts\UserChart;
use App\Charts\UserPieChart;
use App\Models\Admin;
use App\Models\Classroom;
use App\Models\TeacherSubject;
use App\Models\Subject;
use App\Models\User;
use App\Models\Fee_assign;
use App\Models\PaymentReference;
use Carbon\Carbon;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class AdminHome extends Component
{
    protected $lazy = false;
    use FindGuard;

    public $account;
    public $orders;
    private $seo_pie_chart;
    private $chart3;
    public $user;
    public bool $readyToLoad = false;

    // Filtros de período para estatísticas de pagamento
    public $period = '7days'; // 7days, 30days, 6months, 1year, custom
    public $customDateFrom;
    public $customDateTo;

    protected $queryString = ['period', 'customDateFrom', 'customDateTo'];

    public function updated($propertyName)
    {
        // Quando period, customDateFrom ou customDateTo mudam, garantir que os dados sejam carregados
        if (in_array($propertyName, ['period', 'customDateFrom', 'customDateTo'])) {
            $this->readyToLoad = true;
        }
    }

    public function loadData()
    {
        $this->readyToLoad = true;
    }

    public function charts()
    {
        // Cores profissionais modernas
        $modernColors = ['#4f46e5', '#10b981', '#f59e0b', '#ef4444', '#8b5cf6', '#06b6d4'];

        // Chart 3 - Gráfico de Barras com estilo moderno
        $this->chart3 = new UserPieChart();
        $this->chart3->labels(['Classes', 'Disciplinas', 'Estudantes']);
        $this->chart3->dataset('Visão Geral', 'bar', [
            count(Classroom::query()
                ->where('name', 'NOT LIKE', '%Não Matriculados%')
                ->where('name', 'NOT LIKE', '%Finalistas%')
                ->where('name', 'NOT LIKE', '%Recem%')
                ->get()),
            count(Subject::query()->get()),
            count(User::query()
                ->whereHas('classroom', function($query) {
                    $query->where('name', 'NOT LIKE', '%Não Matriculados%')
                          ->where('name', 'NOT LIKE', '%Finalistas%')
                          ->where('name', 'NOT LIKE', '%Recem%');
                })
                ->get())
        ])
        ->backgroundColor(collect(['rgba(79, 70, 229, 0.8)', 'rgba(16, 185, 129, 0.8)', 'rgba(245, 158, 11, 0.8)']))
        ->color(collect(['#4f46e5', '#10b981', '#f59e0b']))
        ->options([
            'responsive' => true,
            'maintainAspectRatio' => false,
            'plugins' => [
                'legend' => [
                    'display' => false
                ],
                'tooltip' => [
                    'backgroundColor' => 'rgba(0, 0, 0, 0.8)',
                    'titleColor' => '#fff',
                    'bodyColor' => '#fff',
                    'borderColor' => '#4f46e5',
                    'borderWidth' => 1,
                    'padding' => 12,
                    'displayColors' => true,
                    'cornerRadius' => 8
                ]
            ],
            'scales' => [
                'y' => [
                    'beginAtZero' => true,
                    'grid' => [
                        'color' => 'rgba(0, 0, 0, 0.05)',
                        'drawBorder' => false
                    ],
                    'ticks' => [
                        'color' => '#6b7280',
                        'font' => [
                            'size' => 12,
                            'weight' => '500'
                        ]
                    ]
                ],
                'x' => [
                    'grid' => [
                        'display' => false
                    ],
                    'ticks' => [
                        'color' => '#374151',
                        'font' => [
                            'size' => 13,
                            'weight' => '600'
                        ]
                    ]
                ]
            ],
            'animation' => [
                'duration' => 1500,
                'easing' => 'easeInOutQuart'
            ]
        ]);

        // SEO Pie Chart - Gráfico de Pizza moderno
        $this->seo_pie_chart = new UserPieChart();
        $this->seo_pie_chart->labels(['Classes', 'Disciplinas', 'Estudantes']);
        $this->seo_pie_chart->dataset('Distribuição', 'doughnut', [
            count(Classroom::query()
                ->where('name', 'NOT LIKE', '%Não Matriculados%')
                ->where('name', 'NOT LIKE', '%Finalistas%')
                ->where('name', 'NOT LIKE', '%Recem%')
                ->get()),
            count(Subject::query()->get()),
            count(User::query()
                ->whereHas('classroom', function($query) {
                    $query->where('name', 'NOT LIKE', '%Não Matriculados%')
                          ->where('name', 'NOT LIKE', '%Finalistas%')
                          ->where('name', 'NOT LIKE', '%Recem%');
                })
                ->get())
        ])
        ->backgroundColor(collect([
            'rgba(79, 70, 229, 0.9)',
            'rgba(16, 185, 129, 0.9)',
            'rgba(245, 158, 11, 0.9)'
        ]))
        ->color(collect(['#4f46e5', '#10b981', '#f59e0b']))
        ->options([
            'responsive' => true,
            'maintainAspectRatio' => false,
            'cutout' => '65%',
            'plugins' => [
                'legend' => [
                    'display' => true,
                    'position' => 'bottom',
                    'labels' => [
                        'color' => '#374151',
                        'font' => [
                            'size' => 13,
                            'weight' => '600'
                        ],
                        'padding' => 15,
                        'usePointStyle' => true,
                        'pointStyle' => 'circle'
                    ]
                ],
                'tooltip' => [
                    'backgroundColor' => 'rgba(0, 0, 0, 0.85)',
                    'titleColor' => '#fff',
                    'bodyColor' => '#fff',
                    'borderColor' => '#4f46e5',
                    'borderWidth' => 2,
                    'padding' => 14,
                    'displayColors' => true,
                    'cornerRadius' => 10,
                    'callbacks' => [
                        'label' => 'function(context) {
                            let label = context.label || "";
                            let value = context.parsed || 0;
                            let total = context.dataset.data.reduce((a, b) => a + b, 0);
                            let percentage = ((value / total) * 100).toFixed(1);
                            return label + ": " + value + " (" + percentage + "%)";
                        }'
                    ]
                ]
            ],
            'animation' => [
                'animateRotate' => true,
                'animateScale' => true,
                'duration' => 1800,
                'easing' => 'easeInOutCubic'
            ]
        ]);

        return [
            $this->seo_pie_chart,
            $this->chart3,
        ];
    }

    private function getDateRange()
    {
        switch ($this->period) {
            case '7days':
                return [now()->subDays(6)->startOfDay(), now()];
            case '30days':
                return [now()->subDays(29)->startOfDay(), now()];
            case '6months':
                return [now()->subMonths(6)->startOfDay(), now()];
            case '1year':
                return [now()->subYear()->startOfDay(), now()];
            case 'custom':
                if ($this->customDateFrom && $this->customDateTo) {
                    return [
                        Carbon::parse($this->customDateFrom)->startOfDay(),
                        Carbon::parse($this->customDateTo)->endOfDay()
                    ];
                }
                return [now()->subDays(6)->startOfDay(), now()];
            default:
                return [now()->subDays(6)->startOfDay(), now()];
        }
    }

    private function getPaymentStats()
    {
        $today = now()->startOfDay();
        $weekStart = now()->startOfWeek();
        $monthStart = now()->startOfMonth();

        $dateRange = $this->getDateRange();
        $startDate = $dateRange[0];
        $endDate = $dateRange[1];

        // Calcular o intervalo em dias para o gráfico
        $daysDiff = $startDate->diffInDays($endDate);
        $groupBy = 'date';
        $dateFormat = 'DATE(created_at)';

        // Se for mais de 60 dias, agrupar por semana
        if ($daysDiff > 60) {
            $groupBy = 'week';
            $dateFormat = 'DATE_FORMAT(created_at, "%Y-%u")';
        }
        // Se for mais de 180 dias, agrupar por mês
        if ($daysDiff > 180) {
            $groupBy = 'month';
            $dateFormat = 'DATE_FORMAT(created_at, "%Y-%m")';
        }

        // Calcular totais do período selecionado
        // Fórmula: VALOR BASE + MULTA - DESCONTO = TOTAL
        $payments = Fee_assign::whereBetween('created_at', [$startDate, $endDate])->get();

        $periodBase = $payments->sum('amount') ?? 0;
        $periodFines = $payments->sum('fine') ?? 0;
        $periodDiscounts = $payments->sum('discount') ?? 0;
        $periodTotal = $periodBase + $periodFines - $periodDiscounts;
        $periodCount = $payments->count();

        return [
            'pending_references' => PaymentReference::where('status', 'pending')->count() ?? 0,

            // Dados do período selecionado (para cards e gráfico)
            'period_label' => $this->getPeriodLabel(),
            'period_base' => $periodBase,
            'period_fines' => $periodFines,
            'period_discounts' => $periodDiscounts,
            'period_total' => $periodTotal,
            'period_count' => $periodCount,
            'period_payments' => Fee_assign::select(
                    DB::raw("$dateFormat as period"),
                    DB::raw('SUM(amount) as base'),
                    DB::raw('SUM(COALESCE(fine, 0)) as fines'),
                    DB::raw('SUM(COALESCE(discount, 0)) as discounts'),
                    DB::raw('SUM(amount + COALESCE(fine, 0) - COALESCE(discount, 0)) as total'),
                    DB::raw('COUNT(*) as count')
                )
                ->whereBetween('created_at', [$startDate, $endDate])
                ->groupBy('period')
                ->orderBy('period')
                ->get()
                ->mapWithKeys(function ($item) {
                    return [$item->period => [
                        'period' => $item->period,
                        'base' => (float) $item->base,
                        'fines' => (float) $item->fines,
                        'discounts' => (float) $item->discounts,
                        'total' => (float) $item->total,
                        'count' => (int) $item->count
                    ]];
                })
                ->all(),
            'group_by' => $groupBy,
            'date_range' => [
                'start' => $startDate->format('d/m/Y'),
                'end' => $endDate->format('d/m/Y')
            ]
        ];
    }

    private function getPeriodLabel()
    {
        switch ($this->period) {
            case '7days':
                return 'Últimos 7 Dias';
            case '30days':
                return 'Últimos 30 Dias';
            case '6months':
                return 'Últimos 6 Meses';
            case '1year':
                return 'Último Ano';
            case 'custom':
                if ($this->customDateFrom && $this->customDateTo) {
                    return Carbon::parse($this->customDateFrom)->format('d/m/Y') . ' - ' .
                           Carbon::parse($this->customDateTo)->format('d/m/Y');
                }
                return 'Período Personalizado';
            default:
                return 'Últimos 7 Dias';
        }
    }

    public function getStudentDistribution()
    {
        // Buscar classes, excluindo Não Matriculados, Finalistas e Recém-formados
        // Usar whereRaw para ignorar case sensitivity
        $classrooms = Classroom::whereRaw('LOWER(name) NOT LIKE ?', ['%não matriculado%'])
            ->whereRaw('LOWER(name) NOT LIKE ?', ['%finalista%'])
            ->whereRaw('LOWER(name) NOT LIKE ?', ['%recem%'])
            ->whereRaw('LOWER(name) NOT LIKE ?', ['%recém%'])
            ->orderByRaw('CAST(class AS UNSIGNED)')
            ->orderBy('name')
            ->get();

        // Agrupar por classe (número)
        $distributionByClass = [];

        foreach ($classrooms as $classroom) {
            $maleCount = User::where('classroom_id', $classroom->id)
                ->where('sex', 'Masculino')
                ->where('year_edu', date('Y'))
                ->count();

            $femaleCount = User::where('classroom_id', $classroom->id)
                ->where('sex', 'Feminino')
                ->where('year_edu', date('Y'))
                ->count();

            $totalCount = $maleCount + $femaleCount;

            if ($totalCount > 0) {
                $classNumber = $classroom->class;

                // Se já existe essa classe no array, somar os valores
                if (isset($distributionByClass[$classNumber])) {
                    $distributionByClass[$classNumber]['male'] += $maleCount;
                    $distributionByClass[$classNumber]['female'] += $femaleCount;
                    $distributionByClass[$classNumber]['total'] += $totalCount;
                } else {
                    // Criar nova entrada para essa classe
                    $distributionByClass[$classNumber] = [
                        'name' => $classNumber . 'ª Classe',
                        'male' => $maleCount,
                        'female' => $femaleCount,
                        'total' => $totalCount
                    ];
                }
            }
        }

        // Ordenar por número da classe e retornar como array indexado
        ksort($distributionByClass);
        return array_values($distributionByClass);
    }

    public function render()
    {
       // dd(Auth::user());
         $this->user = $this->findGuardType()->user();
         $teacherId = $this->user->id;

         $paymentStats = $this->readyToLoad ? $this->getPaymentStats() : [
             'pending_references' => 0,
             'period_label' => 'Últimos 7 Dias',
             'period_total' => 0,
             'period_count' => 0,
             'period_payments' => collect([]),
             'group_by' => 'date',
             'date_range' => ['start' => '', 'end' => '']
         ];

        return view('livewire.admin.admin-home', [
            'classes' => $this->readyToLoad ? count(Classroom::query()
                ->where('name', 'NOT LIKE', '%Não Matriculados%')
                ->where('name', 'NOT LIKE', '%Finalistas%')
                ->where('name', 'NOT LIKE', '%Recem%')
                ->get()) : 0,
            'teacher_class' => $this->readyToLoad ? TeacherSubject::query()
        ->where('teacher_id', $teacherId)
        ->distinct('classroom_id')
        ->count() : 0,
            'students' => $this->readyToLoad ? count(User::query()
                ->whereHas('classroom', function($query) {
                    $query->where('name', 'NOT LIKE', '%Não Matriculados%')
                          ->where('name', 'NOT LIKE', '%Finalistas%')
                          ->where('name', 'NOT LIKE', '%Recem%');
                })
                ->get()) : 0,
            'subjects' => $this->readyToLoad ? count(Subject::query()->get()) : 0,
            'admins' => $this->readyToLoad ? count(Admin::query()->get()) : 0,
            'seo_pie_chart' => $this->charts()[0],
            'chart3' => $this->charts()[1],
            'payment_stats' => $paymentStats,
            'student_distribution' => $this->getStudentDistribution(), // Sempre carregar
        ]);
    }
}
