<?php

namespace App\Http\Livewire\Admin\Attendance;

use App\Models\Attendance;
use App\Models\Classroom;
use App\Models\TeacherSubject;
use App\Models\User;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;

class AttendanceReport extends Component
{
    use FindGuard;

    public $year;
    public $classroom_id;
    public $date_from;
    public $date_to;
    public bool $readyToLoad = false;

    public function mount()
    {
        $this->year = date('Y');
        $this->date_from = date('Y-m-01');
        $this->date_to = date('Y-m-d');
    }

    public function loadData()
    {
        $this->readyToLoad = true;
    }

    public function render()
    {
        $user = $this->findGuardType()->user();
        $isTeacher = ($user->role == 2);

        $classrooms = collect();
        if ($isTeacher) {
            $classroomIds = TeacherSubject::where('teacher_id', $user->id)
                ->whereNotNull('classroom_id')
                ->distinct()
                ->pluck('classroom_id');
            $classrooms = Classroom::whereIn('id', $classroomIds)->orderBy('class')->orderBy('name')->get();
        } else {
            $classrooms = Classroom::orderBy('class')->orderBy('name')->get();
        }

        $summary = [
            'total_days' => 0,
            'total_present' => 0,
            'total_absent' => 0,
            'total_late' => 0,
            'total_justified' => 0,
            'rate' => 0,
        ];
        $studentBreakdown = collect();

        if ($this->readyToLoad && $this->classroom_id) {
            $query = Attendance::where('year', $this->year)
                ->where('classroom_id', $this->classroom_id);

            if ($this->date_from) {
                $query->where('date', '>=', $this->date_from);
            }
            if ($this->date_to) {
                $query->where('date', '<=', $this->date_to);
            }

            $records = $query->get();

            $summary['total_days'] = $records->pluck('date')->unique()->count();
            $summary['total_present'] = $records->where('status', 'present')->count();
            $summary['total_absent'] = $records->where('status', 'absent')->count();
            $summary['total_late'] = $records->where('status', 'late')->count();
            $summary['total_justified'] = $records->where('status', 'justified')->count();

            $totalRecords = $records->count();
            if ($totalRecords > 0) {
                $summary['rate'] = round(
                    (($summary['total_present'] + $summary['total_late']) / $totalRecords) * 100,
                    1
                );
            }

            // Per-student breakdown
            $students = User::where('classroom_id', $this->classroom_id)
                ->where('year_edu', $this->year)
                ->where('is_active', true)
                ->orderBy('name')
                ->get();

            $studentBreakdown = $students->map(function ($student) use ($records) {
                $studentRecords = $records->where('student_id', $student->id);
                $total = $studentRecords->count();
                $present = $studentRecords->where('status', 'present')->count();
                $absent = $studentRecords->where('status', 'absent')->count();
                $late = $studentRecords->where('status', 'late')->count();
                $justified = $studentRecords->where('status', 'justified')->count();

                return [
                    'name' => $student->name,
                    'student_id' => $student->student_id,
                    'total' => $total,
                    'present' => $present,
                    'absent' => $absent,
                    'late' => $late,
                    'justified' => $justified,
                    'rate' => $total > 0 ? round((($present + $late) / $total) * 100, 1) : 0,
                ];
            });
        }

        return view('livewire.admin.attendance.attendance-report', [
            'classrooms' => $classrooms,
            'summary' => $summary,
            'studentBreakdown' => $studentBreakdown,
        ]);
    }
}
