<?php

namespace App\Http\Livewire\Admin\Certificates;

use App\Models\Certificate;
use App\Models\CertificateBatch;
use App\Models\Classroom;
use App\Models\DocumentClassConfiguration;
use App\Models\User;
use App\Services\CertificateService;
use Livewire\Component;
use Livewire\WithPagination;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class ManageCertificates extends Component
{
    use WithPagination, LivewireAlert;

    protected $paginationTheme = 'bootstrap';

    // Tab control
    public $activeTab = 'generate'; // generate, list, batches, config

    // Filters for generation
    public $selectedClass = null;
    public $selectedTurma = null;
    public $academicYear;
    public $availableTurmas = [];
    public $availableClasses = []; // Loaded from DocumentClassConfiguration

    // Current document configuration
    public $currentDocConfig = null;

    // Eligible students data
    public $eligibleStudents = [];
    public $ineligibleStudents = [];
    public $showEligibilityResults = false;

    // Individual generation
    public $selectedStudentId = null;

    // Filters for listing
    public $searchTerm = '';
    public $filterStatus = '';
    public $filterYear = '';
    public $filterClassLevel = '';
    public $filterTurma = '';

    // Revocation
    public $revokeCertificateId = null;
    public $revocationReason = '';

    // Document Configuration Editing
    public $editConfigId = null;
    public $editClassLevel = '';
    public $editDocumentType = 'declaration';
    public $editDocumentTitle = '';
    public $editDescription = '';
    public $showConfigModal = false;

    // Statistics
    public $stats = [];

    // Listeners
    protected $listeners = [
        'confirmGenerate',
        'confirmBatchGenerate',
        'confirmRevoke',
        'confirmDeleteConfig',
        'refreshComponent' => '$refresh',
    ];

    public function mount()
    {
        $this->academicYear = (int) date('Y');
        $this->filterYear = $this->academicYear;
        $this->loadAvailableClasses();
        // Default to first configured class
        $this->selectedClass = !empty($this->availableClasses) ? $this->availableClasses[0] : null;
        $this->loadDocConfig();
        $this->loadTurmas();
        $this->loadStats();
    }

    public function loadAvailableClasses()
    {
        $this->availableClasses = DocumentClassConfiguration::getConfiguredClasses($this->academicYear);
    }

    public function loadDocConfig()
    {
        if ($this->selectedClass) {
            $this->currentDocConfig = DocumentClassConfiguration::getForClass($this->selectedClass, $this->academicYear);
        } else {
            $this->currentDocConfig = null;
        }
    }

    public function loadTurmas()
    {
        $query = Classroom::where('is_active', 1)
            ->where('name', 'NOT LIKE', '%Recem%')
            ->where('name', 'NOT LIKE', '%Recém%')
            ->where('name', 'NOT LIKE', '%Não Matriculado%')
            ->where('name', 'NOT LIKE', '%Finalista%');

        if ($this->selectedClass) {
            $query->where('class', $this->selectedClass);
        } else {
            $query->whereIn('class', $this->availableClasses);
        }

        $this->availableTurmas = $query->orderByRaw('CAST(class AS UNSIGNED)')->orderBy('name')->get();
    }

    public function updatedSelectedClass()
    {
        $this->selectedTurma = null;
        $this->showEligibilityResults = false;
        $this->eligibleStudents = [];
        $this->ineligibleStudents = [];
        $this->loadDocConfig();
        $this->loadTurmas();
    }

    public function loadStats()
    {
        $service = app(CertificateService::class);
        $this->stats = $service->getStatistics($this->academicYear);
    }

    public function updatedSelectedTurma($turmaId)
    {
        $this->showEligibilityResults = false;
        $this->eligibleStudents = [];
        $this->ineligibleStudents = [];
        $this->selectedStudentId = null;
    }

    public function updatedAcademicYear()
    {
        $this->showEligibilityResults = false;
        $this->eligibleStudents = [];
        $this->ineligibleStudents = [];
        $this->loadAvailableClasses();
        $this->loadDocConfig();
        $this->loadTurmas();
        $this->loadStats();
    }

    public function checkEligibility()
    {
        if (!$this->selectedTurma) {
            $this->alert('warning', 'Por favor, selecione uma turma.');
            return;
        }

        $service = app(CertificateService::class);
        $result = $service->getEligibleStudents($this->selectedTurma, $this->academicYear, $this->selectedClass);

        $this->eligibleStudents = $result['eligible'];
        $this->ineligibleStudents = $result['ineligible'];
        $this->showEligibilityResults = true;
    }

    public function generateIndividual()
    {
        if (!$this->selectedStudentId) {
            $this->alert('warning', 'Por favor, selecione um estudante.');
            return;
        }

        // Check permission
        $admin = auth('admin')->user();
        if (!$admin->hasPermission('certificates.generate')) {
            $this->alert('error', 'Voce nao tem permissao para gerar certificados.');
            return;
        }

        $this->confirm('Gerar certificado para este estudante?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, Gerar',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmGenerate',
        ]);
    }

    public function confirmGenerate()
    {
        try {
            $student = User::find($this->selectedStudentId);
            if (!$student) {
                $this->alert('error', 'Estudante nao encontrado.');
                return;
            }

            $service = app(CertificateService::class);
            $certificate = $service->createCertificate(
                $student,
                $this->academicYear,
                auth('admin')->id()
            );

            $this->alert('success', "Certificado gerado com sucesso! Numero: {$certificate->certificate_number}");

            $this->selectedStudentId = null;
            $this->checkEligibility(); // Refresh list
            $this->loadStats();

        } catch (\Exception $e) {
            Log::error('Erro ao gerar certificado', [
                'student_id' => $this->selectedStudentId,
                'error' => $e->getMessage(),
            ]);
            $this->alert('error', 'Erro ao gerar certificado: ' . $e->getMessage());
        }
    }

    public function generateBatch()
    {
        if (!$this->selectedTurma) {
            $this->alert('warning', 'Por favor, selecione uma turma.');
            return;
        }

        // Check permission
        $admin = auth('admin')->user();
        if (!$admin->hasPermission('certificates.batch')) {
            $this->alert('error', 'Voce nao tem permissao para gerar lotes de certificados.');
            return;
        }

        $eligibleCount = count(array_filter($this->eligibleStudents, fn($e) => !$e['has_certificate']));

        if ($eligibleCount === 0) {
            $this->alert('warning', 'Nenhum estudante elegivel sem certificado.');
            return;
        }

        $this->confirm("Gerar certificados para {$eligibleCount} estudantes?", [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, Gerar Todos',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmBatchGenerate',
        ]);
    }

    public function confirmBatchGenerate()
    {
        try {
            $service = app(CertificateService::class);

            $batch = $service->createBatch(
                $this->selectedTurma,
                $this->academicYear,
                auth('admin')->id(),
                $this->selectedClass
            );

            // Process batch
            $service->processBatch($batch);

            $this->alert('success', "Lote {$batch->batch_number} processado! {$batch->success_count} certificados gerados.");

            $this->checkEligibility(); // Refresh
            $this->loadStats();

        } catch (\Exception $e) {
            Log::error('Erro ao gerar lote', [
                'classroom_id' => $this->selectedTurma,
                'error' => $e->getMessage(),
            ]);
            $this->alert('error', 'Erro ao gerar lote: ' . $e->getMessage());
        }
    }

    public function downloadCertificate($certificateId)
    {
        $certificate = Certificate::find($certificateId);

        if (!$certificate || !$certificate->pdf_path) {
            $this->alert('error', 'Certificado nao encontrado.');
            return;
        }

        if (!Storage::disk('public')->exists($certificate->pdf_path)) {
            // Regenerate PDF
            try {
                $service = app(CertificateService::class);
                $pdfPath = $service->generatePDF($certificate);
                $certificate->update(['pdf_path' => $pdfPath]);
            } catch (\Exception $e) {
                $this->alert('error', 'Erro ao gerar PDF: ' . $e->getMessage());
                return;
            }
        }

        return Storage::disk('public')->download($certificate->pdf_path);
    }

    public function downloadCertificateWord($certificateId)
    {
        $certificate = Certificate::find($certificateId);

        if (!$certificate) {
            $this->alert('error', 'Certificado não encontrado.');
            return;
        }

        try {
            $service = app(CertificateService::class);
            $wordPath = $service->generateWordDocument($certificate);

            return Storage::disk('public')->download($wordPath);

        } catch (\Exception $e) {
            Log::error('Erro ao gerar Word', ['error' => $e->getMessage()]);
            $this->alert('error', 'Erro ao gerar documento Word: ' . $e->getMessage());
        }
    }

    public function downloadBatchZip($batchId)
    {
        $batch = CertificateBatch::find($batchId);

        if (!$batch) {
            $this->alert('error', 'Lote nao encontrado.');
            return;
        }

        try {
            $service = app(CertificateService::class);

            if (!$batch->zip_path || !Storage::disk('public')->exists($batch->zip_path)) {
                $zipPath = $service->generateBatchZip($batch);
            } else {
                $zipPath = $batch->zip_path;
            }

            return Storage::disk('public')->download($zipPath);

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao gerar ZIP: ' . $e->getMessage());
        }
    }

    public function downloadAllZip()
    {
        try {
            $service = app(CertificateService::class);
            // Use filterTurma from list tab or selectedTurma from generate tab
            $turmaId = $this->activeTab === 'list'
                ? ($this->filterTurma ? Classroom::where('name', $this->filterTurma)->value('id') : null)
                : $this->selectedTurma;
            $year = $this->activeTab === 'list' ? $this->filterYear : $this->academicYear;

            $zipPath = $service->generateDirectZip($turmaId, $year ?: $this->academicYear);

            return Storage::disk('public')->download($zipPath);

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao gerar ZIP: ' . $e->getMessage());
        }
    }

    public function regenerateCertificate($certificateId)
    {
        $certificate = Certificate::find($certificateId);

        if (!$certificate) {
            $this->alert('error', 'Documento não encontrado.');
            return;
        }

        try {
            $service = app(CertificateService::class);

            // Regenerate the Word document (force refresh photos)
            $service->generateWordDocument($certificate, null, true);

            $this->alert('success', 'Documento regenerado com sucesso!');

        } catch (\Exception $e) {
            Log::error('Erro ao regenerar documento', ['certificate_id' => $certificateId, 'error' => $e->getMessage()]);
            $this->alert('error', 'Erro ao regenerar: ' . $e->getMessage());
        }
    }

    public function regenerateBatch($batchId)
    {
        $batch = CertificateBatch::find($batchId);

        if (!$batch) {
            $this->alert('error', 'Lote não encontrado.');
            return;
        }

        try {
            $service = app(CertificateService::class);

            // Get all certificates for this batch's classroom and year
            $query = Certificate::where('academic_year', $batch->academic_year)
                ->where('status', 'issued');

            if ($batch->classroom_id) {
                $query->where('classroom_id', $batch->classroom_id);
            }

            $certificates = $query->get();
            $regeneratedCount = 0;

            foreach ($certificates as $cert) {
                try {
                    // Regenerate the Word document (force refresh photos)
                    $service->generateWordDocument($cert, null, true);
                    $regeneratedCount++;
                } catch (\Exception $e) {
                    Log::warning("Erro ao regenerar documento {$cert->id}: " . $e->getMessage());
                }
            }

            // Clear cached ZIP so it regenerates on next download
            $batch->update([
                'zip_path' => null,
                'zip_generated_at' => null,
            ]);

            $this->alert('success', "{$regeneratedCount} documentos regenerados com sucesso!");

        } catch (\Exception $e) {
            Log::error('Erro ao regenerar lote', ['batch_id' => $batchId, 'error' => $e->getMessage()]);
            $this->alert('error', 'Erro ao regenerar lote: ' . $e->getMessage());
        }
    }

    public function openRevokeModal($certificateId)
    {
        $this->revokeCertificateId = $certificateId;
        $this->revocationReason = '';
        $this->dispatchBrowserEvent('open-revoke-modal');
    }

    public function revokeCertificate()
    {
        if (!$this->revokeCertificateId) {
            return;
        }

        if (empty(trim($this->revocationReason))) {
            $this->alert('warning', 'Por favor, informe o motivo da revogacao.');
            return;
        }

        // Check permission
        $admin = auth('admin')->user();
        if (!$admin->hasPermission('certificates.revoke')) {
            $this->alert('error', 'Voce nao tem permissao para revogar certificados.');
            return;
        }

        try {
            $certificate = Certificate::find($this->revokeCertificateId);

            if (!$certificate) {
                $this->alert('error', 'Certificado nao encontrado.');
                return;
            }

            $service = app(CertificateService::class);
            $service->revokeCertificate($certificate, $this->revocationReason, auth('admin')->id());

            $this->alert('success', 'Certificado revogado com sucesso.');

            $this->revokeCertificateId = null;
            $this->revocationReason = '';
            $this->loadStats();
            $this->dispatchBrowserEvent('close-revoke-modal');

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao revogar certificado: ' . $e->getMessage());
        }
    }

    // ===== DOCUMENT CONFIGURATION MANAGEMENT =====

    public function openAddConfigModal()
    {
        $this->resetConfigForm();
        $this->showConfigModal = true;
        $this->dispatchBrowserEvent('open-config-modal');
    }

    public function openEditConfigModal($configId)
    {
        $config = DocumentClassConfiguration::find($configId);

        if (!$config) {
            $this->alert('error', 'Configuração não encontrada.');
            return;
        }

        // Cast values to correct types for form binding
        $this->editConfigId = $config->id;
        $this->editClassLevel = (string) $config->class_level;
        $this->editDocumentType = $config->document_type ?? 'declaration';
        $this->editDocumentTitle = $config->document_title ?? '';
        $this->editDescription = $config->description ?? '';
        $this->showConfigModal = true;

        $this->dispatchBrowserEvent('open-config-modal');
    }

    public function resetConfigForm()
    {
        $this->editConfigId = null;
        $this->editClassLevel = '';
        $this->editDocumentType = 'declaration';
        $this->editDocumentTitle = '';
        $this->editDescription = '';
    }

    public function closeConfigModal()
    {
        $this->showConfigModal = false;
        $this->resetConfigForm();
        $this->dispatchBrowserEvent('close-config-modal');
    }

    public function saveConfig()
    {
        $this->validate([
            'editClassLevel' => 'required|integer|min:1|max:12',
            'editDocumentType' => 'required|in:certificate,declaration',
            'editDocumentTitle' => 'required|string|max:255',
        ], [
            'editClassLevel.required' => 'A classe é obrigatória.',
            'editClassLevel.integer' => 'A classe deve ser um número.',
            'editClassLevel.min' => 'A classe deve ser entre 1 e 12.',
            'editClassLevel.max' => 'A classe deve ser entre 1 e 12.',
            'editDocumentType.required' => 'O tipo de documento é obrigatório.',
            'editDocumentTitle.required' => 'O título do documento é obrigatório.',
        ]);

        try {
            $admin = auth('admin')->user();

            if ($this->editConfigId) {
                // Update existing
                $config = DocumentClassConfiguration::find($this->editConfigId);

                if (!$config) {
                    $this->alert('error', 'Configuração não encontrada.');
                    return;
                }

                // Check if document type changed
                $oldDocumentType = $config->document_type;
                $newDocumentType = $this->editDocumentType;
                $documentTypeChanged = $oldDocumentType !== $newDocumentType;

                $config->update([
                    'class_level' => $this->editClassLevel,
                    'document_type' => $this->editDocumentType,
                    'document_title' => $this->editDocumentTitle,
                    'description' => $this->editDescription,
                    'updated_by' => $admin->id,
                ]);

                // If document type changed, update all existing certificates for this class
                if ($documentTypeChanged) {
                    $updatedCount = $this->updateExistingCertificatesType(
                        (int) $this->editClassLevel,
                        $oldDocumentType,
                        $newDocumentType
                    );

                    $newTypeLabel = $newDocumentType === 'certificate' ? 'Certificado' : 'Declaração';
                    $this->alert('success', "Configuração actualizada! {$updatedCount} documentos actualizados para {$newTypeLabel}.");
                } else {
                    $this->alert('success', 'Configuração actualizada com sucesso!');
                }
            } else {
                // Check if class already exists
                $exists = DocumentClassConfiguration::where('class_level', $this->editClassLevel)
                    ->whereNull('academic_year')
                    ->exists();

                if ($exists) {
                    $this->alert('error', "Já existe uma configuração para a {$this->editClassLevel}ª classe.");
                    return;
                }

                // Create new
                DocumentClassConfiguration::create([
                    'class_level' => $this->editClassLevel,
                    'document_type' => $this->editDocumentType,
                    'document_title' => $this->editDocumentTitle,
                    'description' => $this->editDescription,
                    'is_active' => true,
                    'created_by' => $admin->id,
                ]);

                $this->alert('success', 'Configuração adicionada com sucesso!');
            }

            $this->closeConfigModal();
            $this->loadAvailableClasses();
            $this->loadDocConfig();

        } catch (\Exception $e) {
            Log::error('Erro ao salvar configuração', ['error' => $e->getMessage()]);
            $this->alert('error', 'Erro ao salvar: ' . $e->getMessage());
        }
    }

    public function toggleConfigStatus($configId)
    {
        try {
            $config = DocumentClassConfiguration::find($configId);

            if (!$config) {
                $this->alert('error', 'Configuração não encontrada.');
                return;
            }

            $config->update([
                'is_active' => !$config->is_active,
                'updated_by' => auth('admin')->id(),
            ]);

            $status = $config->is_active ? 'activada' : 'desactivada';
            $this->alert('success', "Configuração {$status} com sucesso!");

            $this->loadAvailableClasses();

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao alterar status: ' . $e->getMessage());
        }
    }

    /**
     * Update existing certificates when document type changes in configuration
     * Changes certificate_number prefix from DECL to CERT or vice versa
     */
    protected function updateExistingCertificatesType(int $classLevel, string $oldType, string $newType): int
    {
        $oldPrefix = $oldType === 'certificate' ? 'CERT' : 'DECL';
        $newPrefix = $newType === 'certificate' ? 'CERT' : 'DECL';

        // Get all certificates for this class level with the old prefix
        $certificates = Certificate::where('class_level', $classLevel)
            ->where('certificate_number', 'LIKE', $oldPrefix . '-%')
            ->get();

        $updatedCount = 0;

        foreach ($certificates as $certificate) {
            // Replace the prefix in certificate_number
            $newCertificateNumber = preg_replace(
                '/^' . $oldPrefix . '-/',
                $newPrefix . '-',
                $certificate->certificate_number
            );

            $certificate->update([
                'certificate_number' => $newCertificateNumber,
            ]);

            $updatedCount++;
        }

        Log::info('Updated certificate types', [
            'class_level' => $classLevel,
            'old_type' => $oldType,
            'new_type' => $newType,
            'updated_count' => $updatedCount,
        ]);

        return $updatedCount;
    }

    public function deleteConfig($configId)
    {
        $this->confirm('Tem certeza que deseja eliminar esta configuração?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, Eliminar',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmDeleteConfig',
            'inputAttributes' => ['configId' => $configId],
        ]);

        $this->editConfigId = $configId;
    }

    public function confirmDeleteConfig()
    {
        try {
            $config = DocumentClassConfiguration::find($this->editConfigId);

            if (!$config) {
                $this->alert('error', 'Configuração não encontrada.');
                return;
            }

            $config->delete();

            $this->alert('success', 'Configuração eliminada com sucesso!');
            $this->editConfigId = null;
            $this->loadAvailableClasses();

        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao eliminar: ' . $e->getMessage());
        }
    }

    public function getCertificatesProperty()
    {
        $query = Certificate::with(['student', 'classroom', 'issuedBy'])
            ->orderBy('turma_name', 'asc')
            ->orderBy('student_name', 'asc');

        if ($this->searchTerm) {
            $query->where(function ($q) {
                $q->where('certificate_number', 'like', '%' . $this->searchTerm . '%')
                  ->orWhere('student_name', 'like', '%' . $this->searchTerm . '%')
                  ->orWhere('student_id_number', 'like', '%' . $this->searchTerm . '%');
            });
        }

        if ($this->filterYear) {
            $query->where('academic_year', $this->filterYear);
        }

        if ($this->filterStatus) {
            $query->where('status', $this->filterStatus);
        }

        if ($this->filterClassLevel) {
            $query->where('class_level', $this->filterClassLevel);
        }

        if ($this->filterTurma) {
            $query->where('turma_name', $this->filterTurma);
        }

        return $query->paginate(15);
    }

    public function getBatchesProperty()
    {
        return CertificateBatch::with(['classroom', 'createdBy'])
            ->where('academic_year', $this->academicYear)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
    }

    public function getDocumentConfigurationsProperty()
    {
        return DocumentClassConfiguration::orderBy('class_level')
            ->get();
    }

    /**
     * Get the document type label for display
     */
    public function getDocumentTypeLabel(): string
    {
        if ($this->currentDocConfig) {
            return $this->currentDocConfig->isCertificate() ? 'Certificado' : 'Declaração';
        }
        return 'Documento';
    }

    public function render()
    {
        return view('livewire.admin.certificates.manage-certificates', [
            'certificates' => $this->certificates,
            'batches' => $this->batches,
            'documentConfigurations' => $this->documentConfigurations,
            'documentTypeLabel' => $this->getDocumentTypeLabel(),
        ]);
    }
}
