<?php

namespace App\Http\Livewire\Admin\Noticeboard;


use App\Http\Controllers\SystemController;
use App\Models\Noticeboard;

use App\Models\User;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithFileUploads;
use Note\Note;

class EditNotice extends Component
{
    use FindGuard, LivewireAlert, WithFileUploads;

    public $title, $description, $photo, $category_id, $pdf, $user, $slug;
    public $validatedData;
    public $updateMode = false;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function mount($slug)
    {
        $model = Noticeboard::query()->where('slug', $slug)->first();

        if (!$model) {
            abort(404, 'Comunicado não encontrado');
        }

        $this->slug = $slug;
        $this->title = $model->title ?? '';
        $this->description = $model->description ?? '';
    }

    public function dehydrate()
    {
        // Clean UTF-8 encoding before sending to browser to prevent JSON encoding errors
        if (is_string($this->title)) {
            $this->title = $this->sanitizeUtf8($this->title);
        }
        if (is_string($this->description)) {
            $this->description = $this->sanitizeUtf8($this->description);
        }
    }

    public function hydrate()
    {
        // Clean UTF-8 encoding on every request to prevent JSON encoding errors
        if (is_string($this->title)) {
            $this->title = $this->sanitizeUtf8($this->title);
        }
        if (is_string($this->description)) {
            $this->description = $this->sanitizeUtf8($this->description);
        }
    }

    private function sanitizeUtf8($string)
{
    if (empty($string) || !is_string($string)) {
        return $string;
    }

    // Remove qualquer byte inválido
    $string = mb_convert_encoding($string, 'UTF-8', 'UTF-8');

    // Remove caracteres de controlo e bytes ilegais
    $string = preg_replace('/[^\PC\s]/u', '', $string);

    // Última verificação: força JSON válido
    try {
        json_encode($string, JSON_THROW_ON_ERROR | JSON_UNESCAPED_UNICODE);
    } catch (\JsonException $e) {
        // Como fallback, usa uma limpeza bruta de bytes 0x00–0x1F
        $string = preg_replace('/[\x00-\x1F\x7F-\xFF]/', '', $string);
    }

    return $string;
}


    protected array $rules = [
        'title' => ['required', 'string'],
        'description' => ['nullable', 'string'],
        'photo' => ['nullable'], // 2MB Max
    ];

    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function submit()
    {
        $this->validatedData = $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        $model = Noticeboard::query()->where('slug', $this->slug)->first();

        if (!$model) {
            $this->alert('error', 'Comunicado não encontrado');
            return;
        }

        // Sanitize UTF-8 and prepare update data
        $updateData = [
            'title' => $this->sanitizeUtf8($this->title),
            'description' => $this->sanitizeUtf8($this->description),
        ];

        // Handle photo upload
        if ($this->photo) {
            $details = SystemController::store_media($this->photo);
            $updateData['media_name'] = $details[0];
            $updateData['media_url'] = $details[1];
        }

        // Single update call with all data
        try {
            $model->update($updateData);
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao atualizar: ' . $e->getMessage());
            return;
        }

        Note::createSystemNotification(\App\Models\Admin::class, 'Detalhes', 'Atualizado com sucesso ' . $this->title);
        $this->emit('noteAdded');
        $this->alert('success', 'Atualizado com sucesso ' . $this->title);

        return redirect()->route('admin.notice');
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }
    public function render()
    {
        return view('livewire.admin.noticeboard.edit-notice', [
            //'subjects' => Category::query()->orderBy('name')->get(),
        ]);
    }

    // Override to ensure all data is UTF-8 safe before JSON encoding
    public function __get($property)
    {
        $value = parent::__get($property);

        if (is_string($value)) {
            return $this->sanitizeUtf8($value);
        }

        return $value;
    }
}
