<?php

namespace App\Http\Livewire\Admin\Notifications;

use App\Models\Admin;
use App\Models\Classroom;
use App\Models\Guardian;
use App\Models\User;
use App\Helpers\FirebaseHelper;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;
use Livewire\Component;
use Illuminate\Support\Facades\Log;
use Kreait\Firebase\Exception\Messaging\NotFound;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Note\Note;

class AddPushNotification extends Component
{
    use LivewireAlert;

    public $type = 'all';
    public $class_id;
    public $class_ids = [];
    public $student_id;
    public $guardian_id;
    public $message;
    public $student_type;
    public $title;

    protected function initializeFirebase()
    {
        return (new Factory)
            ->withServiceAccount(FirebaseHelper::getCredentials())
            ->createMessaging();
    }

    protected function sendNotificationToToken($user, $token, $messaging)
    {
        try {
            $notification = Notification::create($this->title, $this->message);

            $message = CloudMessage::withTarget('token', $token)
                ->withNotification($notification)
                ->withHighestPossiblePriority()
                ->withDefaultSounds();

            $messaging->send($message);

            Log::channel('log_notifications')->info("Notification sent successfully to: {$user->name} - Token: $token");
            return true;

        } catch (NotFound $e) {
            $user->deviceTokens()->where('token', $token)->delete();
            Log::channel('log_notifications')->warning("Token not found/deleted for {$user->name}: {$token}. Error: {$e->getMessage()}");
        } catch (\Kreait\Firebase\Exception\MessagingException $e) {
            Log::channel('log_notifications')->error("Messaging error for {$user->name}: {$e->getMessage()}");
        } catch (\Exception $e) {
            Log::channel('log_notifications')->error("General error for {$user->name}: {$e->getMessage()}");
        }

        return false;
    }

    protected function sendToRecipients($recipients, $messaging, &$successCount, &$failureCount)
    {
        foreach ($recipients as $recipient) {
            $tokens = $recipient->deviceTokens()->pluck('token');
            foreach ($tokens as $token) {
                if ($this->sendNotificationToToken($recipient, $token, $messaging)) {
                    $successCount++;
                } else {
                    $failureCount++;
                }
            }
        }
    }

    public function submit()
    {
        $rules = [
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => 'required|in:all,all_students,all_guardians,class,class_guardians,student,guardian',
        ];

        if ($this->type == 'class' || $this->type == 'class_guardians') {
            $rules['class_ids'] = 'required|array|min:1';
        }
        if ($this->type == 'student') {
            $rules['class_id'] = 'required';
            $rules['student_id'] = 'required';
        }
        if ($this->type == 'guardian') {
            $rules['guardian_id'] = 'required';
        }

        $this->validate($rules);

        $messaging = $this->initializeFirebase();
        $successCount = 0;
        $failureCount = 0;

        try {
            switch ($this->type) {
                case 'all':
                    // Enviar para todos os estudantes E encarregados
                    $users = User::whereHas('deviceTokens')->get();
                    $guardians = Guardian::whereHas('deviceTokens')->get();
                    Log::channel('log_notifications')->info("Sending to ALL. Students: {$users->count()}, Guardians: {$guardians->count()}");
                    $this->sendToRecipients($users, $messaging, $successCount, $failureCount);
                    $this->sendToRecipients($guardians, $messaging, $successCount, $failureCount);
                    break;

                case 'all_students':
                    $users = User::whereHas('deviceTokens')->get();
                    Log::channel('log_notifications')->info("Sending to all students. Total: {$users->count()}");
                    $this->sendToRecipients($users, $messaging, $successCount, $failureCount);
                    break;

                case 'all_guardians':
                    $guardians = Guardian::whereHas('deviceTokens')->get();
                    Log::channel('log_notifications')->info("Sending to all guardians. Total: {$guardians->count()}");
                    $this->sendToRecipients($guardians, $messaging, $successCount, $failureCount);
                    break;

                case 'class':
                    // Estudantes das turmas selecionadas
                    $users = User::whereIn('classroom_id', $this->class_ids)
                        ->whereHas('deviceTokens')->get();
                    Log::channel('log_notifications')->info("Sending to students in " . count($this->class_ids) . " class(es). Total: {$users->count()}");
                    $this->sendToRecipients($users, $messaging, $successCount, $failureCount);
                    break;

                case 'class_guardians':
                    // Encarregados dos estudantes das turmas selecionadas
                    $studentIds = User::whereIn('classroom_id', $this->class_ids)->pluck('id');
                    $guardians = Guardian::whereHas('students', function ($q) use ($studentIds) {
                        $q->whereIn('users.id', $studentIds);
                    })->whereHas('deviceTokens')->get();
                    Log::channel('log_notifications')->info("Sending to guardians of " . count($this->class_ids) . " class(es). Total guardians: {$guardians->count()}");
                    $this->sendToRecipients($guardians, $messaging, $successCount, $failureCount);
                    break;

                case 'student':
                    $user = User::find($this->student_id);
                    if ($user) {
                        Log::channel('log_notifications')->info("Sending to student: {$user->name}");
                        $this->sendToRecipients(collect([$user]), $messaging, $successCount, $failureCount);
                    }
                    break;

                case 'guardian':
                    $guardian = Guardian::find($this->guardian_id);
                    if ($guardian) {
                        Log::channel('log_notifications')->info("Sending to guardian: {$guardian->name}");
                        $this->sendToRecipients(collect([$guardian]), $messaging, $successCount, $failureCount);
                    }
                    break;
            }

            $message = "Notificações enviadas. Sucesso: {$successCount}, Falhas: {$failureCount}";
            $this->alert('success', $message);
            Log::channel('log_notifications')->info($message);

        } catch (\Exception $e) {
            $errorMsg = "Erro ao enviar notificações: " . $e->getMessage();
            $this->alert('error', $errorMsg);
            Log::channel('log_notifications')->error($errorMsg);
        }

        $this->reset(['type', 'class_id', 'class_ids', 'student_id', 'guardian_id', 'message', 'title']);
    }

    public function render()
    {
        return view('livewire.admin.notifications.add-notifications', [
            'classrooms' => Classroom::query()->orderBy('class')->get(),
            'students' => User::when($this->class_id, function($query) {
                return $query->where('classroom_id', $this->class_id);
            })->orderBy('name')->get(),
            'guardians' => Guardian::where('is_active', true)->orderBy('name')->get(),
            'class_ids' => $this->class_ids,
        ]);
    }
}
