<?php

namespace App\Http\Livewire\Admin\POS;

use App\Models\POS\Sale;
use App\Models\POS\Category;
use App\Models\User;
use App\Models\Admin;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Livewire\WithPagination;

class SalesReport extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    // Filters
    public $date_from = '';
    public $date_to = '';
    public $payment_method = '';
    public $status = '';
    public $student_id = '';
    public $cashier_id = '';
    public $min_amount = '';
    public $max_amount = '';
    public $search = '';

    // Statistics
    public $total_sales = 0;
    public $total_revenue = 0;
    public $total_tax = 0;
    public $total_discount = 0;
    public $average_sale = 0;

    protected $queryString = [
        'date_from',
        'date_to',
        'payment_method',
        'status',
        'search'
    ];

    public function mount()
    {
        // Default to current month
        $this->date_from = now()->startOfMonth()->format('Y-m-d');
        $this->date_to = now()->format('Y-m-d');
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function applyFilters()
    {
        $this->resetPage();
        $this->calculateStatistics();
    }

    public function resetFilters()
    {
        $this->date_from = now()->startOfMonth()->format('Y-m-d');
        $this->date_to = now()->format('Y-m-d');
        $this->payment_method = '';
        $this->status = '';
        $this->student_id = '';
        $this->cashier_id = '';
        $this->min_amount = '';
        $this->max_amount = '';
        $this->search = '';
        $this->resetPage();
    }

    public function setDateRange($range)
    {
        switch ($range) {
            case 'today':
                $this->date_from = now()->format('Y-m-d');
                $this->date_to = now()->format('Y-m-d');
                break;
            case 'yesterday':
                $this->date_from = now()->subDay()->format('Y-m-d');
                $this->date_to = now()->subDay()->format('Y-m-d');
                break;
            case 'week':
                $this->date_from = now()->startOfWeek()->format('Y-m-d');
                $this->date_to = now()->format('Y-m-d');
                break;
            case 'month':
                $this->date_from = now()->startOfMonth()->format('Y-m-d');
                $this->date_to = now()->format('Y-m-d');
                break;
            case 'last_month':
                $this->date_from = now()->subMonth()->startOfMonth()->format('Y-m-d');
                $this->date_to = now()->subMonth()->endOfMonth()->format('Y-m-d');
                break;
            case 'year':
                $this->date_from = now()->startOfYear()->format('Y-m-d');
                $this->date_to = now()->format('Y-m-d');
                break;
        }
        $this->applyFilters();
    }

    public function calculateStatistics()
    {
        $query = $this->getBaseQuery();

        $stats = $query->select(
            DB::raw('COUNT(*) as total_sales'),
            DB::raw('SUM(total) as total_revenue'),
            DB::raw('SUM(tax) as total_tax'),
            DB::raw('SUM(discount) as total_discount'),
            DB::raw('AVG(total) as average_sale')
        )->first();

        $this->total_sales = $stats->total_sales ?? 0;
        $this->total_revenue = $stats->total_revenue ?? 0;
        $this->total_tax = $stats->total_tax ?? 0;
        $this->total_discount = $stats->total_discount ?? 0;
        $this->average_sale = $stats->average_sale ?? 0;
    }

    private function getBaseQuery()
    {
        $query = Sale::query();

        if ($this->date_from) {
            $query->whereDate('sale_date', '>=', $this->date_from);
        }

        if ($this->date_to) {
            $query->whereDate('sale_date', '<=', $this->date_to);
        }

        if ($this->payment_method) {
            $query->where('payment_method', $this->payment_method);
        }

        if ($this->status) {
            $query->where('status', $this->status);
        }

        if ($this->student_id) {
            $query->where('user_id', $this->student_id);
        }

        if ($this->cashier_id) {
            $query->where('cashier_id', $this->cashier_id);
        }

        if ($this->min_amount) {
            $query->where('total', '>=', $this->min_amount);
        }

        if ($this->max_amount) {
            $query->where('total', '<=', $this->max_amount);
        }

        if ($this->search) {
            $query->where(function($q) {
                $q->where('sale_number', 'like', '%' . $this->search . '%')
                  ->orWhere('notes', 'like', '%' . $this->search . '%');
            });
        }

        return $query;
    }

    public function exportExcel()
    {
        // TODO: Implement Excel export
        session()->flash('info', 'Exportação para Excel em desenvolvimento');
    }

    public function exportPDF()
    {
        // TODO: Implement PDF export
        session()->flash('info', 'Exportação para PDF em desenvolvimento');
    }

    public function render()
    {
        $this->calculateStatistics();

        $sales = $this->getBaseQuery()
            ->with(['user', 'cashier', 'items.product'])
            ->orderBy('sale_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $cashiers = Admin::whereIn('id', Sale::distinct()->pluck('cashier_id'))
            ->get(['id', 'name']);

        return view('livewire.admin.pos.sales-report', [
            'sales' => $sales,
            'cashiers' => $cashiers,
        ]);
    }
}
