<?php

namespace App\Http\Livewire\Admin\Permissions;

use App\Models\Admin;
use App\Models\AdminPermission;
use App\Models\PermissionDefinition;
use App\Models\PermissionGroup;
use App\Models\RolePermission;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;

class ManageRolePermissions extends Component
{
    use LivewireAlert;

    // Tab management
    public $activeTab = 'roles'; // 'roles' or 'permissions'

    // Role permissions tab
    public $roles = [];
    public $selectedRole = null;
    public $rolePermissions = [];

    // Permission definitions tab
    public $permissionGroups = [];
    public $editingPermission = null;
    public $editingGroup = null;

    // Forms
    public $newPermission = [
        'key' => '',
        'label' => '',
        'description' => '',
        'group_id' => '',
    ];
    public $newGroup = [
        'key' => '',
        'label' => '',
    ];

    public $availablePermissions = [];

    protected $listeners = [
        'confirmedSave',
        'confirmedSeedDefaults',
        'confirmedDeletePermission',
        'confirmedDeleteGroup',
    ];

    protected $rules = [
        'newPermission.key' => 'required|string|max:100',
        'newPermission.label' => 'required|string|max:150',
        'newPermission.group_id' => 'required|exists:permission_groups,id',
        'newGroup.key' => 'required|string|max:50',
        'newGroup.label' => 'required|string|max:100',
    ];

    public function mount()
    {
        $this->roles = RolePermission::getRoleNames();
        // Remove Super Admin from list (role 0 always has all permissions)
        unset($this->roles[0]);

        $this->loadPermissions();
    }

    public function loadPermissions()
    {
        $this->availablePermissions = AdminPermission::availablePermissions();
        $this->permissionGroups = PermissionGroup::with('permissions')
            ->orderBy('sort_order')
            ->orderBy('label')
            ->get();
    }

    // ==================== ROLE PERMISSIONS TAB ====================

    public function selectRole($roleId)
    {
        $this->selectedRole = (int) $roleId;
        $this->loadRolePermissions();
    }

    public function loadRolePermissions()
    {
        if ($this->selectedRole === null) {
            $this->rolePermissions = [];
            return;
        }

        $this->rolePermissions = RolePermission::getPermissionsForRole($this->selectedRole);
    }

    public function togglePermission($permissionKey)
    {
        if ($this->selectedRole === null) {
            $this->alert('warning', 'Selecione um role primeiro');
            return;
        }

        $index = array_search($permissionKey, $this->rolePermissions);

        if ($index !== false) {
            unset($this->rolePermissions[$index]);
            $this->rolePermissions = array_values($this->rolePermissions);
        } else {
            $this->rolePermissions[] = $permissionKey;
        }
    }

    public function hasPermission($permissionKey)
    {
        return in_array($permissionKey, $this->rolePermissions);
    }

    public function grantAll()
    {
        if ($this->selectedRole === null) {
            $this->alert('warning', 'Selecione um role primeiro');
            return;
        }

        $this->rolePermissions = array_keys(Admin::getAllPermissionsStatic());
        $this->alert('info', 'Todas as permissões marcadas. Clique em Salvar para confirmar.');
    }

    public function revokeAll()
    {
        if ($this->selectedRole === null) {
            $this->alert('warning', 'Selecione um role primeiro');
            return;
        }

        $this->rolePermissions = [];
        $this->alert('info', 'Todas as permissões desmarcadas. Clique em Salvar para confirmar.');
    }

    public function save()
    {
        if ($this->selectedRole === null) {
            $this->alert('warning', 'Selecione um role primeiro');
            return;
        }

        $this->alert('question', 'Tem certeza que deseja salvar as permissões para este role?', [
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, Salvar',
            'showCancelButton' => true,
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedSave',
            'timer' => null,
        ]);
    }

    public function confirmedSave()
    {
        RolePermission::setPermissions($this->selectedRole, $this->rolePermissions);

        $roleName = RolePermission::getRoleName($this->selectedRole);
        $this->alert('success', "Permissões do role '{$roleName}' salvas com sucesso!");
    }

    public function seedDefaults()
    {
        $this->alert('question', 'Isto irá restaurar as permissões padrão para TODOS os roles. Continuar?', [
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, Restaurar',
            'showCancelButton' => true,
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedSeedDefaults',
            'timer' => null,
        ]);
    }

    public function confirmedSeedDefaults()
    {
        // Seed permission definitions first
        PermissionDefinition::seedDefaults();

        // Then seed role permissions
        RolePermission::seedDefaultPermissions();

        $this->loadPermissions();
        $this->loadRolePermissions();

        $this->alert('success', 'Permissões padrão restauradas!');
    }

    // ==================== PERMISSION DEFINITIONS TAB ====================

    public function createGroup()
    {
        $this->validate([
            'newGroup.key' => 'required|string|max:50|unique:permission_groups,key',
            'newGroup.label' => 'required|string|max:100',
        ]);

        $maxOrder = PermissionGroup::max('sort_order') ?? 0;

        PermissionGroup::create([
            'key' => $this->newGroup['key'],
            'label' => $this->newGroup['label'],
            'sort_order' => $maxOrder + 1,
        ]);

        $this->newGroup = ['key' => '', 'label' => ''];
        $this->loadPermissions();

        $this->alert('success', 'Grupo criado com sucesso!');
    }

    public function editGroup($groupId)
    {
        $this->editingGroup = PermissionGroup::find($groupId);
    }

    public function updateGroup()
    {
        if (!$this->editingGroup) return;

        $this->editingGroup->save();
        $this->editingGroup = null;
        $this->loadPermissions();

        PermissionGroup::clearCache();
        $this->alert('success', 'Grupo atualizado!');
    }

    public function deleteGroup($groupId)
    {
        $group = PermissionGroup::find($groupId);
        if ($group && $group->permissions()->count() > 0) {
            $this->alert('error', 'Não é possível excluir um grupo com permissões. Remova as permissões primeiro.');
            return;
        }

        $this->alert('question', 'Tem certeza que deseja excluir este grupo?', [
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, Excluir',
            'showCancelButton' => true,
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedDeleteGroup',
            'data' => ['groupId' => $groupId],
            'timer' => null,
        ]);
    }

    public function confirmedDeleteGroup($data)
    {
        $groupId = $data['groupId'] ?? null;
        if ($groupId) {
            PermissionGroup::destroy($groupId);
            $this->loadPermissions();
            PermissionGroup::clearCache();
            $this->alert('success', 'Grupo excluído!');
        }
    }

    public function createPermission()
    {
        $this->validate([
            'newPermission.key' => 'required|string|max:100|unique:permission_definitions,key',
            'newPermission.label' => 'required|string|max:150',
            'newPermission.group_id' => 'required|exists:permission_groups,id',
        ]);

        $maxOrder = PermissionDefinition::where('group_id', $this->newPermission['group_id'])->max('sort_order') ?? 0;

        PermissionDefinition::create([
            'key' => $this->newPermission['key'],
            'label' => $this->newPermission['label'],
            'description' => $this->newPermission['description'] ?? null,
            'group_id' => $this->newPermission['group_id'],
            'sort_order' => $maxOrder + 1,
        ]);

        $this->newPermission = ['key' => '', 'label' => '', 'description' => '', 'group_id' => ''];
        $this->loadPermissions();

        PermissionDefinition::clearCache();
        $this->alert('success', 'Permissão criada com sucesso!');
    }

    public function editPermission($permissionId)
    {
        $this->editingPermission = PermissionDefinition::find($permissionId);
    }

    public function updatePermission()
    {
        if (!$this->editingPermission) return;

        $this->editingPermission->save();
        $this->editingPermission = null;
        $this->loadPermissions();

        PermissionDefinition::clearCache();
        $this->alert('success', 'Permissão atualizada!');
    }

    public function deletePermission($permissionId)
    {
        $this->alert('question', 'Tem certeza que deseja excluir esta permissão?', [
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, Excluir',
            'showCancelButton' => true,
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedDeletePermission',
            'data' => ['permissionId' => $permissionId],
            'timer' => null,
        ]);
    }

    public function confirmedDeletePermission($data)
    {
        $permissionId = $data['permissionId'] ?? null;
        if ($permissionId) {
            $perm = PermissionDefinition::find($permissionId);
            if ($perm) {
                // Also remove from role_permissions
                RolePermission::where('permission_key', $perm->key)->delete();
                $perm->delete();
            }

            $this->loadPermissions();
            PermissionDefinition::clearCache();
            RolePermission::clearCache();

            $this->alert('success', 'Permissão excluída!');
        }
    }

    public function togglePermissionActive($permissionId)
    {
        $permission = PermissionDefinition::find($permissionId);
        if ($permission) {
            $permission->is_active = !$permission->is_active;
            $permission->save();
            $this->loadPermissions();
            PermissionDefinition::clearCache();
        }
    }

    public function toggleGroupActive($groupId)
    {
        $group = PermissionGroup::find($groupId);
        if ($group) {
            $group->is_active = !$group->is_active;
            $group->save();
            $this->loadPermissions();
            PermissionGroup::clearCache();
        }
    }

    public function render()
    {
        return view('livewire.admin.permissions.manage-role-permissions');
    }
}
