<?php

namespace App\Http\Livewire\Admin\Secretary;

use App\Models\Admin;
use App\Models\Classroom;
use App\Models\Guardian;
use App\Models\PreEnrollment;
use App\Models\User;
use App\Services\SmsService;
use Exception;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithFileUploads;
use Note\Note;

class EnrollStudent extends Component
{
    use FindGuard, LivewireAlert, WithFileUploads;

    public $preEnrollment;

    // Student fields
    public $name, $sex, $dob, $email, $phone_number, $student_id;
    public $class, $classroom_id, $year_edu;
    public $father_name, $mother_name;
    public $birth_place, $birth_district, $birth_province;
    public $document_type, $document_id;
    public $entry_type;
    public $observation;

    // ID toggle + photo
    public $auto_generate_id = true;
    public $avatar;

    protected $listeners = ['confirmed', 'cancelled'];

    protected array $rules = [
        'name' => ['required', 'string', 'max:255'],
        'sex' => ['required', 'string', 'max:255'],
        'email' => ['required', 'string', 'max:255', 'unique:users'],
        'student_id' => ['required', 'string', 'max:255', 'unique:users'],
        'classroom_id' => ['required', 'string', 'max:255'],
        'class' => ['required', 'string', 'max:255'],
        'dob' => ['required', 'string', 'max:255'],
        'phone_number' => ['required', 'numeric'],
        'year_edu' => ['required', 'integer', 'min:2020', 'max:2035'],
        'father_name' => ['nullable', 'string', 'max:255'],
        'mother_name' => ['nullable', 'string', 'max:255'],
        'birth_place' => ['nullable', 'string', 'max:255'],
        'birth_district' => ['nullable', 'string', 'max:255'],
        'birth_province' => ['nullable', 'string', 'max:255'],
        'document_id' => ['nullable', 'string', 'max:255'],
        'document_type' => ['nullable', 'string', 'in:BI,Cedula,Passaporte'],
    ];

    public function mount($preEnrollmentId)
    {
        $this->preEnrollment = PreEnrollment::findOrFail($preEnrollmentId);

        if (!$this->preEnrollment->isConfirmada()) {
            session()->flash('error', 'Esta pré-inscrição não está confirmada.');
            return redirect()->route('admin.secretary.pre-enrollments');
        }

        // Pre-fill from pre-enrollment
        $this->name = $this->preEnrollment->candidate_name;
        $this->sex = $this->preEnrollment->candidate_sex;
        $this->dob = $this->preEnrollment->candidate_dob ? $this->preEnrollment->candidate_dob->format('Y-m-d') : '';
        $this->email = $this->preEnrollment->candidate_email;
        $this->phone_number = $this->preEnrollment->candidate_phone;
        $this->class = $this->preEnrollment->desired_class;
        $this->classroom_id = $this->preEnrollment->desired_classroom_id;
        $this->year_edu = $this->preEnrollment->academic_year ?? date('Y');
        $this->father_name = $this->preEnrollment->father_name;
        $this->mother_name = $this->preEnrollment->mother_name;
        $this->birth_place = $this->preEnrollment->birth_place;
        $this->birth_district = $this->preEnrollment->birth_district;
        $this->birth_province = $this->preEnrollment->birth_province;
        $this->document_type = $this->preEnrollment->document_type;
        $this->document_id = $this->preEnrollment->document_id;
        $this->entry_type = $this->preEnrollment->entry_type;
        $this->observation = $this->preEnrollment->observation;

        // Auto-generate student_id
        $this->student_id = $this->generateStudentId();
    }

    public function updatedAutoGenerateId($value)
    {
        if ($value) {
            $this->student_id = $this->generateStudentId();
        } else {
            $this->student_id = '';
        }
    }

    /**
     * Gera codigo do estudante no formato CPM{letra}{numero}
     * Letra: A=2022, B=2023, C=2024, D=2025, E=2026, F=2027...
     * Numero: comeca em 1001 cada ano
     */
    private function generateStudentId(): string
    {
        $year = (int) date('Y');
        $letter = chr(ord('A') + ($year - 2022));
        $prefix = 'CPM' . $letter;

        // Buscar ultimo codigo deste ano
        $last = User::where('student_id', 'like', "{$prefix}%")
            ->orderBy('student_id', 'desc')
            ->first();

        $seq = 1001;
        if ($last) {
            $num = intval(substr($last->student_id, 4));
            $seq = $num + 1;
        }

        $studentId = $prefix . $seq;

        // Garantir unicidade
        while (User::where('student_id', $studentId)->exists()) {
            $seq++;
            $studentId = $prefix . $seq;
        }

        return $studentId;
    }

    public function submit()
    {
        $this->validate();
        $this->confirm('Confirmar matrícula do estudante?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, matricular!',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        $password = rand(100000, 500000);

        $student = User::create([
            'name' => $this->name,
            'sex' => $this->sex,
            'email' => $this->email,
            'student_id' => $this->student_id,
            'classroom_id' => $this->classroom_id,
            'dob' => $this->dob,
            'password' => bcrypt($password),
            'is_paid' => true,
            'phone_number' => $this->phone_number,
            'year_edu' => $this->year_edu,
            'father_name' => $this->father_name,
            'mother_name' => $this->mother_name,
            'birth_place' => $this->birth_place,
            'birth_district' => $this->birth_district,
            'birth_province' => $this->birth_province,
            'document_id' => $this->document_id,
            'document_type' => $this->document_type,
            'avatar' => $this->getAvatarPath(),
        ]);

        // Update pre-enrollment
        $this->preEnrollment->update([
            'status' => 'matriculada',
            'enrolled_by' => auth()->guard('admin')->id(),
            'resulting_user_id' => $student->id,
        ]);

        // Auto-create guardian from pre-enrollment data
        $this->createGuardianFromPreEnrollment($student);

        // Send SMS with credentials
        try {
            $message = "Caro pai, o seu educando " . $this->name .
                ", foi matriculado no COPMOZ. ID: {$this->student_id}, Senha: {$password}" .
                "\n\nAceda: https://siga.copmoz.ac.mz";
            (new SmsService())->send($this->phone_number, $message);
        } catch (Exception $e) {
            $this->alert('warning', 'Matrícula realizada, mas erro ao enviar SMS.');
        }

        Note::createSystemNotification(Admin::class, 'Nova Matrícula',
            'Estudante ' . $this->name . ' matriculado via pré-inscrição ' . $this->preEnrollment->code);

        $this->alert('success', 'Matrícula realizada com sucesso!');

        return redirect()->route('admin.secretary.pre-enrollments');
    }

    private function createGuardianFromPreEnrollment(User $student): void
    {
        $guardianPhone = $this->preEnrollment->guardian_phone;
        $guardianName = $this->preEnrollment->guardian_name;

        if (empty($guardianPhone) || empty($guardianName)) {
            return;
        }

        try {
            // Normalize phone: remove non-digits, add 258 prefix
            $phone = preg_replace('/\D/', '', $guardianPhone);
            if (strlen($phone) === 9 && !str_starts_with($phone, '258')) {
                $phone = '258' . $phone;
            }

            // Find existing or create new guardian
            $guardian = Guardian::where('phone', $phone)->first();
            $isNew = false;

            $guardianPassword = null;

            if (!$guardian) {
                $guardianPassword = rand(100000, 500000);
                $guardian = Guardian::create([
                    'name'         => $guardianName,
                    'phone'        => $phone,
                    'email'        => $this->preEnrollment->guardian_email ?: null,
                    'relationship' => $this->preEnrollment->guardian_relationship ?: 'encarregado',
                    'password'     => Hash::make($guardianPassword),
                    'is_active'    => true,
                ]);
                $isNew = true;
            }

            // Link guardian to student (if not already linked)
            if (!$guardian->students()->where('users.id', $student->id)->exists()) {
                $guardian->students()->attach($student->id, [
                    'id'           => (string) \Illuminate\Support\Str::uuid(),
                    'relationship' => $this->preEnrollment->guardian_relationship ?: 'encarregado',
                    'is_primary'   => true,
                ]);
            }

            // Send SMS to guardian with portal credentials (only for new guardians)
            if ($isNew && $guardianPassword) {
                try {
                    $smsService = new SmsService();
                    $guardianMsg = "Caro(a) {$guardianName}, a sua conta no Portal do Encarregado COPMOZ foi criada." .
                        "\nTelefone: {$phone}" .
                        "\nSenha: {$guardianPassword}" .
                        "\n\nAceda: https://siga.copmoz.ac.mz/encarregado";
                    $smsService->send($phone, $guardianMsg);
                } catch (Exception $e) {
                    Log::warning('Failed to send guardian credentials SMS', [
                        'guardian_id' => $guardian->id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            Log::info('Guardian linked to student via enrollment', [
                'guardian_id' => $guardian->id,
                'student_id' => $student->id,
                'is_new_guardian' => $isNew,
            ]);
        } catch (Exception $e) {
            Log::error('Failed to create/link guardian during enrollment', [
                'pre_enrollment_id' => $this->preEnrollment->id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    private function getAvatarPath(): string
    {
        if ($this->avatar) {
            return $this->avatar->store('avatars', 'public');
        }

        // Usar foto da pré-inscrição se existir
        if ($this->preEnrollment->candidate_photo &&
            Storage::disk('public')->exists($this->preEnrollment->candidate_photo)) {
            $avatarPath = 'avatars/' . basename($this->preEnrollment->candidate_photo);
            Storage::disk('public')->copy($this->preEnrollment->candidate_photo, $avatarPath);
            return $avatarPath;
        }

        return 'avatars/default-avatar.png';
    }

    public function cancelled()
    {
        $this->alert('info', 'Matrícula cancelada.');
    }

    public function render()
    {
        $this->findGuardType()->user();

        $classrooms = Classroom::query()
            ->when($this->class, fn($q) => $q->where('class', $this->class))
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $availableClasses = Classroom::where('is_active', true)
            ->select('class')
            ->distinct()
            ->orderByRaw('CAST(class AS UNSIGNED)')
            ->pluck('class');

        return view('livewire.admin.secretary.enroll-student', [
            'classrooms' => $classrooms,
            'availableClasses' => $availableClasses,
        ]);
    }
}
