<?php

//namespace App\Http\Livewire\User\Fee;
//
//use App\Models\Fee_assign;
//use App\Models\PaymentReference;
//use App\Models\User;
//use Livewire\Component;
//use App\Models\FeeStructure;
//use Carbon\Carbon;
//use LaravelMultipleGuards\Traits\FindGuard;
//use Barryvdh\DomPDF\Facade\Pdf;
//use Illuminate\Support\Facades\Log;
//use Illuminate\Support\Facades\Storage;
//use Illuminate\Support\Str;
//
//class FeeCollectionUser extends Component
//{
//    use FindGuard;
//    
//    public $studentId;
//    public $feeGroup;
//    public $date;
//    public $discountGroup;
//    public $discount = 0;
//    public $paymentMode = 'Cash';
//    public $note;
//    public $students = [];
//    public $selectedStudent;
//    public $selectedFee;
//    public $selectedFeeAssignId;
//    public $month;
//    public $year;
//    public $amount;
//    public $fine;
//
//    protected $listeners = [
//        'openModal' => 'preparePayment',
//        'collectPayment' => 'processPayment',
//        'generateReference' => 'generateReference'
//    ];
//
//    public function mount()
//    {
//        $user = $this->findGuardType()->user();
//        if ($user) {
//            $this->studentId = $user->id;
//            $this->selectedStudent = User::findOrFail($this->studentId);
//            $this->date = now()->format('d/m/Y');
//        } else {
//            abort(404, 'User not found');
//        }
//    }
//
//    /**
//     * Calcular data de vencimento da referência baseada na fee structure
//     */
//    private function calculateReferenceExpiryDate($month, $year)
//    {
//        $student = $this->selectedStudent;
//        $studentClassroom = $student->class->class ?? null;
//
//        // Buscar fee structure para obter dia de vencimento
//        $feeStructures = FeeStructure::where('active', 1)->get();
//        $filteredFeeStructures = $feeStructures->filter(function ($fee) use ($studentClassroom) {
//            $gradesArray = explode(',', $fee->grades);
//            return in_array($studentClassroom, $gradesArray);
//        });
//
//        $feeStructure = $filteredFeeStructures->first();
//        $dueDay = $feeStructure->payment_due_day ?? 5; // Default: dia 5
//
//        // Converter nome do mês para número
//        $monthNumber = $this->getMonthNumber($month);
//        
//        try {
//            // Criar data de vencimento
//            $expiryDate = Carbon::createFromDate($year, $monthNumber, $dueDay);
//            
//            // Se a data já passou no ano atual, não alterar (manter a data original)
//            if ($expiryDate->isPast() && $year == now()->year) {
//                // Para meses que já passaram, manter a data como estava
//                $expiryDate = $expiryDate; // Manter a data mesmo que seja passada
//            }
//            
//            Log::info('Calculated reference expiry date', [
//                'month' => $month,
//                'month_number' => $monthNumber,
//                'year' => $year,
//                'due_day' => $dueDay,
//                'expiry_date' => $expiryDate->toDateString(),
//                'is_past' => $expiryDate->isPast()
//            ]);
//            
//            return $expiryDate;
//            
//        } catch (\Exception $e) {
//            Log::error('Error calculating expiry date', [
//                'error' => $e->getMessage(),
//                'month' => $month,
//                'year' => $year,
//                'due_day' => $dueDay
//            ]);
//            
//            // Fallback: usar data atual + TTL days
//            $ttlDays = (int) config('payments.reference_ttl_days', 3);
//            return now()->addDays($ttlDays);
//        }
//    }
//    
//    public function generateReference($data)
//    {
//        Log::info('generateReference called', ['data' => $data]);
//        
//        try {
//            $validated = validator($data, [
//                'month' => 'required|string',
//                'year'  => 'required|integer|min:2020|max:' . (date('Y') + 1),
//                'amount'=> 'required|numeric|min:0.01',
//                'fine'  => 'nullable|numeric|min:0',
//                'discount' => 'nullable|numeric|min:0',
//                'paymentMode' => 'required|in:Reference,Entity_Reference',
//                'note' => 'nullable|string|max:500',
//            ])->validate();
//
//            $student = auth()->user();
//            if (!$student) {
//                throw new \RuntimeException('Sessão expirada');
//            }
//
//            // Verificar se já existe referência pendente para este mês/ano
//            $existingReference = PaymentReference::where([
//                'student_id' => $student->id,
//                'fee_month' => $validated['month'],
//                'fee_year' => $validated['year'],
//                'status' => 'pending'
//            ])->first();
//
//            if ($existingReference) {
//                throw new \RuntimeException('Já existe uma referência pendente para ' . $validated['month'] . '/' . $validated['year']);
//            }
//
//            $entity  = config('payments.entity', '11111');
//            $ttlDays = (int) config('payments.reference_ttl_days', 3);
//            $amount  = round(($validated['amount'] ?? 0), 2);
//            
//            // 1) calcula a validade (fim do dia) ANTES de gravar
//            $expiresAt = $this->calculateReferenceExpiryDate($validated['month'], (int)$validated['year']);
//
//
//            $referenceGenerator = app(\App\Services\ReferenceGenerator::class);
//            
//            // NOVO: Usar método que escolhe automaticamente o melhor campo do estudante
//            $reference = $referenceGenerator->makeFromStudent(
//                $validated['month'],
//                (int)$validated['year'],
//                $student,
//                $amount
//            );
//
//            if (!$referenceGenerator->validate($reference)) {
//                throw new \RuntimeException('Referência gerada é inválida');
//            }
//
//            Log::info('Reference generated using BCI algorithm', [
//                'student_id' => $student->id,
//                'month' => $validated['month'],
//                'year' => $validated['year'],
//                'amount' => $amount,
//                'reference' => $reference,
//                'reference_length' => strlen($reference),
//                'expires_at'       => $expiresAt->toDateTimeString()
//            ]);
//
//            // Salvar na tabela payment_references
//            $paymentReference = PaymentReference::create([
//                'student_id' => $student->id,
//                'entity_code' => $entity,
//                'reference_number' => $reference,
//                'amount' => $amount,
//                'fee_month' => $validated['month'],
//                'fee_year' => $validated['year'],
//                'expires_at' => $expiresAt,     // ✅ gravado
//                'status' => 'pending'
//            ]);
//
//            Log::info('Payment reference record created', ['payment_reference_id' => $paymentReference->id]);
//
//            
//            
//            Log::info('Reference expiry calculated', [
//                'month' => $validated['month'],
//                'year' => $validated['year'],
//                'expires_at' => $expiresAt->toDateString()
//            ]);
//            
//            try {
//                // Verificar se a view existe
//                if (!view()->exists('pdf.reference')) {
//                    Log::warning('PDF template not found, creating simple response');
//                    throw new \Exception('PDF template not available');
//                }
//
//                $pdf = Pdf::loadView('pdf.reference', [
//                    'student'    => $student,
//                    'payment'    => $paymentReference,
//                    'entity'     => $entity,
//                    'reference'  => $reference,
//                    'reference_formatted' => $referenceGenerator->format($reference),
//                    'amount'     => $amount,
//                    'expires_at' => $expiresAt,
//                ])
//                ->setOptions([
//                    'isRemoteEnabled' => false,
//                    'isHtml5ParserEnabled' => true,
//                    'isPhpEnabled' => false
//                ])
//                ->output();
//
//                // Nome do arquivo no formato SLUG_ID + REFERENCE + TIMESTAMP
//                $timestamp = now()->format('d-m-Y_H-i'); // Formato: dd-mm-yyyy_HH-mm
//                $studentSlug = Str::slug($student->name ?? $student->id);
//                $filename = "{$studentSlug}_{$reference}_{$timestamp}.pdf";
//                $path = "references/{$filename}";
//                
//                // Criar diretório se não existir
//                Storage::disk('public')->makeDirectory('references');
//                
//                // Salvar o arquivo
//                $saved = Storage::disk('public')->put($path, $pdf);
//                
//                if (!$saved) {
//                    throw new \Exception('Falhou ao salvar o arquivo PDF');
//                }
//                
//                // Verificar se o arquivo foi criado
//                if (!Storage::disk('public')->exists($path)) {
//                    throw new \Exception('Arquivo PDF não foi encontrado após criação');
//                }
//                
//                // Gerar URLs
//                $url = asset('storage/' . $path);
//                $urlAlt = url('storage/' . $path);
//                $urlManual = request()->getSchemeAndHttpHost() . '/storage/' . $path;
//                
//                $fileSize = Storage::disk('public')->size($path);
//                $fullPath = storage_path('app/public/' . $path);
//                
//                Log::info('PDF generated successfully', [
//                    'student_id' => $student->id,
//                    'student_slug' => $studentSlug,
//                    'reference' => $reference,
//                    'timestamp' => $timestamp,
//                    'filename' => $filename,
//                    'path' => $path,
//                    'url_asset' => $url,
//                    'url_alt' => $urlAlt,
//                    'url_manual' => $urlManual,
//                    'size' => $fileSize,
//                    'full_path' => $fullPath,
//                    'file_exists' => file_exists($fullPath),
//                    'public_path_exists' => file_exists(public_path('storage/' . $path))
//                ]);
//
//                $this->emit('referenceReady', [
//                    'pdf_url'   => $url,
//                    'pdf_url_alt' => $urlAlt,
//                    'pdf_url_manual' => $urlManual,
//                    'reference' => $reference,
//                    'reference_formatted' => $referenceGenerator->format($reference),
//                    'entity'    => $entity,
//                    'amount'    => $amount,
//                    'expires_at'=> $expiresAt->toDateString(),
//                    'expires_at_formatted' => $expiresAt->format('d/m/Y'),
//                    'payment_reference_id' => $paymentReference->id,
//                    'student_name' => $student->name,
//                    'file_info' => [
//                        'filename' => $filename,
//                        'path' => $path,
//                        'size' => $fileSize
//                    ]
//                ]);
//
//            } catch (\Exception $pdfError) {
//                Log::error('PDF generation failed', ['error' => $pdfError->getMessage()]);
//                
//                // Mesmo se o PDF falhar, emitir sucesso com dados da referência
//                $this->emit('referenceReady', [
//                    'pdf_url'   => null,
//                    'reference' => $reference,
//                    'reference_formatted' => $referenceGenerator->format($reference),
//                    'entity'    => $entity,
//                    'amount'    => $amount,
//                    'expires_at'=> $expiresAt->toDateString(),
//                    'expires_at_formatted' => $expiresAt->format('d/m/Y'),
//                    'payment_reference_id' => $paymentReference->id,
//                    'student_name' => $student->name,
//                    'manual_pdf' => true
//                ]);
//            }
//
//        } catch (\Illuminate\Validation\ValidationException $e) {
//            Log::error('Validation failed', ['errors' => $e->errors()]);
//            $this->emit('referenceError', 'Dados inválidos: ' . implode(', ', array_flatten($e->errors())));
//        } catch (\Throwable $e) {
//            Log::error('generateReference failed', [
//                'message' => $e->getMessage(),
//                'file' => $e->getFile(),
//                'line' => $e->getLine(),
//                'trace' => $e->getTraceAsString()
//            ]);
//            $this->emit('referenceError', 'Erro ao gerar a referência: ' . $e->getMessage());
//        }
//    }
//
//    private function calculateBaseFee()
//    {
//        $student = $this->selectedStudent;
//        $studentClassroom = $student->class->class ?? null;
//
//        $feeStructures = FeeStructure::where('active', 1)->get();
//        
//        $filteredFeeStructures = $feeStructures->filter(function ($fee) use ($studentClassroom) {
//            $gradesArray = explode(',', $fee->grades);
//            return in_array($studentClassroom, $gradesArray);
//        });
//
//        return $filteredFeeStructures->sum('monthly_fee');
//    }
//
//    private function calculateFine($month, $year)
//    {
//        $student = $this->selectedStudent;
//        $studentClassroom = $student->class->class ?? null;
//
//        $feeStructures = FeeStructure::where('active', 1)->get();
//        $filteredFeeStructures = $feeStructures->filter(function ($fee) use ($studentClassroom) {
//            $gradesArray = explode(',', $fee->grades);
//            return in_array($studentClassroom, $gradesArray);
//        });
//
//        $feeStructure = $filteredFeeStructures->first();
//        if (!$feeStructure) {
//            return 0;
//        }
//
//        $dueDay = $feeStructure->payment_due_day ?? 5;
//        $dueDate = Carbon::createFromDate($year, $month, $dueDay);
//        $currentDate = Carbon::now();
//
//        if ($currentDate->gt($dueDate)) {
//            $baseFee = $this->calculateBaseFee();
//            $latePenalty = $feeStructure->late_penalty_percentage ?? 0;
//            return ($baseFee * $latePenalty) / 100;
//        }
//
//        return 0;
//    }
//
//    private function validatePayment($month, $year)
//    {
//        $monthNumber = $this->getMonthNumber($month);
//        
//        $existingPayment = Fee_assign::where([
//            'student_id' => $this->studentId,
//            'month' => $month,
//            'year' => $year
//        ])->first();
//
//        if ($existingPayment) {
//            throw new \Exception("Já existe um pagamento registrado para {$month}/{$year}");
//        }
//
//        if ($monthNumber < 1 || $monthNumber > 12) {
//            throw new \Exception("Mês inválido");
//        }
//
//        if ($year < 2020 || $year > (date('Y') + 1)) {
//            throw new \Exception("Ano inválido");
//        }
//
//        return true;
//    }
//
//    private function getMonthNumber($monthName)
//    {
//        $months = [
//            'january' => 1, 'february' => 2, 'march' => 3, 'april' => 4,
//            'may' => 5, 'june' => 6, 'july' => 7, 'august' => 8,
//            'september' => 9, 'october' => 10, 'november' => 11, 'december' => 12
//        ];
//
//        return $months[strtolower($monthName)] ?? 0;
//    }
//
//    public function processPayment($paymentData)
//    {
//        try {
//            $month = $paymentData['month'];
//            $year = $paymentData['year'];
//            
//            $this->validatePayment($month, $year);
//            
//            $baseFee = $this->calculateBaseFee();
//            $fine = $this->calculateFine($this->getMonthNumber($month), $year);
//            
//            $discount = $this->validateDiscount($paymentData['discount'] ?? 0);
//            
//            $finalAmount = $baseFee + $fine - $discount;
//            
//            $finalAmount = max(0, $finalAmount);
//
//            $this->fill([
//                'month' => $month,
//                'year' => $year,
//                'discount' => $discount,
//                'paymentMode' => $paymentData['paymentMode'] ?? 'Cash',
//                'note' => $paymentData['note'] ?? null
//            ]);
//
//            $this->collectFee($finalAmount, $fine);
//
//        } catch (\Exception $e) {
//            $this->emit('paymentError', $e->getMessage());
//            session()->flash('error', $e->getMessage());
//        }
//    }
//
//    private function validateDiscount($requestedDiscount)
//    {
//        if ($requestedDiscount <= 0) {
//            return 0;
//        }
//
//        $student = $this->selectedStudent;
//        $maxDiscountAllowed = 0;
//        
//        if ($student->discount_group) {
//            $maxDiscountAllowed = $student->discount_group->max_discount ?? 0;
//        }
//        
//        return min($requestedDiscount, $maxDiscountAllowed);
//    }
//
//    public function preparePayment($feeId, $amount, $month, $year)
//    {
//        $this->selectedFeeAssignId = $feeId;
//        $this->month = $month;
//        $this->year = $year;
//        
//        $this->dispatchBrowserEvent('openModal');
//    }
//
//    public function collectFee($calculatedAmount = null, $calculatedFine = null)
//    {
//        if ($calculatedAmount === null) {
//            $calculatedAmount = $this->amount;
//        }
//        if ($calculatedFine === null) {
//            $calculatedFine = $this->fine;
//        }
//
//        $this->validate([
//            'month' => 'required|string',
//            'year' => 'required|numeric',
//            'discount' => 'numeric|min:0',
//            'paymentMode' => 'required',
//        ]);
//
//        try {
//            Fee_assign::create([
//                'student_id' => $this->studentId,
//                'amount' => $calculatedAmount,
//                'discount' => $this->discount,
//                'fine' => $calculatedFine,
//                'payment_mode' => $this->paymentMode,
//                'note' => $this->note,
//                'pay_day' => date("d"),
//                'pay_month' => date("m"),
//                'pay_year' => date("Y"),
//                'pay_type' => strtolower($this->paymentMode),
//                'status' => 'Paid',
//                'month' => $this->month,
//                'year' => $this->year,
//            ]);
//
//            $this->dispatchBrowserEvent('closeModal');
//            $this->emit('paymentSuccess');
//            
//            session()->flash('message', 'Pagamento registrado com sucesso!');
//            $this->reset(['discount', 'paymentMode', 'note', 'month', 'year']);
//
//        } catch (\Exception $e) {
//            $this->emit('paymentError', $e->getMessage());
//            session()->flash('error', $e->getMessage());
//        }
//    }
//
//    public function getPaymentDetails($month, $year)
//    {
//        $monthNumber = $this->getMonthNumber($month);
//        $baseFee = $this->calculateBaseFee();
//        $fine = $this->calculateFine($monthNumber, $year);
//        
//        return [
//            'baseFee' => $baseFee,
//            'fine' => $fine,
//            'total' => $baseFee + $fine,
//            'month' => $month,
//            'year' => $year
//        ];
//    }
//
//    public function render()
//    {
//        $currentYear = date('Y');
//        $student = $this->selectedStudent;
//        $studentClassroom = $student->class->class ?? null;
//
//        $feeStructures = FeeStructure::where('active', 1)->get();
//        $filteredFeeStructures = $feeStructures->filter(function ($fee) use ($studentClassroom) {
//            $gradesArray = explode(',', $fee->grades);
//            return in_array($studentClassroom, $gradesArray);
//        });
//
//        $fees = Fee_assign::where('student_id', $this->studentId)->get();
//        
//        $payment_references = PaymentReference::where('student_id', $this->studentId)
//            ->where('fee_year', $currentYear)
//            ->get();
//
//        $totalExpected = 0;
//        $totalPaid = 0;
//        $totalPending = 0;
//        $totalOverdue = 0;
//
//        $months = [
//            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
//            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
//            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
//        ];
//
//        $monthlyData = [];
//
//        foreach ($months as $num => $name) {
//            $baseFee = $this->calculateBaseFee();
//            $fine = $this->calculateFine($num, $currentYear);
//            
//            $feeStructure = $filteredFeeStructures->first();
//            $dueDay = $feeStructure->payment_due_day ?? 5;
//            $dueDate = Carbon::createFromDate($currentYear, $num, $dueDay);
//            $currentDate = Carbon::now();
//
//            $feeRecord = $fees->first(function ($f) use ($name, $currentYear) {
//                return strtolower($f->month) == strtolower($name) && $f->year == $currentYear;
//            });
//
//            $paid = $feeRecord && strtolower($feeRecord->status) == 'paid';
//            $amountPaid = $feeRecord->amount ?? 0;
//
//            $expected = $baseFee;
//            $totalExpected += $expected;
//
//            if ($paid) {
//                $totalPaid += $amountPaid;
//            } else {
//                if ($currentDate->gt($dueDate)) {
//                    $totalOverdue += $expected + $fine;
//                } else {
//                    $totalPending += $expected;
//                }
//            }
//
//            $monthlyData[] = [
//                'month' => "$name $currentYear",
//                'month_name' => $name,
//                'month_number' => $num,
//                'year' => $currentYear,
//                'due_date' => $dueDate->format('d/m/Y'),
//                'status' => $paid ? 'Pago' : ($currentDate->gt($dueDate) ? 'Em Atraso' : 'Não Pago'),
//                'expected' => $expected,
//                'fine' => $fine,
//                'paid' => $amountPaid,
//                'remaining' => max(0, $expected - $amountPaid),
//            ];
//        }
//
//        return view('livewire.user.fee.fee-collection', [
//            'student' => $student,
//            'fees' => $fees,
//            'feestructures' => $filteredFeeStructures,
//            'monthlyData' => $monthlyData,
//            'totalExpected' => $totalExpected,
//            'totalPaid' => $totalPaid,
//            'totalPending' => $totalPending,
//            'totalOverdue' => $totalOverdue,
//            'currentYear' => $currentYear,
//            'payment_references' => $payment_references,
//        ]);
//    }
//
//    public function openModal($feeAssignId, $amount, $month, $year)
//    {
//        $this->selectedFeeAssignId = $feeAssignId;
//        $this->month = (string) $month;
//        $this->year = (int) $year;
//        
//        $this->dispatchBrowserEvent('openModal');
//        $this->emit('openModal', $feeAssignId, $amount, $month, $year);
//    }
//}


namespace App\Http\Livewire\User\Fee;

use App\Models\Fee_assign;
use App\Models\PaymentReference;
use App\Models\User;
use App\Services\FeeCalculationService;
use Livewire\Component;
use App\Models\FeeStructure;
use Carbon\Carbon;
use LaravelMultipleGuards\Traits\FindGuard;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Arr;

class FeeCollectionUser extends Component
{
    use FindGuard;
    
    public $studentId;
    public $feeGroup;
    public $date;
    public $discountGroup;
    public $discount = 0;
    public $paymentMode = 'Cash';
    public $note;
    public $students = [];
    public $selectedStudent;
    public $selectedFee;
    public $selectedFeeAssignId;
    public $month;
    public $year;
    public $amount;
    public $fine;
    public $selectedYear;
    public $selectedReference;
    public $showReferenceModal = false;

    protected $feeCalculationService;

    protected $listeners = [
        'openModal' => 'preparePayment',
        'collectPayment' => 'processPayment',
        'generateReference' => 'generateReference'
    ];

    public function boot(FeeCalculationService $feeCalculationService)
    {
        $this->feeCalculationService = $feeCalculationService;
    }

    public function mount()
    {
        $user = $this->findGuardType()->user();
        if ($user) {
            $this->studentId = $user->id;
            $this->selectedStudent = User::findOrFail($this->studentId);
            $this->date = now()->format('d/m/Y');
            $this->selectedYear = now()->year;
        } else {
            abort(404, 'User not found');
        }
    }

    /**
     * Atualizar quando o ano selecionado mudar
     */
    public function updatedSelectedYear()
    {
        // Força re-renderização com novo ano
        $this->render();
    }

    /**
     * Calcular data de vencimento da referência baseada na fee structure
     */
    private function calculateReferenceExpiryDate($month, $year)
    {
        try {
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $this->selectedStudent,
                $month,
                (int)$year
            );

            // Se tem due_date calculada, usa ela
            if ($calculation['due_date']) {
                return $calculation['due_date'];
            }

            // Fallback para lógica original
            $student = $this->selectedStudent;
            $studentClassroom = $student->class->class ?? $student->classroom->class ?? null;

            $feeStructures = FeeStructure::where('active', 1)->get();
            $filteredFeeStructures = $feeStructures->filter(function ($fee) use ($studentClassroom) {
                $gradesArray = explode(',', $fee->grades);
                return in_array($studentClassroom, $gradesArray);
            });

            $feeStructure = $filteredFeeStructures->first();
            $dueDay = $feeStructure->payment_due_day ?? 5;

            $monthNumber = $this->getMonthNumber($month);
            $expiryDate = Carbon::createFromDate($year, $monthNumber, $dueDay);
            
            Log::info('Calculated reference expiry date', [
                'month' => $month,
                'month_number' => $monthNumber,
                'year' => $year,
                'due_day' => $dueDay,
                'expiry_date' => $expiryDate->toDateString(),
                'is_past' => $expiryDate->isPast()
            ]);
            
            return $expiryDate;
            
        } catch (\Exception $e) {
            Log::error('Error calculating expiry date', [
                'error' => $e->getMessage(),
                'month' => $month,
                'year' => $year
            ]);
            
            $ttlDays = (int) config('payments.reference_ttl_days', 3);
            return now()->addDays($ttlDays);
        }
    }
    
    public function generateReference($data)
    {
        Log::info('generateReference called', ['data' => $data]);

        // Rate limiting: 3 tentativas por minuto
        $rateLimitKey = 'generate-reference:' . auth()->id();

        if (\Illuminate\Support\Facades\RateLimiter::tooManyAttempts($rateLimitKey, 3)) {
            $seconds = \Illuminate\Support\Facades\RateLimiter::availableIn($rateLimitKey);
            throw new \RuntimeException("Muitas tentativas. Tente novamente em {$seconds} segundos.");
        }

        \Illuminate\Support\Facades\RateLimiter::hit($rateLimitKey, 60);

        try {
            $student = auth()->user();
            if (!$student) {
                throw new \RuntimeException('Sessão expirada');
            }

            // USAR O SERVICE UNIFICADO para calcular valores corretos primeiro
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $student,
                $data['month'] ?? null,
                (int)($data['year'] ?? date('Y'))
            );

            // Validação básica (SEM validar amount, pois vamos usar o valor calculado)
            $validated = validator($data, [
                'month' => 'required|string',
                'year'  => 'required|integer|min:2020|max:' . (date('Y') + 1),
                'paymentMode' => 'required|in:Reference,Entity_Reference',
                'note' => 'nullable|string|max:500',
                'fee_structure_id' => 'nullable|integer',
            ])->validate();

            $entity = config('payments.entity', '11111');
            $feeStructureId = isset($data['fee_structure_id']) ? (int)$data['fee_structure_id'] : null;

            // Se tem fee_structure_id, calcular apenas para essa taxa específica
            if ($feeStructureId) {
                $feeStructure = FeeStructure::find($feeStructureId);
                if (!$feeStructure) {
                    throw new \RuntimeException('Taxa não encontrada');
                }

                // Buscar referências existentes para esta taxa (não pagas)
                $existingRefs = PaymentReference::where('student_id', $student->id)
                    ->where('fee_month', $validated['month'])
                    ->where('fee_year', $validated['year'])
                    ->where('status', '!=', 'paid')
                    ->where(function($q) use ($feeStructureId) {
                        $q->whereJsonContains('metadata->fee_structure_id', $feeStructureId)
                          ->orWhereJsonContains('metadata->fee_structure_id', (string)$feeStructureId);
                    })
                    ->orderBy('created_at', 'desc')
                    ->get();

                // Lógica de regeneração em 3 níveis:
                // 1ª vez: Estudante gera (sem referência existente)
                // 2ª vez: Estudante regenera (se 1ª expirou) - regeneration_count = 1
                // 3ª vez+: Só admin (se 2ª expirou após 7 dias)

                $isRegeneration = false;
                $newRegenerationCount = 0;

                if ($existingRefs->isNotEmpty()) {
                    $latestRef = $existingRefs->first();
                    $isExpired = $latestRef->expires_at && Carbon::parse($latestRef->expires_at)->isPast();
                    $regenerationCount = $latestRef->metadata['regeneration_count'] ?? 0;

                    if (!$isExpired) {
                        // Referência ainda válida - não pode gerar nova
                        throw new \RuntimeException('Já existe uma referência pendente válida para esta taxa: ' . $feeStructure->fee_name);
                    }

                    if ($regenerationCount >= 1) {
                        // Já foi regenerada uma vez pelo estudante (2ª referência)
                        // Verificar se passou 7 dias desde a expiração
                        $expiredAt = Carbon::parse($latestRef->expires_at);
                        $daysSinceExpiry = $expiredAt->diffInDays(now());

                        if ($daysSinceExpiry >= 7) {
                            // Passou 7+ dias - só admin pode regenerar
                            throw new \RuntimeException('A referência regenerada expirou há mais de 7 dias. Contacte o administrador para gerar nova referência.');
                        } else {
                            // Ainda não passou 7 dias - só admin pode regenerar (já usou sua chance)
                            throw new \RuntimeException('Já utilizou sua opção de regenerar. Contacte o administrador para nova referência.');
                        }
                    }

                    // Estudante pode regenerar (1ª vez expirou, ainda não regenerou)
                    $isRegeneration = true;
                    $newRegenerationCount = $regenerationCount + 1;

                    Log::info('Student regenerating expired reference', [
                        'student_id' => $student->id,
                        'fee_structure_id' => $feeStructureId,
                        'old_reference_id' => $latestRef->id,
                        'regeneration_count' => $newRegenerationCount
                    ]);

                    // Marcar referência antiga como substituída
                    $latestRef->update([
                        'status' => 'replaced',
                        'metadata' => array_merge($latestRef->metadata ?? [], [
                            'replaced_at' => now()->toDateTimeString(),
                            'replaced_reason' => 'student_regeneration',
                            'replaced_by_regeneration' => true
                        ])
                    ]);
                }

                // Calcular valores para esta taxa específica
                $baseAmount = (float)$feeStructure->monthly_fee;
                $fineAmount = 0;

                // Calcular multa se estiver em atraso
                $monthNumber = $this->getMonthNumber($validated['month']);
                $dueDay = $feeStructure->payment_due_day ?? 5;
                // Obter último dia do mês para evitar datas inválidas
                $yearInt = (int)$validated['year'];
                $lastDayOfMonth = Carbon::create($yearInt, $monthNumber, 1)->endOfMonth()->day;
                $dueDate = Carbon::create($yearInt, $monthNumber, min($dueDay, $lastDayOfMonth), 23, 59, 59);

                Log::info('Due date calculation for fee structure', [
                    'fee_structure_id' => $feeStructureId,
                    'year' => $yearInt,
                    'month_number' => $monthNumber,
                    'due_day_config' => $dueDay,
                    'last_day_of_month' => $lastDayOfMonth,
                    'due_date' => $dueDate->format('Y-m-d H:i:s'),
                    'is_past' => $dueDate->isPast(),
                    'now' => now()->format('Y-m-d H:i:s')
                ]);

                if ($dueDate->isPast() && ($feeStructure->late_penalty_percentage ?? 0) > 0) {
                    $penaltyType = $feeStructure->penalty_type ?? 'percentage';
                    if (strtolower($penaltyType) === 'fixed') {
                        $fineAmount = (float)$feeStructure->late_penalty_percentage;
                    } else {
                        $fineAmount = ($baseAmount * $feeStructure->late_penalty_percentage) / 100;
                    }
                }

                $discount = (float)($data['discount'] ?? 0);
                $totalAmount = round($baseAmount + $fineAmount - $discount, 2);
                $taxName = $feeStructure->fee_name;

                Log::info('Generating reference for specific fee structure', [
                    'fee_structure_id' => $feeStructureId,
                    'fee_name' => $taxName,
                    'base_amount' => $baseAmount,
                    'fine_amount' => $fineAmount,
                    'total_amount' => $totalAmount
                ]);
            } else {
                // Comportamento antigo: calcular todas as taxas do mês
                $discount = (float)($data['discount'] ?? 0);
                $totalAmount = round($calculation['base_amount'] + $calculation['fine_amount'] - $discount, 2);
                $baseAmount = $calculation['base_amount'];
                $fineAmount = $calculation['fine_amount'];
                $taxName = null;
                $feeStructureId = null;
            }

            Log::info('Using calculated amount', [
                'frontend_amount' => $data['amount'] ?? 'not provided',
                'calculated_base' => $baseAmount,
                'calculated_fine' => $fineAmount,
                'discount' => $discount,
                'final_total' => $totalAmount,
                'fee_structure_id' => $feeStructureId
            ]);

            // Calcula validade baseada na FeeStructure específica (se disponível)
            if ($feeStructureId && isset($feeStructure)) {
                // Usar data de vencimento da taxa específica
                $expiresAt = $dueDate; // $dueDate já foi calculada acima para esta taxa
            } else {
                // Comportamento antigo: usar primeira FeeStructure
                $expiresAt = $this->calculateReferenceExpiryDate($validated['month'], (int)$validated['year']);
            }

            // Manter a data de vencimento original conforme configuração no sistema

            $referenceGenerator = app(\App\Services\ReferenceGenerator::class);

            $reference = $referenceGenerator->makeV1FromStudent(
                $validated['month'],
                $student,
                $totalAmount
            );

            if (!$referenceGenerator->validate($reference)) {
                throw new \RuntimeException('Referência gerada é inválida');
            }

            Log::info('Reference generated with unified service', [
                'student_id' => $student->id,
                'month' => $validated['month'],
                'year' => $validated['year'],
                'base_amount' => $calculation['base_amount'],
                'calculated_fine' => $calculation['fine_amount'],
                'discount' => $discount,
                'total_amount' => $totalAmount,
                'reference' => $reference,
                'expires_at' => $expiresAt->toDateTimeString(),
                'is_late_payment' => $calculation['is_late_payment']
            ]);

            // INICIAR TRANSAÇÃO DB
            \Illuminate\Support\Facades\DB::beginTransaction();

            try {
                // Preparar metadata com fee_structure_id se disponível
                $metadata = [];
                if ($feeStructureId) {
                    $metadata['fee_structure_id'] = $feeStructureId;
                    $metadata['fee_name'] = $taxName;
                    $metadata['description'] = $taxName;

                    // Adicionar regeneration_count se for regeneração
                    if ($isRegeneration) {
                        $metadata['regeneration_count'] = $newRegenerationCount;
                        $metadata['regenerated_at'] = now()->toDateTimeString();
                        $metadata['is_student_regeneration'] = true;

                        // Para referências regeneradas, validade é +7 dias a partir de hoje
                        $expiresAt = now()->addDays(7)->endOfDay();

                        Log::info('Regenerated reference - setting +7 days expiry', [
                            'original_expiry' => $dueDate->format('Y-m-d'),
                            'new_expiry' => $expiresAt->format('Y-m-d'),
                            'regeneration_count' => $newRegenerationCount
                        ]);
                    }
                }

                // Salvar na tabela payment_references com valores corretos
                $paymentReference = PaymentReference::create([
                    'student_id' => $student->id,
                    'entity_code' => $entity,
                    'reference_number' => $reference,
                    'amount' => $totalAmount,
                    'fine_amount' => $fineAmount,
                    'fee_month' => $validated['month'],
                    'fee_year' => $validated['year'],
                    'expires_at' => $expiresAt,
                    'status' => 'pending',
                    'metadata' => !empty($metadata) ? $metadata : null
                ]);

                // Commit da transação APÓS criar o registro
                \Illuminate\Support\Facades\DB::commit();

            } catch (\Illuminate\Database\QueryException $e) {
                \Illuminate\Support\Facades\DB::rollBack();

                // Verificar se é erro de duplicate entry (constraint única)
                if ($e->getCode() == 23000) {
                    throw new \RuntimeException('Já existe uma referência pendente para ' . $validated['month'] . '/' . $validated['year']);
                }
                throw $e;
            }

            try {
                if (!view()->exists('pdf.reference')) {
                    Log::warning('PDF template not found, creating simple response');
                    throw new \Exception('PDF template not available');
                }

                $pdf = Pdf::loadView('pdf.reference', [
                    'student'    => $student,
                    'payment'    => $paymentReference,
                    'entity'     => $entity,
                    'reference'  => $reference,
                    'reference_formatted' => $referenceGenerator->format($reference),
                    'amount'     => $totalAmount,
                    'base_amount' => $baseAmount,
                    'fine_amount' => $fineAmount,
                    'discount_amount' => $discount,
                    'expires_at' => $expiresAt,
                    'calculation_details' => $calculation,
                    'fee_name' => $taxName
                ])
                ->setOptions([
                    'isRemoteEnabled' => false,
                    'isHtml5ParserEnabled' => true,
                    'isPhpEnabled' => false
                ])
                ->output();

                $timestamp = now()->format('d-m-Y_H-i');
                $studentSlug = Str::slug($student->name ?? $student->id);
                $filename = "{$studentSlug}_{$reference}_{$timestamp}.pdf";
                $path = "references/{$filename}";
                
                Storage::disk('public')->makeDirectory('references');
                $saved = Storage::disk('public')->put($path, $pdf);
                
                if (!$saved || !Storage::disk('public')->exists($path)) {
                    throw new \Exception('Falhou ao salvar o arquivo PDF');
                }
                
                $url = asset('storage/' . $path);
                $fileSize = Storage::disk('public')->size($path);
                
                Log::info('PDF generated successfully with preserved fines', [
                    'student_id' => $student->id,
                    'reference' => $reference,
                    'filename' => $filename,
                    'base_amount' => $calculation['base_amount'],
                    'fine_amount' => $calculation['fine_amount'],
                    'total_amount' => $totalAmount,
                    'size' => $fileSize
                ]);

                $this->emit('referenceReady', [
                    'pdf_url' => $url,
                    'reference' => $reference,
                    'reference_formatted' => $referenceGenerator->format($reference),
                    'entity' => $entity,
                    'amount' => $totalAmount,
                    'base_amount' => $calculation['base_amount'],
                    'fine_amount' => $calculation['fine_amount'],
                    'discount_amount' => $discount,
                    'expires_at' => $expiresAt->toDateString(),
                    'expires_at_formatted' => $expiresAt->format('d/m/Y'),
                    'payment_reference_id' => $paymentReference->id,
                    'student_name' => $student->name,
                    'is_late_payment' => $calculation['is_late_payment'],
                    'file_info' => [
                        'filename' => $filename,
                        'path' => $path,
                        'size' => $fileSize
                    ]
                ]);

                // Emitir também evento de sucesso para fechar o modal
                $this->emit('referenceGenerated', [
                    'message' => 'Referência gerada com sucesso!'
                ]);

            } catch (\Exception $pdfError) {
                Log::error('PDF generation failed', ['error' => $pdfError->getMessage()]);

                $this->emit('referenceReady', [
                    'pdf_url' => null,
                    'reference' => $reference,
                    'reference_formatted' => $referenceGenerator->format($reference),
                    'entity' => $entity,
                    'amount' => $totalAmount,
                    'base_amount' => $calculation['base_amount'],
                    'fine_amount' => $calculation['fine_amount'],
                    'expires_at' => $expiresAt->toDateString(),
                    'expires_at_formatted' => $expiresAt->format('d/m/Y'),
                    'payment_reference_id' => $paymentReference->id,
                    'student_name' => $student->name,
                    'manual_pdf' => true
                ]);

                // Emitir também evento de sucesso para fechar o modal
                $this->emit('referenceGenerated', [
                    'message' => 'Referência gerada com sucesso! (PDF não disponível)'
                ]);
            }

        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation failed', ['errors' => $e->errors()]);
            $this->emit('referenceError', 'Dados inválidos: ' . implode(', ', Arr::flatten($e->errors())));
        } catch (\Throwable $e) {
            Log::error('generateReference failed', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            $this->emit('referenceError', 'Erro ao gerar a referência: ' . $e->getMessage());
        }
    }

    public function processPayment($paymentData)
    {
        try {
            $month = $paymentData['month'];
            $year = $paymentData['year'];
            
            $this->validatePayment($month, $year);
            
            // USA O SERVICE UNIFICADO
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $this->selectedStudent,
                $month,
                (int)$year
            );
            
            $discount = $this->validateDiscount($paymentData['discount'] ?? 0);
            $manualFine = (float)($paymentData['fine'] ?? 0);
            
            // Valor final com multas preservadas
            $totalFine = max($calculation['fine_amount'], $manualFine);
            $finalAmount = max(0, $calculation['base_amount'] + $totalFine - $discount);

            $this->fill([
                'month' => $month,
                'year' => $year,
                'discount' => $discount,
                'paymentMode' => $paymentData['paymentMode'] ?? 'Cash',
                'note' => $paymentData['note'] ?? null
            ]);

            // Usa service unificado para criar pagamento
            $paymentInfo = [
                'month' => $month,
                'year' => $year,
                'amount' => $finalAmount,
                'fine' => $manualFine, // Multa manual adicional
                'discount' => $discount,
                'payment_mode' => $paymentData['paymentMode'] ?? 'Cash',
                'paymentMode' => $paymentData['paymentMode'] ?? 'Cash',
                'note' => $paymentData['note'] ?? null,
                'pay_type' => strtolower($paymentData['paymentMode'] ?? 'cash')
            ];

            $feeAssign = $this->feeCalculationService->createFeePayment(
                $this->selectedStudent,
                $paymentInfo,
                $calculation
            );

            Log::info('User fee payment processed with preserved fines', [
                'student_id' => $this->studentId,
                'fee_assign_id' => $feeAssign->id,
                'base_calculated' => $calculation['base_amount'],
                'fine_calculated' => $calculation['fine_amount'],
                'fine_manual' => $manualFine,
                'fine_preserved' => $feeAssign->fine,
                'total_amount' => $finalAmount,
                'is_late_payment' => $calculation['is_late_payment']
            ]);

            $this->emit('paymentSuccess');

        } catch (\Exception $e) {
            Log::error('User payment processing failed', [
                'student_id' => $this->studentId,
                'error' => $e->getMessage()
            ]);
            $this->emit('paymentError', $e->getMessage());
            session()->flash('error', $e->getMessage());
        }
    }

    private function validatePayment($month, $year)
    {
        $monthNumber = $this->getMonthNumber($month);
        
        // Usa service para verificar pagamento existente
        if ($this->feeCalculationService->hasPaymentForPeriod($this->selectedStudent, $month, $year)) {
            throw new \Exception("Já existe um pagamento registrado para {$month}/{$year}");
        }

        if ($monthNumber < 1 || $monthNumber > 12) {
            throw new \Exception("Mês inválido");
        }

        if ($year < 2020 || $year > (date('Y') + 1)) {
            throw new \Exception("Ano inválido");
        }

        return true;
    }

    private function getMonthNumber($monthName)
    {
        $months = [
            // Inglês
            'january' => 1, 'february' => 2, 'march' => 3, 'april' => 4,
            'may' => 5, 'june' => 6, 'july' => 7, 'august' => 8,
            'september' => 9, 'october' => 10, 'november' => 11, 'december' => 12,
            // Português
            'janeiro' => 1, 'fevereiro' => 2, 'março' => 3, 'abril' => 4,
            'maio' => 5, 'junho' => 6, 'julho' => 7, 'agosto' => 8,
            'setembro' => 9, 'outubro' => 10, 'novembro' => 11, 'dezembro' => 12
        ];

        return $months[strtolower($monthName)] ?? 0;
    }

    private function validateDiscount($requestedDiscount)
    {
        if ($requestedDiscount <= 0) {
            return 0;
        }

        $student = $this->selectedStudent;
        $maxDiscountAllowed = 0;
        
        if ($student->discount_group) {
            $maxDiscountAllowed = $student->discount_group->max_discount ?? 0;
        }
        
        return min($requestedDiscount, $maxDiscountAllowed);
    }

    public function preparePayment($feeId, $amount, $month, $year)
    {
        $this->selectedFeeAssignId = $feeId;
        $this->month = $month;
        $this->year = $year;
        
        $this->dispatchBrowserEvent('openModal');
    }

    public function collectFee($calculatedAmount = null, $calculatedFine = null)
    {
        if ($calculatedAmount === null) {
            $calculatedAmount = $this->amount;
        }
        if ($calculatedFine === null) {
            $calculatedFine = $this->fine;
        }

        $this->validate([
            'month' => 'required|string',
            'year' => 'required|numeric',
            'discount' => 'numeric|min:0',
            'paymentMode' => 'required',
        ]);

        try {
            // Usa service unificado
            $paymentData = [
                'month' => $this->month,
                'year' => $this->year,
                'amount' => $calculatedAmount,
                'fine' => $calculatedFine,
                'discount' => $this->discount,
                'payment_mode' => $this->paymentMode,
                'paymentMode' => $this->paymentMode,
                'note' => $this->note,
                'pay_type' => strtolower($this->paymentMode)
            ];

            $feeAssign = $this->feeCalculationService->createFeePayment(
                $this->selectedStudent,
                $paymentData
            );

            $this->dispatchBrowserEvent('closeModal');
            $this->emit('paymentSuccess');
            
            session()->flash('message', 'Pagamento registrado com sucesso! Multa preservada no histórico.');
            $this->reset(['discount', 'paymentMode', 'note', 'month', 'year']);

        } catch (\Exception $e) {
            Log::error('User manual payment failed', [
                'student_id' => $this->studentId,
                'error' => $e->getMessage()
            ]);
            $this->emit('paymentError', $e->getMessage());
            session()->flash('error', $e->getMessage());
        }
    }

    public function getPaymentDetails($month, $year)
    {
        $calculation = $this->feeCalculationService->calculateFeeForStudent(
            $this->selectedStudent,
            $month,
            (int)$year
        );
        
        return [
            'baseFee' => $calculation['base_amount'],
            'fine' => $calculation['fine_amount'],
            'total' => $calculation['total_amount'],
            'month' => $month,
            'year' => $year,
            'isLate' => $calculation['is_late_payment'],
            'dueDate' => $calculation['due_date']?->format('d/m/Y')
        ];
    }

    /**
     * Baixar comprovativo de pagamento
     */
    public function downloadReceipt($paymentId)
    {
        \Log::info('downloadReceipt chamado', ['payment_id' => $paymentId]);

        try {
            $payment = PaymentReference::findOrFail($paymentId);
            \Log::info('Payment encontrado', ['payment' => $payment->toArray()]);

            // Verificar se o pagamento pertence ao estudante logado
            if ($payment->student_id !== $this->studentId) {
                \Log::warning('Acesso negado - student_id não coincide', [
                    'payment_student_id' => $payment->student_id,
                    'current_student_id' => $this->studentId
                ]);
                $this->dispatchBrowserEvent('error', [
                    'message' => 'Acesso negado a este comprovativo'
                ]);
                return;
            }

            // Verificar se o pagamento está pago
            if ($payment->status !== 'paid') {
                \Log::warning('Pagamento não confirmado', ['status' => $payment->status]);
                $this->dispatchBrowserEvent('error', [
                    'message' => 'Comprovativo disponível apenas para pagamentos confirmados'
                ]);
                return;
            }

            \Log::info('Chamando PaymentReceiptService');
            // Usar o serviço para gerar/obter o comprovativo
            $receiptService = new \App\Services\PaymentReceiptService();
            $result = $receiptService->downloadReceipt($payment);

            \Log::info('Resultado do serviço', ['result' => $result]);

            if ($result['success']) {
                // Retornar URL para download
                $this->dispatchBrowserEvent('download-receipt', [
                    'url' => $result['url'],
                    'message' => 'Comprovativo gerado com sucesso!'
                ]);
                \Log::info('Evento download-receipt disparado', ['url' => $result['url']]);
            } else {
                $this->dispatchBrowserEvent('error', [
                    'message' => $result['message'] ?? 'Erro ao gerar comprovativo'
                ]);
                \Log::warning('Erro no serviço', ['message' => $result['message'] ?? 'Desconhecido']);
            }

        } catch (\Exception $e) {
            \Log::error('Erro ao baixar comprovativo', [
                'payment_id' => $paymentId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            $this->dispatchBrowserEvent('error', [
                'message' => 'Erro ao processar comprovativo. Tente novamente.'
            ]);
        }
    }

    /**
     * Baixar recibo oficial de pagamento
     */
    public function downloadOfficialReceipt($paymentId)
    {
        try {
            $payment = PaymentReference::findOrFail($paymentId);

            // Verificar se o pagamento pertence ao estudante logado
            if ($payment->student_id !== $this->studentId) {
                $this->dispatchBrowserEvent('error', [
                    'message' => 'Acesso negado a este recibo'
                ]);
                return;
            }

            // Verificar se o pagamento está pago
            if ($payment->status !== 'paid') {
                $this->dispatchBrowserEvent('error', [
                    'message' => 'Recibo disponível apenas para pagamentos confirmados'
                ]);
                return;
            }

            // Usar o serviço para gerar/obter o recibo oficial
            $receiptService = new \App\Services\PaymentReceiptService();
            $result = $receiptService->downloadOfficialReceipt($payment);

            if ($result['success']) {
                // Retornar URL para download
                $this->dispatchBrowserEvent('download-receipt', [
                    'url' => $result['url'],
                    'message' => 'Recibo oficial gerado com sucesso!'
                ]);
            } else {
                $this->dispatchBrowserEvent('error', [
                    'message' => $result['message'] ?? 'Erro ao gerar recibo oficial'
                ]);
            }

        } catch (\Exception $e) {
            Log::error('Erro ao baixar recibo oficial', [
                'payment_id' => $paymentId,
                'error' => $e->getMessage()
            ]);

            $this->dispatchBrowserEvent('error', [
                'message' => 'Erro ao processar recibo oficial. Tente novamente.'
            ]);
        }
    }

    /**
     * Alias para downloadCashPosReceipt (compatibilidade com componente)
     */
    public function downloadPaymentReceipt($feeAssignId)
    {
        return $this->downloadCashPosReceipt($feeAssignId);
    }

    /**
     * Baixar recibo de pagamento Cash/POS (MESMO MÉTODO DO ADMIN)
     */
    public function downloadCashPosReceipt($feeAssignId)
    {
        try {
            $feeAssign = Fee_assign::findOrFail($feeAssignId);

            // Verificar se o pagamento pertence ao estudante logado
            if ($feeAssign->student_id !== $this->studentId) {
                $this->dispatchBrowserEvent('error', [
                    'message' => 'Acesso negado a este recibo'
                ]);
                return;
            }

            // Verificar se está pago
            if (strtolower($feeAssign->status) !== 'paid') {
                $this->dispatchBrowserEvent('error', [
                    'message' => 'Recibo disponível apenas para pagamentos confirmados'
                ]);
                return;
            }

            // Gerar ou obter número de recibo sequencial
            $receiptNumber = \App\Services\ReceiptNumberService::getOrGenerateReceiptNumber(
                'fee_assign',
                $feeAssign->id,
                'student',
                auth()->id() ? (int)auth()->id() : null
            );

            // Preparar dados compatíveis com receipt-official.blade.php (IGUAL AO ADMIN)
            $baseAmount = $feeAssign->amount;
            $fineAmount = $feeAssign->fine ?? 0;
            $discountAmount = $feeAssign->discount ?? 0;
            $totalPaid = $baseAmount + $fineAmount - $discountAmount;

            // Criar objeto simulado para compatibilidade
            $paymentData = (object)[
                'id' => $feeAssign->id,
                'fee_month' => $feeAssign->month,
                'fee_year' => $feeAssign->year,
            ];

            // Determinar método de pagamento
            $paymentMethod = 'Dinheiro';
            if ($feeAssign->payment_mode === 'POS' || strtolower($feeAssign->pay_type ?? '') === 'pos') {
                $paymentMethod = 'POS';
            }

            // Gerar PDF do recibo oficial (MESMO TEMPLATE DO ADMIN)
            $pdf = Pdf::loadView('pdf.receipt-official', [
                'payment' => $paymentData,
                'student' => $this->selectedStudent,
                'base_amount' => $baseAmount,
                'fine_amount' => $fineAmount,
                'discount_amount' => $discountAmount,
                'amount_paid' => $totalPaid,
                'paid_at' => \Carbon\Carbon::parse($feeAssign->payment_date ?? $feeAssign->created_at),
                'payment_method' => $paymentMethod,
                'transaction_id' => $feeAssign->transaction_id ?? null,
                'receipt_number' => $receiptNumber,
            ]);

            return response()->streamDownload(function() use ($pdf) {
                echo $pdf->output();
            }, "recibo_pagamento_{$feeAssign->student_id}_{$feeAssign->month}_{$feeAssign->year}.pdf");

        } catch (\Exception $e) {
            Log::error('Erro ao gerar recibo Cash/POS', [
                'fee_assign_id' => $feeAssignId,
                'error' => $e->getMessage()
            ]);

            $this->dispatchBrowserEvent('error', [
                'message' => 'Erro ao processar recibo. Tente novamente.'
            ]);
        }
    }

    /**
     * Exportar PDF com histórico completo do estudante
     */
    public function exportPdf()
    {
        try {
            $student = $this->selectedStudent;
            $currentYear = $this->selectedYear ?? now()->year;

            // Buscar histórico completo
            $payment_references = $this->getAllPaymentReferencesAndFees($currentYear);

            // Calcular totais
            $yearSummary = $this->feeCalculationService->calculateYearSummary($student, $currentYear);

            $pdf = Pdf::loadView('exports.student-finance-pdf', [
                'student' => $student,
                'year' => $currentYear,
                'payment_references' => $payment_references,
                'totalExpected' => $yearSummary['total_expected'] ?? 0,
                'totalPaid' => $yearSummary['total_paid'] ?? 0,
                'totalPending' => $yearSummary['total_pending'] ?? 0,
                'totalOverdue' => $yearSummary['total_overdue'] ?? 0,
                'generated_at' => now()->format('d/m/Y H:i'),
            ]);

            return response()->streamDownload(function() use ($pdf) {
                echo $pdf->output();
            }, "historico_pagamentos_{$student->student_id}_{$currentYear}.pdf");

        } catch (\Exception $e) {
            Log::error('Erro ao gerar PDF de histórico', [
                'student_id' => $this->studentId,
                'error' => $e->getMessage()
            ]);

            $this->dispatchBrowserEvent('error', [
                'message' => 'Erro ao gerar PDF. Tente novamente.'
            ]);
        }
    }

    /**
     * Ver detalhes de uma referência específica
     */
    public function viewReferenceById($referenceId)
    {
        try {
            // Verificar se é pendente (da FeeStructure)
            if (str_starts_with($referenceId, 'pending_')) {
                // Extrair informações do ID: pending_{feeStructureId}_{month}_{year}
                $parts = explode('_', $referenceId);
                $feeStructureId = $parts[1] ?? null;
                $monthName = $parts[2] ?? '';
                $year = $parts[3] ?? date('Y');

                $student = $this->selectedStudent;
                $feeStructure = FeeStructure::find($feeStructureId);

                if (!$feeStructure) {
                    throw new \Exception('Taxa não encontrada');
                }

                // Calcular valores
                $baseAmount = $feeStructure->monthly_fee;
                $fineAmount = 0;
                $dueDate = null;
                $isOverdue = false;

                $monthNumber = $this->getMonthNumber($monthName);
                if ($monthNumber) {
                    $dueDay = $feeStructure->payment_due_day ?? 5;
                    $dueDate = Carbon::createFromDate($year, $monthNumber, min($dueDay, 28));
                    $isOverdue = $dueDate->isPast();

                    if ($isOverdue && ($feeStructure->late_penalty_percentage ?? 0) > 0) {
                        $penaltyType = $feeStructure->penalty_type ?? 'percentage';
                        if (strtolower($penaltyType) === 'fixed') {
                            $fineAmount = $feeStructure->late_penalty_percentage;
                        } else {
                            $fineAmount = ($baseAmount * $feeStructure->late_penalty_percentage) / 100;
                        }
                    }
                }

                $this->selectedReference = [
                    'id' => $referenceId,
                    'reference_number' => 'Pendente',
                    'entity_code' => config('payments.entity', '90013'),
                    'amount' => $baseAmount + $fineAmount,
                    'fine_amount' => $fineAmount,
                    'discount_amount' => 0,
                    'status' => $isOverdue ? 'overdue' : 'pending',
                    'fee_month' => $monthName,
                    'fee_year' => $year,
                    'expires_at' => $dueDate ? $dueDate->format('d/m/Y') : 'N/A',
                    'created_at' => 'N/A',
                    'description' => $feeStructure->fee_name,
                    'student_name' => $student->name,
                    'student_id' => $student->student_id,
                    'classroom' => ($student->class->class ?? 'N/A') . ' - ' . ($student->class->name ?? 'N/A'),
                    'is_pending' => true,
                    'base_amount' => $baseAmount,
                ];

                $this->showReferenceModal = true;
                return;
            }

            // Verificar se é fee_assign ou PaymentReference
            if (str_starts_with($referenceId, 'fee_')) {
                // É um fee_assign
                $feeAssignId = str_replace('fee_', '', $referenceId);
                $feeAssign = Fee_assign::with('student.classroom')->findOrFail($feeAssignId);

                // Buscar fee structures para descrição
                $studentClassroom = $feeAssign->student->class->class ?? $feeAssign->student->classroom->class ?? null;
                $feeStructures = collect();
                if ($studentClassroom) {
                    $feeStructures = FeeStructure::where('active', 1)
                        ->whereRaw("FIND_IN_SET(?, grades)", [$studentClassroom])
                        ->get();
                }

                $feeNames = $feeStructures->filter(function($s) use ($feeAssign) {
                    return empty($s->months) || in_array($feeAssign->month, explode(',', $s->months));
                })->pluck('name')->implode(', ') ?: 'Pagamento Direto';

                $this->selectedReference = [
                    'id' => 'fee_' . $feeAssign->id,
                    'reference_number' => 'N/A',
                    'entity_code' => 'N/A',
                    'amount' => $feeAssign->amount,
                    'fine_amount' => $feeAssign->fine ?? 0,
                    'discount_amount' => $feeAssign->discount ?? 0,
                    'status' => 'paid',
                    'fee_month' => $feeAssign->month,
                    'fee_year' => $feeAssign->year,
                    'expires_at' => null,
                    'created_at' => $feeAssign->created_at->format('d/m/Y H:i'),
                    'description' => $feeNames,
                    'student_name' => $feeAssign->student->name,
                    'student_id' => $feeAssign->student->student_id,
                    'classroom' => ($feeAssign->student->class->class ?? 'N/A') . ' - ' . ($feeAssign->student->class->name ?? 'N/A'),
                ];
            } else {
                // É PaymentReference
                $reference = PaymentReference::with('student.classroom')->findOrFail($referenceId);

                // Verificar se o estudante logado é dono da referência
                if ($reference->student_id != $this->studentId) {
                    throw new \Exception('Acesso negado');
                }

                $description = is_array($reference->metadata ?? null)
                    ? ($reference->metadata['description'] ?? $reference->metadata['custom_fee_type'] ?? 'N/A')
                    : 'N/A';

                $this->selectedReference = [
                    'id' => $reference->id,
                    'reference_number' => $reference->reference_number,
                    'entity_code' => $reference->entity_code,
                    'amount' => $reference->amount,
                    'fine_amount' => $reference->fine_amount ?? 0,
                    'discount_amount' => $reference->discount_amount ?? 0,
                    'status' => $reference->status,
                    'fee_month' => $reference->fee_month,
                    'fee_year' => $reference->fee_year,
                    'expires_at' => $reference->expires_at ? $reference->expires_at->format('d/m/Y H:i') : 'N/A',
                    'created_at' => $reference->created_at->format('d/m/Y H:i'),
                    'description' => $description,
                    'student_name' => $reference->student->name,
                    'student_id' => $reference->student->student_id,
                    'classroom' => ($reference->student->class->class ?? 'N/A') . ' - ' . ($reference->student->class->name ?? 'N/A'),
                ];
            }

            $this->showReferenceModal = true;
        } catch (\Exception $e) {
            Log::error('Erro ao visualizar referência', [
                'reference_id' => $referenceId,
                'error' => $e->getMessage()
            ]);

            $this->dispatchBrowserEvent('error', [
                'message' => 'Erro ao carregar detalhes da referência'
            ]);
        }
    }

    public function closeReferenceModal()
    {
        $this->showReferenceModal = false;
        $this->selectedReference = null;
    }

    /**
     * Buscar todas as referências de pagamento, fee_assigns e mensalidades pendentes
     */
    private function getAllPaymentReferencesAndFees($year)
    {
        // 1. Buscar todas as referências de pagamento do estudante (aprovadas e pendentes de aprovação)
        $paymentReferences = PaymentReference::where('student_id', $this->studentId)
            ->where('fee_year', $year)
            ->whereIn('approval_status', ['approved', 'pending_approval']) // Estudantes veem suas próprias referências
            ->get();

        // 2. Buscar fee_assigns que não têm referência de pagamento
        $feeAssigns = Fee_assign::where('student_id', $this->studentId)
            ->where('year', $year)
            ->whereDoesntHave('paymentReference')
            ->get();

        // 3. Buscar fee structures para obter nomes das taxas
        $student = $this->selectedStudent;
        $studentClassroom = $student->class->class ?? $student->classroom->class ?? null;

        $feeStructures = collect();
        if ($studentClassroom) {
            $feeStructures = FeeStructure::where('active', 1)
                ->whereRaw("FIND_IN_SET(?, grades)", [$studentClassroom])
                ->get();
        }

        // 4. Converter fee_assigns para o formato de PaymentReference
        $convertedFees = $feeAssigns->map(function($fee) use ($feeStructures) {
            // Buscar nomes das taxas aplicáveis ao mês
            $feeNames = $feeStructures->filter(function($s) use ($fee) {
                return empty($s->months) || in_array($fee->month, explode(',', $s->months));
            })->pluck('name')->implode(', ') ?: 'Pagamento Direto';

            $fakeReference = new \stdClass();
            $fakeReference->id = 'fee_' . $fee->id;
            $fakeReference->reference_number = 'N/A';
            $fakeReference->amount = $fee->amount;
            $fakeReference->fine_amount = $fee->fine ?? 0;
            $fakeReference->discount_amount = $fee->discount ?? 0;
            $fakeReference->status = 'paid';
            $fakeReference->approval_status = 'approved';
            $fakeReference->fee_month = $fee->month;
            $fakeReference->fee_year = $fee->year;
            $fakeReference->expires_at = null;
            $fakeReference->created_at = $fee->created_at;
            $fakeReference->metadata = ['description' => $feeNames, 'is_fee_assign' => true];

            return $fakeReference;
        });

        // 5. NOVO: Adicionar mensalidades pendentes das FeeStructures (cada taxa separada)
        $pendingFromStructures = collect();

        // Identificar fee_structure_ids que JÁ têm referência para cada mês
        // Formato: "feeStructureId_month" => true
        $coveredFeeStructures = [];

        foreach ($paymentReferences as $ref) {
            $metadata = $ref->metadata;
            if (is_array($metadata) && isset($metadata['fee_structure_id'])) {
                $key = $metadata['fee_structure_id'] . '_' . $ref->fee_month;
                $coveredFeeStructures[$key] = true;
            }
        }

        // Meses com fee_assign (pagamentos diretos) - esses cobrem todas as taxas do mês
        $monthsWithFeeAssign = $feeAssigns->pluck('month')->unique()->toArray();

        // Buscar FeeStructures aplicáveis ao estudante
        foreach ($feeStructures as $feeStructure) {
            // Verificar quais meses esta taxa se aplica
            $applicableMonths = !empty($feeStructure->months)
                ? explode(',', $feeStructure->months)
                : ['Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho',
                   'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];

            foreach ($applicableMonths as $monthName) {
                $monthName = trim($monthName);

                // Se já existe fee_assign para este mês (pagamento direto), não adicionar pendente
                if (in_array($monthName, $monthsWithFeeAssign)) {
                    continue;
                }

                // Verificar se já existe referência para esta taxa específica neste mês
                $coverKey = $feeStructure->id . '_' . $monthName;
                if (isset($coveredFeeStructures[$coverKey])) {
                    continue;
                }

                // Calcular multa para esta taxa específica
                $baseAmount = $feeStructure->monthly_fee;
                $fineAmount = 0;
                $isOverdue = false;
                $dueDate = null;

                // Calcular data de vencimento
                $monthNumber = $this->getMonthNumber($monthName);
                if ($monthNumber) {
                    $dueDay = $feeStructure->payment_due_day ?? 5;
                    try {
                        // Obter último dia do mês para evitar datas inválidas
                        $lastDayOfMonth = Carbon::createFromDate($year, $monthNumber, 1)->endOfMonth()->day;
                        $dueDate = Carbon::createFromDate($year, $monthNumber, min($dueDay, $lastDayOfMonth));
                        $isOverdue = $dueDate->isPast();

                        // Calcular multa se estiver em atraso
                        if ($isOverdue && ($feeStructure->late_penalty_percentage ?? 0) > 0) {
                            $penaltyType = $feeStructure->penalty_type ?? 'percentage';
                            if (strtolower($penaltyType) === 'fixed') {
                                $fineAmount = $feeStructure->late_penalty_percentage;
                            } else {
                                $fineAmount = ($baseAmount * $feeStructure->late_penalty_percentage) / 100;
                            }
                        }
                    } catch (\Exception $e) {
                        $dueDate = null;
                    }
                }

                $pendingRef = new \stdClass();
                $pendingRef->id = 'pending_' . $feeStructure->id . '_' . $monthName . '_' . $year;
                $pendingRef->reference_number = 'Pendente';
                $pendingRef->amount = $baseAmount + $fineAmount;
                $pendingRef->fine_amount = $fineAmount;
                $pendingRef->discount_amount = 0;
                $pendingRef->status = $isOverdue ? 'overdue' : 'pending';
                $pendingRef->approval_status = 'pending';
                $pendingRef->fee_month = $monthName;
                $pendingRef->fee_year = $year;
                $pendingRef->expires_at = $dueDate;
                $pendingRef->created_at = Carbon::now();
                $pendingRef->metadata = [
                    'description' => $feeStructure->fee_name,
                    'is_pending_structure' => true,
                    'fee_structure_id' => $feeStructure->id,
                    'base_amount' => $baseAmount,
                    'fine_amount' => $fineAmount,
                    'due_date' => $dueDate
                ];

                $pendingFromStructures->push($pendingRef);
            }
        }

        // 6. Combinar tudo
        return $paymentReferences->concat($convertedFees)->concat($pendingFromStructures);
    }

    public function render()
    {
        $currentYear = $this->selectedYear ?? date('Y');
        $student = $this->selectedStudent;

        // USA SERVICE UNIFICADO para calcular resumo do ano
        $yearSummary = $this->feeCalculationService->calculateYearSummary($student, $currentYear);

        $fees = Fee_assign::where('student_id', $this->studentId)->get();

        // Buscar histórico completo (referências + pagamentos diretos)
        $payment_references = $this->getAllPaymentReferencesAndFees($currentYear);

        return view('livewire.user.fee.fee-collection', [
            'student' => $student,
            'fees' => $fees,
            'feestructures' => collect(), // Não usado mais diretamente
            'monthlyData' => $yearSummary['months_detail'],
            'totalExpected' => $yearSummary['total_expected'],
            'totalPaid' => $yearSummary['total_paid'],
            'totalPending' => $yearSummary['total_pending'],
            'totalOverdue' => $yearSummary['total_overdue'],
            'totalFines' => $yearSummary['total_fines'],
            'currentYear' => $currentYear,
            'payment_references' => $payment_references,
        ]);
    }

    public function openModal($feeAssignId, $amount, $month, $year)
    {
        $this->selectedFeeAssignId = $feeAssignId;
        $this->month = (string) $month;
        $this->year = (int) $year;
        
        $this->dispatchBrowserEvent('openModal');
        $this->emit('openModal', $feeAssignId, $amount, $month, $year);
    }
}
