<?php

namespace App\Models;

use App\Traits\Uuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class Certificate extends Model
{
    use HasFactory, Uuids, SoftDeletes;

    public $incrementing = false;
    protected $keyType = 'string';

    protected $fillable = [
        'student_id',
        'classroom_id',
        'certificate_number',
        'verification_code',
        'verification_hash',
        'academic_year',
        'class_level',
        'turma_name',
        'final_average',
        'subject_grades',
        'student_name',
        'student_id_number',
        'student_dob',
        'student_sex',
        'status',
        'issued_at',
        'revoked_at',
        'revocation_reason',
        'issued_by',
        'revoked_by',
        'pdf_path',
    ];

    protected $casts = [
        'subject_grades' => 'array',
        'issued_at' => 'datetime',
        'revoked_at' => 'datetime',
        'student_dob' => 'date',
        'final_average' => 'decimal:2',
        'academic_year' => 'integer',
    ];

    /**
     * Relationships
     */
    public function student(): BelongsTo
    {
        return $this->belongsTo(User::class, 'student_id');
    }

    public function classroom(): BelongsTo
    {
        return $this->belongsTo(Classroom::class, 'classroom_id');
    }

    public function issuedBy(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'issued_by');
    }

    public function revokedBy(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'revoked_by');
    }

    /**
     * Scopes
     */
    public function scopeIssued($query)
    {
        return $query->where('status', 'issued');
    }

    public function scopeRevoked($query)
    {
        return $query->where('status', 'revoked');
    }

    public function scopeForYear($query, $year)
    {
        return $query->where('academic_year', $year);
    }

    public function scopeForClassroom($query, $classroomId)
    {
        return $query->where('classroom_id', $classroomId);
    }

    /**
     * Helpers
     */
    public function isValid(): bool
    {
        return $this->status === 'issued' && is_null($this->revoked_at);
    }

    public function isRevoked(): bool
    {
        return $this->status === 'revoked';
    }

    public function getVerificationUrl(): string
    {
        return route('certificate.verify', ['code' => $this->verification_code]);
    }

    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            'issued' => '<span class="badge bg-success">Emitido</span>',
            'revoked' => '<span class="badge bg-danger">Revogado</span>',
            'draft' => '<span class="badge bg-warning">Rascunho</span>',
            default => '<span class="badge bg-secondary">Desconhecido</span>',
        };
    }

    public function getFormattedFinalAverageAttribute(): string
    {
        return number_format($this->final_average, 2, ',', '.');
    }
}
