<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class DocumentScan extends Model
{
    use HasFactory, SoftDeletes;

    protected $keyType = 'string';
    public $incrementing = false;

    protected $fillable = [
        'student_id',
        'student_code',
        'image_path',
        'image_filename',
        'image_size',
        'document_side',
        'back_image_path',
        'back_image_filename',
        'back_image_size',
        'back_raw_ocr_text',
        'back_ocr_data',
        'status',
        'error_message',
        'raw_ocr_text',
        'ocr_data',
        'document_id',
        'document_type',
        'full_name',
        'dob',
        'father_name',
        'mother_name',
        'birth_place',
        'birth_district',
        'birth_province',
        'sex',
        'confidence_score',
        'uploaded_by',
        'confirmed_by',
        'confirmed_at',
        'requires_review',
    ];

    protected $casts = [
        'ocr_data' => 'array',
        'back_ocr_data' => 'array',
        'dob' => 'date',
        'confirmed_at' => 'datetime',
        'requires_review' => 'boolean',
        'confidence_score' => 'integer',
        'image_size' => 'integer',
        'back_image_size' => 'integer',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->id)) {
                $model->id = (string) Str::uuid();
            }
        });
    }

    // Relacionamentos
    public function student()
    {
        return $this->belongsTo(User::class, 'student_id');
    }

    public function uploader()
    {
        return $this->belongsTo(User::class, 'uploaded_by');
    }

    public function confirmer()
    {
        return $this->belongsTo(User::class, 'confirmed_by');
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeProcessed($query)
    {
        return $query->where('status', 'processed');
    }

    public function scopeRequiresReview($query)
    {
        return $query->where('requires_review', true);
    }

    // Métodos auxiliares
    public function markAsProcessed()
    {
        $this->update(['status' => 'processed']);
    }

    public function markAsConfirmed($userId = null)
    {
        $this->update([
            'status' => 'confirmed',
            'confirmed_by' => $userId ?? auth()->id(),
            'confirmed_at' => now(),
            'requires_review' => false,
        ]);
    }

    public function markAsFailed($errorMessage)
    {
        $this->update([
            'status' => 'failed',
            'error_message' => $errorMessage,
        ]);
    }
}
