<?php

namespace App\Models;

use App\Traits\Uuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Cache;

class ExamConfiguration extends Model
{
    use HasFactory, Uuids, SoftDeletes;

    public $incrementing = false;
    protected $keyType = 'string';

    protected $fillable = [
        'subject_id',
        'class_level',
        'exam_type',
        'is_active',
        'academic_year',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'class_level' => 'integer',
        'academic_year' => 'integer'
    ];

    // ========== RELATIONSHIPS ==========

    public function subject(): BelongsTo
    {
        return $this->belongsTo(Subject::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    public function updater(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'updated_by');
    }

    // ========== STATIC METHODS (HELPERS) ==========

    /**
     * Verifica se uma disciplina tem exame em uma classe específica
     *
     * @param string $subjectId ID da disciplina
     * @param int $classLevel Nível da classe (1-12)
     * @param string $examType Tipo de exame ('NE', 'AF', ou null para qualquer)
     * @param int|null $academicYear Ano letivo (opcional)
     * @return bool
     */
    public static function hasExam(
        string $subjectId,
        int $classLevel,
        ?string $examType = null,
        ?int $academicYear = null
    ): bool {
        $cacheKey = "exam_config_{$subjectId}_{$classLevel}_{$examType}_{$academicYear}";

        return Cache::remember($cacheKey, 3600, function () use ($subjectId, $classLevel, $examType, $academicYear) {
            $query = self::where('subject_id', $subjectId)
                ->where('class_level', $classLevel)
                ->where('is_active', true);

            if ($academicYear) {
                $query->where('academic_year', $academicYear);
            }

            if ($examType) {
                $query->where('exam_type', $examType);
            } else {
                $query->whereIn('exam_type', ['NE', 'AF']);
            }

            return $query->exists();
        });
    }

    /**
     * Obtém todas as disciplinas com exame para uma classe específica
     *
     * @param int $classLevel Nível da classe (1-12)
     * @param string|null $examType Tipo de exame ('NE', 'AF', ou null para ambos)
     * @param int|null $academicYear Ano letivo (opcional)
     * @return array Array de nomes de disciplinas
     */
    public static function getSubjectsWithExam(
        int $classLevel,
        ?string $examType = null,
        ?int $academicYear = null
    ): array {
        $cacheKey = "subjects_with_exam_{$classLevel}_{$examType}_{$academicYear}";

        return Cache::remember($cacheKey, 3600, function () use ($classLevel, $examType, $academicYear) {
            $query = self::where('class_level', $classLevel)
                ->where('is_active', true)
                ->with('subject');

            if ($academicYear) {
                $query->where('academic_year', $academicYear);
            }

            if ($examType) {
                $query->where('exam_type', $examType);
            } else {
                $query->whereIn('exam_type', ['NE', 'AF']);
            }

            return $query->get()
                ->pluck('subject.name')
                ->filter()
                ->values()
                ->toArray();
        });
    }

    /**
     * Obtém o tipo de exame para uma disciplina e classe
     *
     * @param string $subjectId ID da disciplina
     * @param int $classLevel Nível da classe (1-12)
     * @param int|null $academicYear Ano letivo (opcional)
     * @return string|null 'NE', 'AF' ou null
     */
    public static function getExamType(
        string $subjectId,
        int $classLevel,
        ?int $academicYear = null
    ): ?string {
        $config = self::where('subject_id', $subjectId)
            ->where('class_level', $classLevel)
            ->where('is_active', true);

        if ($academicYear) {
            $config->where('academic_year', $academicYear);
        }

        $config = $config->first();

        return $config && $config->exam_type !== 'none' ? $config->exam_type : null;
    }

    /**
     * Obtém todas as configurações para uma classe
     *
     * @param int $classLevel Nível da classe (1-12)
     * @param int|null $academicYear Ano letivo (opcional)
     * @return \Illuminate\Support\Collection
     */
    public static function getConfigurationsForClass(
        int $classLevel,
        ?int $academicYear = null
    ) {
        $query = self::where('class_level', $classLevel)
            ->with('subject');

        if ($academicYear) {
            $query->where('academic_year', $academicYear);
        }

        return $query->get();
    }

    /**
     * Limpa o cache de configurações
     */
    public static function clearCache(): void
    {
        Cache::flush(); // Ou usar tags se disponível
    }

    // ========== MODEL EVENTS ==========

    protected static function boot()
    {
        parent::boot();

        // Gerar UUID automaticamente ao criar
        static::creating(function ($model) {
            if (empty($model->{$model->getKeyName()})) {
                $model->{$model->getKeyName()} = (string) \Webpatser\Uuid\Uuid::generate(4);
            }
        });

        // Limpar cache quando houver mudanças
        static::saved(function () {
            self::clearCache();
        });

        static::deleted(function () {
            self::clearCache();
        });

        // Validações
        static::saving(function ($model) {
            if ($model->class_level < 1 || $model->class_level > 12) {
                throw new \Exception('Classe deve estar entre 1 e 12');
            }

            if (!in_array($model->exam_type, ['NE', 'AF', 'none'])) {
                throw new \Exception('Tipo de exame inválido');
            }
        });
    }
}
