<?php

namespace App\Models;

use App\Traits\Uuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Passport\HasApiTokens;

class Guardian extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, Uuids, SoftDeletes;

    public $incrementing = false;
    protected $keyType = 'string';

    protected $fillable = [
        'name',
        'phone',
        'email',
        'relationship',
        'password',
        'is_active',
        'last_login_at',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'is_active'     => 'boolean',
        'last_login_at' => 'datetime',
    ];

    // Etiqueta traduzida da relação
    public function getRelationshipLabelAttribute(): string
    {
        return match($this->relationship) {
            'pai'          => 'Pai',
            'mae'          => 'Mãe',
            'encarregado'  => 'Encarregado',
            'outro'        => 'Outro',
            default        => ucfirst($this->relationship ?? 'Encarregado'),
        };
    }

    // Estudantes ligados a este encarregado
    public function students(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'guardian_student', 'guardian_id', 'student_id')
            ->withPivot(['relationship', 'is_primary'])
            ->withTimestamps();
    }

    // Device tokens para push notifications (FCM)
    public function deviceTokens()
    {
        return $this->hasMany(GuardianDeviceToken::class);
    }

    // Verifica se o encarregado tem acesso a um estudante específico
    public function hasStudent(string $studentId): bool
    {
        return $this->students()->where('users.id', $studentId)->exists();
    }
}
