<?php

namespace App\Models\POS;

use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Product extends Model
{
    use HasFactory, HasUuids, SoftDeletes;

    protected $table = 'pos_products';

    protected $fillable = [
        'category_id',
        'name',
        'sku',
        'description',
        'image',
        'cost_price',
        'selling_price',
        'discount_price',
        'quantity',
        'min_stock',
        'unit',
        'variant_type',
        'variant_options',
        'is_active',
        'track_stock',
        'notes',
    ];

    protected $casts = [
        'cost_price' => 'decimal:2',
        'selling_price' => 'decimal:2',
        'discount_price' => 'decimal:2',
        'quantity' => 'integer',
        'min_stock' => 'integer',
        'variant_options' => 'array',
        'is_active' => 'boolean',
        'track_stock' => 'boolean',
    ];

    // Relationships
    public function category()
    {
        return $this->belongsTo(Category::class, 'category_id');
    }

    public function stockMovements()
    {
        return $this->hasMany(StockMovement::class, 'product_id');
    }

    public function saleItems()
    {
        return $this->hasMany(SaleItem::class, 'product_id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeLowStock($query)
    {
        return $query->whereColumn('quantity', '<=', 'min_stock');
    }

    public function scopeOutOfStock($query)
    {
        return $query->where('quantity', '<=', 0);
    }

    public function scopeInStock($query)
    {
        return $query->where('quantity', '>', 0);
    }

    // Accessors & Mutators
    public function getEffectivePriceAttribute()
    {
        return $this->discount_price ?? $this->selling_price;
    }

    public function getProfitMarginAttribute()
    {
        if ($this->cost_price > 0) {
            return (($this->effective_price - $this->cost_price) / $this->cost_price) * 100;
        }
        return 0;
    }

    public function getIsLowStockAttribute()
    {
        return $this->quantity <= $this->min_stock;
    }

    public function getIsOutOfStockAttribute()
    {
        return $this->quantity <= 0;
    }

    public function getStockStatusAttribute()
    {
        if ($this->is_out_of_stock) {
            return 'out_of_stock';
        } elseif ($this->is_low_stock) {
            return 'low_stock';
        }
        return 'in_stock';
    }

    public function getStockStatusColorAttribute()
    {
        return match($this->stock_status) {
            'out_of_stock' => 'red',
            'low_stock' => 'orange',
            'in_stock' => 'green',
        };
    }

    // Methods
    public function adjustStock($quantity, $type = 'adjustment', $userId = null, $notes = null)
    {
        $stockBefore = $this->quantity;
        $this->quantity += $quantity;
        $this->save();

        StockMovement::create([
            'product_id' => $this->id,
            'user_id' => $userId ?? auth()->id(),
            'type' => $type,
            'quantity' => abs($quantity),
            'stock_before' => $stockBefore,
            'stock_after' => $this->quantity,
            'notes' => $notes,
            'movement_date' => now(),
        ]);

        return $this;
    }
}
