<?php

namespace App\Models\POS;

use App\Models\User;
use App\Models\Admin;
use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Sale extends Model
{
    use HasFactory, HasUuids, SoftDeletes;

    protected $table = 'pos_sales';

    protected $fillable = [
        'sale_number',
        'user_id',
        'cashier_id',
        'subtotal',
        'discount',
        'tax',
        'total',
        'payment_method',
        'status',
        'notes',
        'sale_date',
    ];

    protected $casts = [
        'subtotal' => 'decimal:2',
        'discount' => 'decimal:2',
        'tax' => 'decimal:2',
        'total' => 'decimal:2',
        'sale_date' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($sale) {
            if (empty($sale->sale_number)) {
                $sale->sale_number = self::generateSaleNumber();
            }
            if (empty($sale->sale_date)) {
                $sale->sale_date = now();
            }
        });
    }

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function student()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function cashier()
    {
        return $this->belongsTo(Admin::class, 'cashier_id');
    }

    public function items()
    {
        return $this->hasMany(SaleItem::class, 'sale_id');
    }

    // Scopes
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeCancelled($query)
    {
        return $query->where('status', 'cancelled');
    }

    public function scopeToday($query)
    {
        return $query->whereDate('sale_date', today());
    }

    public function scopeThisMonth($query)
    {
        return $query->whereYear('sale_date', now()->year)
            ->whereMonth('sale_date', now()->month);
    }

    // Accessors
    public function getPaymentMethodNameAttribute()
    {
        return match($this->payment_method) {
            'cash' => 'Dinheiro',
            'mpesa' => 'M-Pesa',
            'emola' => 'e-Mola',
            'bank_transfer' => 'Transferência Bancária',
            'credit' => 'Crédito',
        };
    }

    public function getStatusNameAttribute()
    {
        return match($this->status) {
            'completed' => 'Concluída',
            'pending' => 'Pendente',
            'cancelled' => 'Cancelada',
        };
    }

    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'completed' => 'green',
            'pending' => 'orange',
            'cancelled' => 'red',
        };
    }

    // Methods
    public static function generateSaleNumber()
    {
        $prefix = \App\Models\POS\Setting::get('sale_number_prefix', 'POS');
        $year = now()->year;
        $lastSale = self::whereYear('created_at', $year)
            ->orderBy('created_at', 'desc')
            ->first();

        // Match any prefix (POS, FAT, REC, VD, etc.)
        if ($lastSale && preg_match('/[A-Z]+-\d{4}-(\d{5})/', $lastSale->sale_number, $matches)) {
            $number = intval($matches[1]) + 1;
        } else {
            $number = 1;
        }

        return sprintf('%s-%s-%05d', $prefix, $year, $number);
    }

    public function calculateTotals()
    {
        $this->subtotal = $this->items->sum(function ($item) {
            return $item->quantity * $item->unit_price;
        });

        $this->total = $this->subtotal - $this->discount + $this->tax;
        $this->save();

        return $this;
    }
}
