<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class PermissionGroup extends Model
{
    protected $fillable = [
        'key',
        'label',
        'sort_order',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Get permissions in this group
     */
    public function permissions()
    {
        return $this->hasMany(PermissionDefinition::class, 'group_id')
            ->orderBy('sort_order')
            ->orderBy('label');
    }

    /**
     * Get active permissions only
     */
    public function activePermissions()
    {
        return $this->permissions()->where('is_active', true);
    }

    /**
     * Get all groups with their permissions (cached)
     */
    public static function getAllWithPermissions(): array
    {
        return Cache::remember('permission_groups_all', 3600, function () {
            return self::where('is_active', true)
                ->orderBy('sort_order')
                ->orderBy('label')
                ->with(['activePermissions'])
                ->get()
                ->toArray();
        });
    }

    /**
     * Clear permission cache
     */
    public static function clearCache(): void
    {
        Cache::forget('permission_groups_all');
        Cache::forget('available_permissions');
    }

    /**
     * Seed default permission groups
     */
    public static function seedDefaults(): void
    {
        $groups = [
            ['key' => 'pos', 'label' => 'Sistema POS', 'sort_order' => 1],
            ['key' => 'students', 'label' => 'Estudantes', 'sort_order' => 2],
            ['key' => 'teachers', 'label' => 'Professores', 'sort_order' => 3],
            ['key' => 'classes', 'label' => 'Turmas', 'sort_order' => 4],
            ['key' => 'exams', 'label' => 'Notas e Exames', 'sort_order' => 5],
            ['key' => 'payments', 'label' => 'Pagamentos', 'sort_order' => 6],
            ['key' => 'library', 'label' => 'Biblioteca Digital', 'sort_order' => 7],
            ['key' => 'system', 'label' => 'Sistema', 'sort_order' => 8],
            ['key' => 'sms', 'label' => 'SMS', 'sort_order' => 9],
            ['key' => 'certificates', 'label' => 'Certificados', 'sort_order' => 10],
            ['key' => 'student_cards', 'label' => 'Cartões de Estudante', 'sort_order' => 11],
            ['key' => 'attendance', 'label' => 'Presença', 'sort_order' => 12],
            ['key' => 'secretary', 'label' => 'Secretaria', 'sort_order' => 13],
            ['key' => 'renewals', 'label' => 'Renovações de Matrícula', 'sort_order' => 14],
            ['key' => 'communications', 'label' => 'Comunicados e Notificações', 'sort_order' => 15],
            ['key' => 'documents', 'label' => 'Documentos e Matrizes', 'sort_order' => 16],
        ];

        foreach ($groups as $group) {
            self::updateOrCreate(['key' => $group['key']], $group);
        }

        self::clearCache();
    }
}
