<?php

namespace App\Services;

use App\Models\Certificate;
use App\Models\CertificateBatch;
use App\Models\DocumentClassConfiguration;
use App\Models\User;
use App\Models\Classroom;
use App\Models\Exam;
use App\Models\ExamConfiguration;
use App\Models\TeacherSubject;
use App\Models\Subject;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Carbon\Carbon;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use ZipArchive;
use PhpOffice\PhpWord\TemplateProcessor;

class CertificateService
{
    protected FeeCalculationService $feeService;

    public function __construct(FeeCalculationService $feeService)
    {
        $this->feeService = $feeService;
    }

    /**
     * Get configured class levels from database
     */
    public static function getConfiguredClasses(?int $academicYear = null): array
    {
        return DocumentClassConfiguration::getConfiguredClasses($academicYear);
    }

    /**
     * Get document configuration for a class level
     */
    public static function getDocumentConfig(int $classLevel, ?int $academicYear = null): ?DocumentClassConfiguration
    {
        return DocumentClassConfiguration::getForClass($classLevel, $academicYear);
    }

    /**
     * Check if student is eligible for certificate/declaration (based on configured classes + payments cleared)
     * @param User $student
     * @param int $academicYear
     * @param string|null $contextClassroomId - Turma específica para verificar (quando filtramos por turma)
     */
    public function isEligible(User $student, int $academicYear, ?string $contextClassroomId = null): array
    {
        // Se foi passado um contexto de turma específico, usar essa turma directamente
        // Isso acontece quando estamos a filtrar estudantes por turma específica
        if ($contextClassroomId) {
            $classroom = Classroom::find($contextClassroomId);
        } else {
            // IMPORTANTE: Buscar a turma onde o estudante fez exames no ano especificado
            // Isso é necessário porque estudantes podem ter sido transferidos para outra turma
            $examClassroomId = Exam::where('student_id', $student->id)
                ->where('year', $academicYear)
                ->whereNotNull('classroom_id')
                ->value('classroom_id');

            // Se encontrou exames, usar a turma dos exames; senão, usar a turma atual
            $classroom = $examClassroomId
                ? Classroom::find($examClassroomId)
                : $student->classroom;
        }

        if (!$classroom) {
            $classroom = $student->classroom;
        }

        // Check if student's class is configured for document generation
        $configuredClasses = self::getConfiguredClasses($academicYear);
        if (!$classroom || !in_array((int) $classroom->class, $configuredClasses)) {
            return [
                'eligible' => false,
                'reason' => 'Classe do estudante nao esta configurada para emissao de documentos',
                'grades' => null,
                'payment_status' => null,
                'document_config' => null,
            ];
        }

        // Get document configuration for this class
        $docConfig = self::getDocumentConfig((int) $classroom->class, $academicYear);
        if (!$docConfig) {
            return [
                'eligible' => false,
                'reason' => 'Configuracao de documento nao encontrada para esta classe',
                'grades' => null,
                'payment_status' => null,
                'document_config' => null,
            ];
        }

        // Calculate grades - passar o ID da turma para usar o mesmo contexto
        $gradesData = $this->calculateStudentFinalGrades($student, $academicYear, $classroom->id);

        // Check if passed (MFD >= 10)
        if ($gradesData['final_average'] < 10) {
            return [
                'eligible' => false,
                'reason' => 'Media final inferior a 10 (MF: ' . number_format($gradesData['final_average'], 2) . ')',
                'grades' => $gradesData,
                'payment_status' => null,
                'document_config' => $docConfig,
            ];
        }

        // Check Portuguese and Math rule
        $portugues = $gradesData['subjects']['Portugues']['final'] ?? $gradesData['subjects']['Português']['final'] ?? 0;
        $matematica = $gradesData['subjects']['Matematica']['final'] ?? $gradesData['subjects']['Matemática']['final'] ?? 0;

        if ($portugues < 10 && $matematica < 10) {
            return [
                'eligible' => false,
                'reason' => 'Portugues (' . $portugues . ') e Matematica (' . $matematica . ') ambos com nota inferior a 10',
                'grades' => $gradesData,
                'payment_status' => null,
                'document_config' => $docConfig,
            ];
        }

        // Check payments
        $paymentSummary = $this->feeService->calculateYearSummary($student, $academicYear);
        $totalOwing = $paymentSummary['total_pending'] + $paymentSummary['total_overdue'];

        if ($totalOwing > 0) {
            return [
                'eligible' => false,
                'reason' => 'Pagamentos pendentes: ' . number_format($totalOwing, 2, ',', '.') . ' MZN',
                'grades' => $gradesData,
                'payment_status' => $paymentSummary,
                'document_config' => $docConfig,
            ];
        }

        return [
            'eligible' => true,
            'reason' => 'APROVADO E PAGAMENTOS QUITADOS',
            'grades' => $gradesData,
            'payment_status' => $paymentSummary,
            'document_config' => $docConfig,
        ];
    }

    /**
     * Calculate final grades for a student (all subjects)
     * Optimized to reduce database queries
     * @param User $student
     * @param int|null $academicYear
     * @param string|null $contextClassroomId - Turma específica para calcular notas (quando filtramos por turma)
     */
    public function calculateStudentFinalGrades(User $student, ?int $academicYear = null, ?string $contextClassroomId = null): array
    {
        $year = $academicYear ?? (int) date('Y');

        // Se foi passado um contexto de turma específico, usar essa turma directamente
        if ($contextClassroomId) {
            $classroom = Classroom::find($contextClassroomId);
        } else {
            // IMPORTANTE: Buscar a turma onde o estudante fez exames no ano especificado
            // Isso é necessário porque estudantes podem ter sido transferidos para outra turma
            $examClassroomId = Exam::where('student_id', $student->id)
                ->where('year', $year)
                ->whereNotNull('classroom_id')
                ->value('classroom_id');

            // Se encontrou exames, usar a turma dos exames; senão, usar a turma atual
            $classroom = $examClassroomId
                ? Classroom::find($examClassroomId)
                : $student->classroom;
        }

        if (!$classroom) {
            $classroom = $student->classroom;
        }

        $classLevel = (int) $classroom->class;

        // Get subjects for this classroom with eager loading (single query)
        $teacherSubjects = TeacherSubject::where('classroom_id', $classroom->id)
            ->with('subject')
            ->get();

        $subjects = [];
        $subjectIds = [];
        foreach ($teacherSubjects as $ts) {
            if ($ts->subject) {
                $subjects[$ts->subject->name] = $ts->subject->id;
                $subjectIds[] = $ts->subject->id;
            }
        }

        if (empty($subjects)) {
            return [
                'subjects' => [],
                'final_average' => 0,
                'subject_count' => 0,
            ];
        }

        // Get NE subjects configuration (cached)
        static $neSubjectsCache = [];
        $cacheKey = "{$classLevel}_{$year}";
        if (!isset($neSubjectsCache[$cacheKey])) {
            $configCount = ExamConfiguration::where('academic_year', $year)->where('is_active', true)->count();
            $useYear = $configCount > 0 ? $year : null;
            $neSubjectsCache[$cacheKey] = ExamConfiguration::getSubjectsWithExam($classLevel, 'NE', $useYear);
        }
        $neSubjects = $neSubjectsCache[$cacheKey];

        // Batch load ALL exams for this student in ONE query
        // Get exams ONLY for the explicitly selected year - no fallback to previous years
        // NOTE: Don't filter by classroom_id because students may have changed classrooms
        // but their old exams are still valid for certificate generation
        $allExams = Exam::where('student_id', $student->id)
            ->where('year', $year)
            ->whereIn('subject_id', $subjectIds)
            ->get()
            ->groupBy(function ($exam) {
                return $exam->subject_id . '_' . $exam->trimester_id;
            });

        $subjectGrades = [];
        $totalFinal = 0;
        $subjectCount = count($subjects);

        foreach ($subjects as $subjectName => $subjectId) {
            // Get trimester data from cached exams
            $trim1 = $this->getExamValues($allExams, $subjectId, 1);
            $trim2 = $this->getExamValues($allExams, $subjectId, 2);
            $trim3 = $this->getExamValues($allExams, $subjectId, 3);

            $nf1 = round(array_sum(array_slice($trim1, 0, 3)) / 3);
            $nf2 = round(array_sum(array_slice($trim2, 0, 3)) / 3);
            $nf3 = round(array_sum(array_slice($trim3, 0, 3)) / 3);
            $mfd = round(($nf1 + $nf2 + $nf3) / 3);
            $ne = $trim3[3]; // NE from 3rd trimester

            // Calculate final based on whether subject has NE
            $hasNE = in_array($subjectName, $neSubjects);
            if ($hasNE && $ne > 0) {
                $final = round(($mfd * 2 + $ne) / 3);
            } else {
                $final = $mfd;
            }

            $subjectGrades[$subjectName] = [
                'nf1' => $nf1,
                'nf2' => $nf2,
                'nf3' => $nf3,
                'mfd' => $mfd,
                'ne' => $hasNE ? $ne : null,
                'final' => $final,
            ];

            $totalFinal += $final;
        }

        $finalAverage = $subjectCount > 0 ? round($totalFinal / $subjectCount, 2) : 0;

        return [
            'subjects' => $subjectGrades,
            'final_average' => $finalAverage,
            'subject_count' => $subjectCount,
        ];
    }

    /**
     * Get exam values from pre-loaded collection
     */
    private function getExamValues($allExams, $subjectId, $trimesterId): array
    {
        $key = $subjectId . '_' . $trimesterId;
        $exam = $allExams->get($key)?->first();

        return [
            (float) ($exam->ACS1a ?? 0),
            (float) ($exam->ACS2a ?? 0),
            (float) ($exam->AT ?? 0),
            (float) ($exam->NE ?? 0),
        ];
    }

    /**
     * Get trimester values for a student/subject
     */
    private function getTrimesterValues($studentId, $classroomId, $subjectId, $trimesterId): array
    {
        $exam = Exam::where([
            ['trimester_id', $trimesterId],
            ['classroom_id', $classroomId],
            ['subject_id', $subjectId],
            ['student_id', $studentId],
        ])->first();

        return [
            (float) ($exam->ACS1a ?? 0),
            (float) ($exam->ACS2a ?? 0),
            (float) ($exam->AT ?? 0),
            (float) ($exam->NE ?? 0),
        ];
    }

    /**
     * Generate document number based on type (CERT for certificate, DECL for declaration)
     */
    public function generateDocumentNumber(string $documentType = 'certificate'): string
    {
        $year = date('Y');
        $prefix = $documentType === DocumentClassConfiguration::TYPE_CERTIFICATE ? 'CERT' : 'DECL';

        $lastDoc = Certificate::whereYear('created_at', $year)
            ->where('certificate_number', 'LIKE', $prefix . '-%')
            ->orderBy('certificate_number', 'desc')
            ->first();

        if ($lastDoc) {
            $lastNum = (int) substr($lastDoc->certificate_number, -6);
            $nextNum = $lastNum + 1;
        } else {
            $nextNum = 1;
        }

        return $prefix . '-' . $year . '-' . str_pad($nextNum, 6, '0', STR_PAD_LEFT);
    }

    /**
     * @deprecated Use generateDocumentNumber instead
     */
    public function generateCertificateNumber(): string
    {
        return $this->generateDocumentNumber(DocumentClassConfiguration::TYPE_CERTIFICATE);
    }

    /**
     * Generate verification code (for QR and manual verification)
     */
    public function generateVerificationCode(): string
    {
        return Str::random(32);
    }

    /**
     * Generate verification hash (integrity signature)
     */
    public function generateVerificationHash(array $data): string
    {
        $payload = json_encode($data) . config('app.key');
        return hash('sha256', $payload);
    }

    /**
     * Create certificate/declaration for a single student
     */
    public function createCertificate(User $student, int $academicYear, ?string $issuedBy = null): Certificate
    {
        $eligibility = $this->isEligible($student, $academicYear);

        if (!$eligibility['eligible']) {
            throw new \Exception("Estudante nao elegivel: " . $eligibility['reason']);
        }

        // Get document configuration
        $docConfig = $eligibility['document_config'];
        $documentType = $docConfig ? $docConfig->document_type : DocumentClassConfiguration::TYPE_CERTIFICATE;

        // Check if document already exists
        $existing = Certificate::where('student_id', $student->id)
            ->where('academic_year', $academicYear)
            ->where('status', '!=', 'revoked')
            ->first();

        if ($existing) {
            $docLabel = $documentType === DocumentClassConfiguration::TYPE_CERTIFICATE ? 'Certificado' : 'Declaracao';
            throw new \Exception("{$docLabel} ja existe para este estudante neste ano (N: " . $existing->certificate_number . ")");
        }

        $gradesData = $eligibility['grades'];

        // IMPORTANTE: Usar a turma onde o estudante fez exames, não a turma actual
        // (estudante pode ter sido transferido após conclusão do ano lectivo)
        $examClassroomId = Exam::where('student_id', $student->id)
            ->where('year', $academicYear)
            ->whereNotNull('classroom_id')
            ->value('classroom_id');

        $classroom = $examClassroomId
            ? Classroom::find($examClassroomId)
            : $student->classroom;

        if (!$classroom) {
            $classroom = $student->classroom;
        }

        $docNumber = $this->generateDocumentNumber($documentType);
        $verificationCode = $this->generateVerificationCode();
        $issuedAt = now();
        $finalAverage = round((float) $gradesData['final_average'], 2);

        $hashData = [
            'student_id' => (string) $student->id,
            'certificate_number' => $docNumber,
            'academic_year' => (int) $academicYear,
            'final_average' => $finalAverage,
            'issued_at' => $issuedAt->toIso8601String(),
        ];
        $verificationHash = $this->generateVerificationHash($hashData);

        $certificate = Certificate::create([
            'student_id' => $student->id,
            'classroom_id' => $classroom->id,
            'certificate_number' => $docNumber,
            'verification_code' => $verificationCode,
            'verification_hash' => $verificationHash,
            'academic_year' => $academicYear,
            'class_level' => (string) $classroom->class,
            'turma_name' => $classroom->name,
            'final_average' => $finalAverage,
            'subject_grades' => $gradesData['subjects'],
            'student_name' => $student->name,
            'student_id_number' => $student->student_id,
            'student_dob' => $student->dob,
            'student_sex' => $student->sex,
            'status' => 'issued',
            'issued_at' => $issuedAt,
            'issued_by' => $issuedBy,
        ]);

        // Generate PDF using appropriate template
        $pdfPath = $this->generatePDF($certificate, $docConfig);
        $certificate->update(['pdf_path' => $pdfPath]);

        $docLabel = $documentType === DocumentClassConfiguration::TYPE_CERTIFICATE ? 'Certificado' : 'Declaracao';
        Log::info("{$docLabel} gerado", [
            'certificate_id' => $certificate->id,
            'student_id' => $student->id,
            'certificate_number' => $docNumber,
            'document_type' => $documentType,
        ]);

        return $certificate;
    }

    /**
     * Generate PDF document using blade template (includes QR code for verification)
     * Also generates Word document for editing purposes
     */
    public function generatePDF(Certificate $certificate, ?DocumentClassConfiguration $docConfig = null): string
    {
        set_time_limit(120);

        // Ensure student relationship is loaded
        if (!$certificate->relationLoaded('student')) {
            $certificate->load('student');
        }

        // Get document configuration if not provided
        if (!$docConfig) {
            $docConfig = self::getDocumentConfig((int) $certificate->class_level, $certificate->academic_year);
        }

        $documentType = $docConfig ? $docConfig->document_type : DocumentClassConfiguration::TYPE_CERTIFICATE;
        $isDeclaration = $documentType === DocumentClassConfiguration::TYPE_DECLARATION;

        // Generate QR Code for verification
        $verificationUrl = route('certificate.verify', ['code' => $certificate->verification_code]);
        $qrCode = base64_encode(QrCode::format('png')
            ->size(100)
            ->margin(0)
            ->generate($verificationUrl));

        // Check if class has NE exams
        $classLevel = (int) $certificate->class_level;
        $configCount = ExamConfiguration::where('academic_year', $certificate->academic_year)->where('is_active', true)->count();
        $useYear = $configCount > 0 ? $certificate->academic_year : null;
        $neSubjects = ExamConfiguration::getSubjectsWithExam($classLevel, 'NE', $useYear);
        $hasNE = !empty($neSubjects);

        $data = [
            'certificate' => $certificate,
            'qrCode' => $qrCode,
            'verificationUrl' => $verificationUrl,
            'schoolName' => 'COLEGIO POLITECNICO DE MOCAMBIQUE',
            'schoolAddress' => 'Nampula - Mocambique',
            'documentTitle' => $docConfig ? $docConfig->document_title : ($isDeclaration ? 'Declaracao' : 'Certificado'),
            'showGrades' => true,
            'hasNE' => $hasNE,
        ];

        // Select template based on document type
        $template = $isDeclaration ? 'pdf.declaration' : 'pdf.certificate';

        $pdf = Pdf::loadView($template, $data)
            ->setOptions([
                'isRemoteEnabled' => true,
                'isHtml5ParserEnabled' => true,
                'isFontSubsettingEnabled' => true,
            ])
            ->setPaper('a4', 'portrait');

        // File naming based on document type
        $docTypeSlug = $isDeclaration ? 'declaracao' : 'certificado';
        $fileName = Str::slug($certificate->student_name) . "_{$docTypeSlug}_" . $certificate->certificate_number . '.pdf';
        $filePath = "certificates/{$certificate->academic_year}/{$fileName}";

        // Ensure directory exists
        $dir = dirname(Storage::disk('public')->path($filePath));
        if (!file_exists($dir)) {
            mkdir($dir, 0755, true);
        }

        Storage::disk('public')->put($filePath, $pdf->output());

        // Also generate Word document for editing purposes
        try {
            $this->generateWordDocument($certificate, $docConfig);
        } catch (\Exception $e) {
            Log::warning('Word document generation failed', ['error' => $e->getMessage()]);
        }

        return $filePath;
    }

    /**
     * Convert Word to PDF using LibreOffice (if available)
     */
    private function convertWordToPdfWithLibreOffice(string $wordPath, string $pdfPath): bool
    {
        // Check if LibreOffice is available
        $soffice = null;
        foreach (['/usr/bin/soffice', '/usr/bin/libreoffice', '/Applications/LibreOffice.app/Contents/MacOS/soffice'] as $path) {
            if (file_exists($path)) {
                $soffice = $path;
                break;
            }
        }

        // Also try which command
        if (!$soffice) {
            exec('which soffice 2>/dev/null', $output, $returnCode);
            if ($returnCode === 0 && !empty($output[0])) {
                $soffice = $output[0];
            }
        }

        if (!$soffice) {
            Log::debug('LibreOffice não encontrado, usando fallback');
            return false;
        }

        $outputDir = dirname($pdfPath);
        $command = sprintf(
            '%s --headless --convert-to pdf --outdir %s %s 2>&1',
            escapeshellarg($soffice),
            escapeshellarg($outputDir),
            escapeshellarg($wordPath)
        );

        exec($command, $output, $returnCode);

        if ($returnCode !== 0) {
            Log::warning('LibreOffice conversion failed', ['output' => $output, 'return_code' => $returnCode]);
            return false;
        }

        // LibreOffice saves with the same name but .pdf extension
        $generatedPdf = $outputDir . '/' . pathinfo($wordPath, PATHINFO_FILENAME) . '.pdf';

        // Rename if necessary
        if ($generatedPdf !== $pdfPath && file_exists($generatedPdf)) {
            rename($generatedPdf, $pdfPath);
        }

        return file_exists($pdfPath);
    }

    /**
     * Convert Word to PDF using PhpWord HTML writer + DomPDF (fallback)
     */
    private function convertWordToPdfWithPhpWord(string $wordPath, string $pdfPath): bool
    {
        try {
            // Load Word document
            $phpWord = \PhpOffice\PhpWord\IOFactory::load($wordPath);

            // Convert to HTML first
            $htmlWriter = \PhpOffice\PhpWord\IOFactory::createWriter($phpWord, 'HTML');
            $tempHtmlPath = sys_get_temp_dir() . '/word_to_pdf_' . uniqid() . '.html';
            $htmlWriter->save($tempHtmlPath);

            // Read HTML content
            $htmlContent = file_get_contents($tempHtmlPath);

            // Add custom styling to preserve formatting
            $htmlContent = str_replace(
                '<head>',
                '<head>
                <style>
                    @page { margin: 2cm; }
                    body { font-family: "Times New Roman", serif; font-size: 12pt; line-height: 1.5; }
                    table { border-collapse: collapse; width: 100%; margin: 10px 0; }
                    td, th { padding: 5px; }
                    p { margin: 0 0 10px 0; }
                </style>',
                $htmlContent
            );

            // Convert HTML to PDF using DomPDF
            $pdf = Pdf::loadHTML($htmlContent)
                ->setOptions([
                    'isRemoteEnabled' => true,
                    'isHtml5ParserEnabled' => true,
                    'defaultFont' => 'Times New Roman',
                ])
                ->setPaper('a4', 'portrait');

            // Save PDF
            file_put_contents($pdfPath, $pdf->output());

            // Cleanup temp file
            @unlink($tempHtmlPath);

            return file_exists($pdfPath);
        } catch (\Exception $e) {
            Log::error('PhpWord PDF conversion failed', ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Generate Word document from template using PhpWord
     *
     * @param bool $forceRefresh Force re-processing of photos (for regenerate action)
     */
    public function generateWordDocument(Certificate $certificate, ?DocumentClassConfiguration $docConfig = null, bool $forceRefresh = false): string
    {
        set_time_limit(120);

        // Ensure student relationship is loaded
        if (!$certificate->relationLoaded('student')) {
            $certificate->load('student');
        }

        $student = $certificate->student;

        // Get document configuration if not provided
        if (!$docConfig) {
            $docConfig = self::getDocumentConfig((int) $certificate->class_level, $certificate->academic_year);
        }

        $documentType = $docConfig ? $docConfig->document_type : DocumentClassConfiguration::TYPE_CERTIFICATE;
        $isDeclaration = $documentType === DocumentClassConfiguration::TYPE_DECLARATION;
        $classLevel = (int) $certificate->class_level;

        // Select template based on document type and class level
        if ($isDeclaration) {
            $templateName = 'declaration_template.docx';
        } elseif ($classLevel == 9) {
            // 9th class uses different certificate format (grades by extenso)
            $templateName = 'certificate_9class_template.docx';
        } else {
            $templateName = 'certificate_template.docx';
        }
        $templatePath = resource_path("templates/{$templateName}");

        if (!file_exists($templatePath)) {
            throw new \Exception("Template Word não encontrado: {$templateName}");
        }

        // Pre-process template to fix broken macros (Word splits placeholders)
        $tempTemplatePath = sys_get_temp_dir() . '/fixed_template_' . uniqid() . '.docx';
        $this->fixWordTemplateMacros($templatePath, $tempTemplatePath);

        $templateProcessor = new TemplateProcessor($tempTemplatePath);

        // Portuguese months
        $months = ['Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];

        // Parse DOB
        $dob = $certificate->student_dob ? Carbon::parse($certificate->student_dob) : null;

        // Student data replacements (short placeholders)
        $templateProcessor->setValue('NOME', strtoupper($certificate->student_name));
        $templateProcessor->setValue('STUDENT_NAME', strtoupper($certificate->student_name)); // Alternative placeholder
        $templateProcessor->setValue('SEX', $certificate->student_sex == 'M' ? 'Masculino' : 'Feminino');
        $templateProcessor->setValue('NAT', $student?->birth_place ?? 'Nampula');
        $templateProcessor->setValue('DIST', $student?->birth_district ?? 'Nampula');
        $templateProcessor->setValue('PROV', $student?->birth_province ?? 'Nampula');
        $templateProcessor->setValue('DD', $dob ? $dob->format('d') : '---');
        $templateProcessor->setValue('MM', $dob ? $months[$dob->month - 1] : '---');
        $templateProcessor->setValue('AA', $dob ? $dob->format('Y') : '---');
        $templateProcessor->setValue('PAI', $student?->father_name ?? '---');
        $templateProcessor->setValue('MAE', $student?->mother_name ?? '---');
        $templateProcessor->setValue('ANOC', $certificate->academic_year);
        $templateProcessor->setValue('CL', $certificate->class_level);
        $templateProcessor->setValue('AREA', 'Comunicação e Ciências Sociais');
        $templateProcessor->setValue('COD', trim($certificate->student_id_number));

        // Student card ID - formatted for display (e.g., "01")
        // Remove all non-numeric characters and spaces, then get last 2 digits
        $studentIdNum = preg_replace('/[^0-9]/', '', trim($certificate->student_id_number));
        $cardId = str_pad(substr($studentIdNum, -2), 2, '0', STR_PAD_LEFT);
        $templateProcessor->setValue('CARD_ID', $cardId);

        // Student card name - abbreviated middle names for declarations
        // "Salum Said Mohamed" -> "Salum S. Mohamed"
        $fullName = trim($certificate->student_name);
        $nameParts = explode(' ', $fullName);
        $firstName = $nameParts[0] ?? '';
        $lastName = count($nameParts) > 1 ? end($nameParts) : '';

        $cardName = $firstName;

        // Add middle name initials
        if (count($nameParts) > 2) {
            $middleNames = array_slice($nameParts, 1, -1);
            foreach ($middleNames as $middleName) {
                $cardName .= ' ' . strtoupper(substr($middleName, 0, 1)) . '.';
            }
        }

        // Add last name
        if ($lastName && $lastName !== $firstName) {
            $cardName .= ' ' . $lastName;
        }

        $cardName = strtoupper(trim($cardName));
        $templateProcessor->setValue('CARD_NOME', $cardName);

        // Additional placeholders for declaration
        if ($isDeclaration) {
            $turmaLetter = preg_replace('/^\d+/', '', $certificate->turma_name);
            $templateProcessor->setValue('TM', $turmaLetter);
            $templateProcessor->setValue('BI', $student?->document_id ?? $student?->bi_number ?? $student?->id_number ?? '_____________');
            $templateProcessor->setValue('COMP', 'Bom');
        }

        // Student photo for card section (works for both certificate and declaration)
        $photoInserted = false;
        $tempFilesToClean = [];

        if ($student && $student->avatar) {
            try {
                // Try to get the photo from URL or local path
                $avatarUrl = $student->avatar;
                $originalPhotoPath = null;

                if (filter_var($avatarUrl, FILTER_VALIDATE_URL)) {
                    // Download from URL to temp file
                    $photoContent = @file_get_contents($avatarUrl);
                    if ($photoContent) {
                        $originalPhotoPath = sys_get_temp_dir() . '/student_photo_' . $student->id . '.jpg';
                        file_put_contents($originalPhotoPath, $photoContent);
                        $tempFilesToClean[] = $originalPhotoPath;
                    }
                } elseif (Storage::disk('public')->exists($avatarUrl)) {
                    $originalPhotoPath = Storage::disk('public')->path($avatarUrl);
                } elseif (file_exists(public_path($avatarUrl))) {
                    $originalPhotoPath = public_path($avatarUrl);
                }

                if ($originalPhotoPath && file_exists($originalPhotoPath)) {
                    // Try to remove background (with cache support)
                    $bgRemovalResult = $this->removeBackgroundFromPhoto($originalPhotoPath, $student->id, $forceRefresh);

                    // Use processed photo if available, otherwise use original
                    $finalPhotoPath = $bgRemovalResult['path'] ?? $originalPhotoPath;
                    // Don't add cached photos to temp cleanup list
                    if ($bgRemovalResult['path'] && !str_contains($bgRemovalResult['path'], 'processed_photos')) {
                        $tempFilesToClean[] = $bgRemovalResult['path'];
                    }

                    // Show warning if background removal failed
                    if ($bgRemovalResult['error']) {
                        session()->flash('warning', 'Aviso: Não foi possível processar a foto automaticamente. O documento foi gerado com a foto original. Por favor, tente novamente mais tarde.');
                    }

                    // Apply 75% opacity to the image
                    $opacityPhotoPath = $this->applyImageOpacity($finalPhotoPath, 0.75, $student->id);
                    if ($opacityPhotoPath) {
                        $finalPhotoPath = $opacityPhotoPath;
                        $tempFilesToClean[] = $opacityPhotoPath;
                    }

                    // Insert photo
                    $templateProcessor->setImageValue('FOTO', [
                        'path' => $finalPhotoPath,
                        'width' => 50,  // ~1.8cm at 72dpi
                        'height' => 62, // ~2.2cm at 72dpi
                        'ratio' => false,
                    ]);
                    $photoInserted = true;
                }
            } catch (\Exception $e) {
                Log::warning("Erro ao inserir foto do aluno: " . $e->getMessage());
            }
        }

        // Clean up temp files
        foreach ($tempFilesToClean as $tempFile) {
            if (file_exists($tempFile)) {
                @unlink($tempFile);
            }
        }

        // If no photo was inserted, replace placeholder with empty or placeholder image
        if (!$photoInserted) {
            // Try to use a default placeholder image
            $defaultPhoto = resource_path('templates/certificates/default_photo.png');
            if (file_exists($defaultPhoto)) {
                $templateProcessor->setImageValue('FOTO', [
                    'path' => $defaultPhoto,
                    'width' => 50,
                    'height' => 62,
                    'ratio' => false,
                ]);
            } else {
                // Just remove the placeholder if no default image
                $templateProcessor->setValue('FOTO', '');
            }
        }

        // Issue date
        $issueDate = $certificate->issued_at ?? now();
        $templateProcessor->setValue('ID', $issueDate->format('d'));
        $templateProcessor->setValue('IM', $months[$issueDate->month - 1]);
        $templateProcessor->setValue('IY', $issueDate->format('Y'));

        // Exam/pauta data
        $templateProcessor->setValue('EP', $certificate->exam_pauta ?? '---');
        $templateProcessor->setValue('EC', $certificate->exam_chamada ?? '1ª');
        $templateProcessor->setValue('EN', $certificate->exam_number ?? '---');

        // Process grades - map subjects to template positions
        // Template order (based on Word layout):
        // Row 1: Português (1), Geografia (2)
        // Row 2: Inglês (3), Matemática (4)
        // Row 3: Francês (5), Ed. Física (6)
        // Row 4: História (7), TIC (8)
        // Row 5: Filosofia (9)

        $grades = $certificate->subject_grades ?? [];

        // First, map all grades to their positions (collect which positions have values)
        $mappedPositions = [];
        foreach ($grades as $subject => $gradeData) {
            $final = $gradeData['final'] ?? 0;
            $extenso = $this->numberToWords($final);
            $subjectLower = mb_strtolower($subject);

            // Find the position for this subject with precise matching (class level affects order)
            $position = $this->getSubjectPosition($subjectLower, $classLevel);

            if ($position && $position <= 12) {
                $mappedPositions[$position] = [
                    'nota' => $final,
                    'extenso' => $extenso,
                    'subject' => $subject,
                ];
                Log::debug("Disciplina mapeada: {$subject} -> posição {$position}, nota {$final}");
            } else {
                Log::warning("Disciplina não mapeada: {$subject}");
            }
        }

        // Now set values - first the actual grades, then defaults for unmapped positions
        for ($i = 1; $i <= 12; $i++) {
            if (isset($mappedPositions[$i])) {
                // Trim any whitespace/newlines from extenso
                $extenso = trim($mappedPositions[$i]['extenso']);
                $templateProcessor->setValue("E{$i}", $extenso);
                $templateProcessor->setValue("N{$i}", $mappedPositions[$i]['nota']);
            } else {
                $templateProcessor->setValue("E{$i}", '---');
                $templateProcessor->setValue("N{$i}", '-');
            }
        }

        // Final average - separate placeholders for flexibility
        $finalAvg = (float) ($certificate->final_average ?? 0);
        $finalAvgInt = (int) round($finalAvg);
        $templateProcessor->setValue('MF', $finalAvgInt);  // Just number: 12
        $templateProcessor->setValue('ME', $this->numberToWords($finalAvgInt));  // Just extenso: Doze

        // Save document
        $docTypeSlug = $isDeclaration ? 'declaracao' : 'certificado';
        $fileName = Str::slug($certificate->student_name) . "_{$docTypeSlug}_" . $certificate->certificate_number . '.docx';
        $filePath = "certificates/{$certificate->academic_year}/word/{$fileName}";
        $fullPath = Storage::disk('public')->path($filePath);

        // Ensure directory exists
        $dir = dirname($fullPath);
        if (!file_exists($dir)) {
            mkdir($dir, 0755, true);
        }

        $templateProcessor->saveAs($fullPath);

        // Cleanup temp file
        if (file_exists($tempTemplatePath)) {
            @unlink($tempTemplatePath);
        }

        Log::info('Documento Word gerado', [
            'certificate_id' => $certificate->id,
            'file_path' => $filePath,
        ]);

        return $filePath;
    }

    /**
     * Convert number to Portuguese words (extenso)
     */
    private function numberToWords(int $number): string
    {
        $units = ['', 'Um', 'Dois', 'Três', 'Quatro', 'Cinco', 'Seis', 'Sete', 'Oito', 'Nove'];
        $teens = ['Dez', 'Onze', 'Doze', 'Treze', 'Catorze', 'Quinze', 'Dezasseis', 'Dezassete', 'Dezoito', 'Dezanove'];
        $tens = ['', '', 'Vinte', 'Trinta', 'Quarenta', 'Cinquenta', 'Sessenta', 'Setenta', 'Oitenta', 'Noventa'];

        if ($number < 0 || $number > 20) {
            // For grades, typically 0-20
            if ($number > 20) {
                return 'Vinte'; // Cap at 20
            }
            return 'Zero';
        }

        if ($number == 0) {
            return 'Zero';
        }

        if ($number < 10) {
            return $units[$number];
        }

        if ($number < 20) {
            return $teens[$number - 10];
        }

        if ($number == 20) {
            return 'Vinte';
        }

        return $tens[(int)($number / 10)] . ($number % 10 ? ' e ' . $units[$number % 10] : '');
    }

    /**
     * Get template position for a subject name
     * Uses precise matching to avoid conflicts between similar subject names
     *
     * Template positions for 12th class (certificate_template.docx):
     * Col 1: Português(1), Inglês(2), Francês(3), História(4), Filosofia(9)
     * Col 2: Geografia(5), Matemática(6), Ed.Física(7), TIC(8)
     * Extras: Biologia(10), Química(11), Física(12)
     *
     * Template positions for 9th class (certificate_9class_template.docx):
     * 1-Português, 2-Inglês, 3-História, 4-Geografia, 5-Biologia, 6-Física,
     * 7-Química, 8-Matemática, 9-Francês, 10-TIC's, 11-Ed.Visual, 12-Ed.Física
     */
    private function getSubjectPosition(string $subjectLower, int $classLevel = 12): ?int
    {
        // 9th class has different subject order
        if ($classLevel == 9) {
            return $this->getSubjectPosition9thClass($subjectLower);
        }

        // 12th class (default) mapping
        $exactMatches = [
            'português' => 1, 'portugues' => 1,
            'inglês' => 2, 'ingles' => 2,
            'francês' => 3, 'frances' => 3,
            'história' => 4, 'historia' => 4,
            'geografia' => 5,
            'matemática' => 6, 'matematica' => 6,
            'filosofia' => 9,
            'biologia' => 10,
            'química' => 11, 'quimica' => 11,
        ];

        // Check exact matches
        if (isset($exactMatches[$subjectLower])) {
            return $exactMatches[$subjectLower];
        }

        // Check for Ed. Física / Educação Física (position 7) - must check BEFORE standalone "Física"
        if (str_contains($subjectLower, 'educação') && str_contains($subjectLower, 'fisica') ||
            str_contains($subjectLower, 'educacao') && str_contains($subjectLower, 'fisica') ||
            str_contains($subjectLower, 'ed.') && str_contains($subjectLower, 'fisica') ||
            str_contains($subjectLower, 'ed.') && str_contains($subjectLower, 'física')) {
            return 7;
        }

        // Standalone Física (position 12) - only if NOT "Educação Física"
        if (($subjectLower === 'física' || $subjectLower === 'fisica') &&
            !str_contains($subjectLower, 'educação') && !str_contains($subjectLower, 'educacao')) {
            return 12;
        }

        // TIC / Informática (position 8)
        if ($subjectLower === 'tic' || str_contains($subjectLower, 'informática') || str_contains($subjectLower, 'informatica')) {
            return 8;
        }

        return null;
    }

    /**
     * Get template position for 9th class subjects
     * Order: Português(1), Inglês(2), História(3), Geografia(4), Biologia(5),
     *        Física(6), Química(7), Matemática(8), Francês(9), TIC's(10),
     *        Ed.Visual(11), Ed.Física(12)
     */
    private function getSubjectPosition9thClass(string $subjectLower): ?int
    {
        $exactMatches = [
            'português' => 1, 'portugues' => 1,
            'inglês' => 2, 'ingles' => 2,
            'história' => 3, 'historia' => 3,
            'geografia' => 4,
            'biologia' => 5,
            'química' => 7, 'quimica' => 7,
            'matemática' => 8, 'matematica' => 8,
            'francês' => 9, 'frances' => 9,
        ];

        if (isset($exactMatches[$subjectLower])) {
            return $exactMatches[$subjectLower];
        }

        // Física (position 6) - must check it's not "Educação Física"
        if (($subjectLower === 'física' || $subjectLower === 'fisica') &&
            !str_contains($subjectLower, 'educação') && !str_contains($subjectLower, 'educacao') &&
            !str_contains($subjectLower, 'visual')) {
            return 6;
        }

        // TIC / TIC's / Informática (position 10)
        if ($subjectLower === 'tic' || $subjectLower === "tic's" || $subjectLower === 'tics' ||
            str_contains($subjectLower, 'informática') || str_contains($subjectLower, 'informatica')) {
            return 10;
        }

        // Educação Visual (position 11)
        if (str_contains($subjectLower, 'visual') ||
            (str_contains($subjectLower, 'ed.') && str_contains($subjectLower, 'visual'))) {
            return 11;
        }

        // Educação Física (position 12)
        if (str_contains($subjectLower, 'educação') && str_contains($subjectLower, 'fisica') ||
            str_contains($subjectLower, 'educacao') && str_contains($subjectLower, 'fisica') ||
            str_contains($subjectLower, 'ed.') && str_contains($subjectLower, 'fisica') ||
            str_contains($subjectLower, 'ed.') && str_contains($subjectLower, 'física')) {
            return 12;
        }

        return null;
    }

    /**
     * Fix broken macros in Word template
     * Word often splits placeholders like ${nota1} across multiple XML runs
     * This method merges them back together
     */
    private function fixWordTemplateMacros(string $sourcePath, string $destPath): void
    {
        // Copy the template
        copy($sourcePath, $destPath);

        $zip = new ZipArchive();
        if ($zip->open($destPath) !== true) {
            throw new \Exception("Não foi possível abrir o template Word");
        }

        // Get document.xml content
        $xmlContent = $zip->getFromName('word/document.xml');

        if ($xmlContent === false) {
            $zip->close();
            throw new \Exception("Não foi possível ler document.xml do template");
        }

        // Remove proof error markers that break placeholders
        // These are tags like <w:proofErr w:type="gramStart"/> and <w:proofErr w:type="gramEnd"/>
        $xmlContent = preg_replace('/<w:proofErr[^>]*\/?>/i', '', $xmlContent);

        // Fix split placeholders: merge runs that contain parts of ${...}
        // Pattern to find placeholders split across multiple runs
        // Match: </w:t></w:r>...<w:r>...<w:t> within a ${...} placeholder
        $xmlContent = preg_replace_callback(
            '/(\$\{[^}]*?)(<\/w:t><\/w:r>.*?<w:r[^>]*>.*?<w:t[^>]*>)([^}]*?\})/s',
            function ($matches) {
                // Keep only the text parts, remove the XML run boundaries
                return $matches[1] . $matches[3];
            },
            $xmlContent
        );

        // Also handle simpler cases where the closing } is in a separate run
        $xmlContent = preg_replace(
            '/(\$\{[a-zA-Z0-9_]+)<\/w:t><\/w:r><w:r[^>]*><w:t[^>]*>(\})/s',
            '$1$2',
            $xmlContent
        );

        // Note: Font change removed - was causing document corruption
        // Alignment should be handled in the Word template itself using tabs or table structure

        // Save fixed content
        $zip->deleteName('word/document.xml');
        $zip->addFromString('word/document.xml', $xmlContent);
        $zip->close();

        Log::debug('Template Word corrigido para macros fragmentados');
    }

    /**
     * Verify certificate by code
     */
    public function verifyCertificate(string $code): ?array
    {
        $certificate = Certificate::where('verification_code', $code)->first();

        if (!$certificate) {
            return null;
        }

        // Recalculate hash to verify integrity
        $hashData = [
            'student_id' => (string) $certificate->student_id,
            'certificate_number' => $certificate->certificate_number,
            'academic_year' => (int) $certificate->academic_year,
            'final_average' => round((float) $certificate->final_average, 2),
            'issued_at' => $certificate->issued_at->toIso8601String(),
        ];
        $calculatedHash = $this->generateVerificationHash($hashData);

        return [
            'valid' => $certificate->isValid() && $calculatedHash === $certificate->verification_hash,
            'certificate' => $certificate,
            'integrity_verified' => $calculatedHash === $certificate->verification_hash,
            'status' => $certificate->status,
        ];
    }

    /**
     * Get eligible students for certificate/declaration generation
     */
    public function getEligibleStudents(?string $classroomId = null, ?int $academicYear = null, ?int $classLevel = null): array
    {
        $academicYear = $academicYear ?? (int) date('Y');
        $configuredClasses = self::getConfiguredClasses($academicYear);

        // IMPORTANTE: Buscar estudantes com base nos registos de exames do ano especificado
        // MAS também incluir estudantes actualmente na turma (para mostrar todos, mesmo sem exames)
        if ($classroomId) {
            // Turma específica:
            // 1. Estudantes que fizeram exames nessa turma naquele ano
            $studentIdsWithExams = Exam::where('year', $academicYear)
                ->where('classroom_id', $classroomId)
                ->distinct()
                ->pluck('student_id')
                ->toArray();

            // 2. Estudantes actualmente nessa turma (podem não ter exames ainda)
            $studentsInClassroom = User::where('classroom_id', $classroomId)
                ->where('is_active', 1)
                ->pluck('id')
                ->toArray();

            // Combinar ambas as listas
            $allStudentIds = array_unique(array_merge($studentIdsWithExams, $studentsInClassroom));

            $students = User::whereIn('id', $allStudentIds)
                ->where('is_active', 1)
                ->orderBy('name')
                ->get();
        } elseif ($classLevel) {
            // Nível de classe específico:
            $classroomIds = Classroom::where('class', $classLevel)->pluck('id')->toArray();

            // 1. Estudantes com exames em turmas desse nível
            $studentIdsWithExams = Exam::where('year', $academicYear)
                ->whereIn('classroom_id', $classroomIds)
                ->distinct()
                ->pluck('student_id')
                ->toArray();

            // 2. Estudantes actualmente em turmas desse nível
            $studentsInClassrooms = User::whereIn('classroom_id', $classroomIds)
                ->where('is_active', 1)
                ->pluck('id')
                ->toArray();

            // Combinar ambas as listas
            $allStudentIds = array_unique(array_merge($studentIdsWithExams, $studentsInClassrooms));

            $students = User::whereIn('id', $allStudentIds)
                ->where('is_active', 1)
                ->orderBy('name')
                ->get();
        } else {
            // Todas as classes configuradas:
            $classroomIds = Classroom::whereIn('class', $configuredClasses)->pluck('id')->toArray();

            // 1. Estudantes com exames em turmas configuradas
            $studentIdsWithExams = Exam::where('year', $academicYear)
                ->whereIn('classroom_id', $classroomIds)
                ->distinct()
                ->pluck('student_id')
                ->toArray();

            // 2. Estudantes actualmente em turmas configuradas
            $studentsInClassrooms = User::whereIn('classroom_id', $classroomIds)
                ->where('is_active', 1)
                ->pluck('id')
                ->toArray();

            // Combinar ambas as listas
            $allStudentIds = array_unique(array_merge($studentIdsWithExams, $studentsInClassrooms));

            $students = User::whereIn('id', $allStudentIds)
                ->where('is_active', 1)
                ->orderBy('name')
                ->get();
        }
        $eligible = [];
        $ineligible = [];

        foreach ($students as $student) {
            // Passar o contexto da turma quando estamos a filtrar por turma específica
            // Isso garante que a verificação de elegibilidade usa a turma correcta
            $check = $this->isEligible($student, $academicYear, $classroomId);

            if ($check['eligible']) {
                // Check if already has certificate/declaration
                $hasCert = Certificate::where('student_id', $student->id)
                    ->where('academic_year', $academicYear)
                    ->where('status', '!=', 'revoked')
                    ->exists();

                $eligible[] = [
                    'student' => $student,
                    'grades' => $check['grades'],
                    'payment_status' => $check['payment_status'],
                    'has_certificate' => $hasCert,
                    'document_config' => $check['document_config'],
                ];
            } else {
                $ineligible[] = [
                    'student' => $student,
                    'reason' => $check['reason'],
                    'grades' => $check['grades'],
                    'payment_status' => $check['payment_status'],
                    'document_config' => $check['document_config'] ?? null,
                ];
            }
        }

        return [
            'eligible' => $eligible,
            'ineligible' => $ineligible,
            'total_eligible' => count($eligible),
            'total_ineligible' => count($ineligible),
            'eligible_without_cert' => count(array_filter($eligible, fn($e) => !$e['has_certificate'])),
        ];
    }

    /**
     * Create batch and generate certificates for multiple students
     */
    public function createBatch(?string $classroomId, int $academicYear, string $adminId, ?int $classLevel = null): CertificateBatch
    {
        $batchNumber = 'BATCH-' . $academicYear . '-' . str_pad(
            CertificateBatch::whereYear('created_at', date('Y'))->count() + 1,
            3, '0', STR_PAD_LEFT
        );

        $eligibleData = $this->getEligibleStudents($classroomId, $academicYear, $classLevel);
        $eligibleWithoutCert = array_filter($eligibleData['eligible'], fn($e) => !$e['has_certificate']);

        $classroom = $classroomId ? Classroom::find($classroomId) : null;
        $classLabel = $classLevel ? "{$classLevel}a classe" : "6a e 12a classe";

        $batch = CertificateBatch::create([
            'batch_number' => $batchNumber,
            'description' => $classroomId
                ? 'Certificados para turma ' . ($classroom->name ?? 'especifica')
                : "Certificados para todas as turmas da {$classLabel}",
            'classroom_id' => $classroomId,
            'academic_year' => $academicYear,
            'status' => 'pending',
            'total_students' => count($eligibleWithoutCert),
            'created_by' => $adminId,
        ]);

        return $batch;
    }

    /**
     * Process batch (generate certificates)
     */
    public function processBatch(CertificateBatch $batch): void
    {
        $batch->update(['status' => 'processing']);

        // Get class level from classroom if available
        $classLevel = null;
        if ($batch->classroom_id) {
            $classroom = Classroom::find($batch->classroom_id);
            $classLevel = $classroom ? (int) $classroom->class : null;
        }

        $eligibleData = $this->getEligibleStudents($batch->classroom_id, $batch->academic_year, $classLevel);
        $eligibleWithoutCert = array_filter($eligibleData['eligible'], fn($e) => !$e['has_certificate']);

        $successCount = 0;
        $failedCount = 0;
        $failedStudents = [];

        foreach ($eligibleWithoutCert as $item) {
            try {
                $this->createCertificate($item['student'], $batch->academic_year, $batch->created_by);
                $successCount++;
            } catch (\Exception $e) {
                $failedCount++;
                $failedStudents[] = [
                    'student_id' => $item['student']->id,
                    'name' => $item['student']->name,
                    'error' => $e->getMessage(),
                ];
                Log::error('Erro ao gerar certificado em lote', [
                    'batch_id' => $batch->id,
                    'student_id' => $item['student']->id,
                    'error' => $e->getMessage(),
                ]);
            }

            $batch->update([
                'processed_count' => $successCount + $failedCount,
                'success_count' => $successCount,
                'failed_count' => $failedCount,
                'failed_students' => $failedStudents,
            ]);
        }

        $finalStatus = match(true) {
            $failedCount > 0 && $successCount === 0 => 'failed',
            $failedCount > 0 => 'completed', // partial success
            default => 'completed',
        };

        $batch->update(['status' => $finalStatus]);

        Log::info('Lote de certificados processado', [
            'batch_id' => $batch->id,
            'batch_number' => $batch->batch_number,
            'success' => $successCount,
            'failed' => $failedCount,
        ]);
    }

    /**
     * Generate ZIP file for batch download (Word documents)
     */
    public function generateBatchZip(CertificateBatch $batch): string
    {
        $query = Certificate::where('academic_year', $batch->academic_year)
            ->where('status', 'issued');

        if ($batch->classroom_id) {
            $query->where('classroom_id', $batch->classroom_id);
        }

        $certificates = $query->get();

        if ($certificates->isEmpty()) {
            throw new \Exception('Nenhum certificado encontrado para este lote');
        }

        // Build filename: doctype_class_turma_ano_batchcode
        $classroom = $batch->classroom_id ? Classroom::find($batch->classroom_id) : null;
        $classLevel = $classroom ? $classroom->class : 'todas';
        $turmaName = $classroom ? Str::slug($classroom->name) : 'todas';

        // Determine document type from configuration
        $docType = 'doc';
        if ($classroom) {
            $docConfig = self::getDocumentConfig((int) $classroom->class, $batch->academic_year);
            $docType = $docConfig && $docConfig->isCertificate() ? 'cert' : 'decl';
        }

        $zipFileName = "{$docType}_{$classLevel}classe_{$turmaName}_{$batch->academic_year}_{$batch->batch_number}.zip";
        $zipPath = "certificates/batches/{$zipFileName}";
        $fullZipPath = Storage::disk('public')->path($zipPath);

        // Ensure directory exists
        $dir = dirname($fullZipPath);
        if (!file_exists($dir)) {
            mkdir($dir, 0755, true);
        }

        $zip = new ZipArchive();
        if ($zip->open($fullZipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new \Exception("Nao foi possivel criar arquivo ZIP");
        }

        $addedCount = 0;
        foreach ($certificates as $cert) {
            try {
                // Generate Word document for each certificate
                $wordPath = $this->generateWordDocument($cert);
                $fullWordPath = Storage::disk('public')->path($wordPath);

                if (file_exists($fullWordPath)) {
                    $wordContent = file_get_contents($fullWordPath);
                    $zip->addFromString(basename($wordPath), $wordContent);
                    $addedCount++;
                }
            } catch (\Exception $e) {
                Log::warning("Erro ao gerar Word para certificado {$cert->id}: " . $e->getMessage());
            }
        }

        $zip->close();

        if ($addedCount === 0) {
            // Clean up empty zip
            if (file_exists($fullZipPath)) {
                unlink($fullZipPath);
            }
            throw new \Exception('Nenhum documento Word gerado para adicionar ao ZIP');
        }

        $batch->update([
            'zip_path' => $zipPath,
            'zip_generated_at' => now(),
        ]);

        Log::info('ZIP de certificados Word gerado', [
            'batch_id' => $batch->id,
            'zip_path' => $zipPath,
            'certificates_count' => $addedCount,
        ]);

        return $zipPath;
    }

    /**
     * Generate ZIP for all certificates of a year/classroom without batch (Word documents)
     */
    public function generateDirectZip(?string $classroomId, int $academicYear): string
    {
        $query = Certificate::where('academic_year', $academicYear)
            ->where('status', 'issued');

        if ($classroomId) {
            $query->where('classroom_id', $classroomId);
        }

        $certificates = $query->get();

        if ($certificates->isEmpty()) {
            throw new \Exception('Nenhum certificado encontrado');
        }

        $classroom = $classroomId ? Classroom::find($classroomId) : null;
        $classLevel = $classroom ? $classroom->class : 'todas';
        $turmaName = $classroom ? Str::slug($classroom->name) : 'todas_turmas';

        // Determine document type from configuration
        $docType = 'doc';
        if ($classroom) {
            $docConfig = self::getDocumentConfig((int) $classroom->class, $academicYear);
            $docType = $docConfig && $docConfig->isCertificate() ? 'cert' : 'decl';
        }

        $timestamp = now()->format('His');
        $zipFileName = "{$docType}_{$classLevel}classe_{$turmaName}_{$academicYear}_{$timestamp}.zip";
        $zipPath = "certificates/downloads/{$zipFileName}";
        $fullZipPath = Storage::disk('public')->path($zipPath);

        // Ensure directory exists
        $dir = dirname($fullZipPath);
        if (!file_exists($dir)) {
            mkdir($dir, 0755, true);
        }

        $zip = new ZipArchive();
        if ($zip->open($fullZipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new \Exception("Nao foi possivel criar arquivo ZIP");
        }

        $addedCount = 0;
        foreach ($certificates as $cert) {
            try {
                // Generate Word document for each certificate
                $wordPath = $this->generateWordDocument($cert);
                $fullWordPath = Storage::disk('public')->path($wordPath);

                if (file_exists($fullWordPath)) {
                    $wordContent = file_get_contents($fullWordPath);
                    $zip->addFromString(basename($wordPath), $wordContent);
                    $addedCount++;
                }
            } catch (\Exception $e) {
                Log::warning("Erro ao gerar Word para certificado {$cert->id}: " . $e->getMessage());
            }
        }

        $zip->close();

        if ($addedCount === 0) {
            if (file_exists($fullZipPath)) {
                unlink($fullZipPath);
            }
            throw new \Exception('Nenhum documento Word gerado para adicionar ao ZIP');
        }

        return $zipPath;
    }

    /**
     * Revoke a certificate
     */
    public function revokeCertificate(Certificate $certificate, string $reason, string $adminId): void
    {
        if ($certificate->status === 'revoked') {
            throw new \Exception('Certificado ja esta revogado');
        }

        $certificate->update([
            'status' => 'revoked',
            'revoked_at' => now(),
            'revocation_reason' => $reason,
            'revoked_by' => $adminId,
        ]);

        Log::info('Certificado revogado', [
            'certificate_id' => $certificate->id,
            'certificate_number' => $certificate->certificate_number,
            'revoked_by' => $adminId,
            'reason' => $reason,
        ]);
    }

    /**
     * Get statistics
     */
    public function getStatistics(?int $academicYear = null): array
    {
        $year = $academicYear ?? (int) date('Y');

        return [
            'total_certificates' => Certificate::forYear($year)->count(),
            'total_issued' => Certificate::forYear($year)->issued()->count(),
            'total_revoked' => Certificate::forYear($year)->revoked()->count(),
            'pending_batches' => CertificateBatch::forYear($year)->whereIn('status', ['pending', 'processing'])->count(),
            'completed_batches' => CertificateBatch::forYear($year)->completed()->count(),
        ];
    }

    /**
     * Remove background from photo using background removal service
     * Returns array with path and error status
     *
     * @param string $imagePath Path to original image
     * @param string $studentId Student ID for temp file naming
     * @param bool $forceRefresh Force re-processing even if cached version exists
     * @return array ['path' => string|null, 'error' => bool]
     */
    private function removeBackgroundFromPhoto(string $imagePath, string $studentId, bool $forceRefresh = false): array
    {
        $hasError = false;

        // Check for cached processed photo (unless force refresh)
        $cachedPhotoPath = 'processed_photos/' . $studentId . '_nobg.png';
        $fullCachedPath = Storage::disk('public')->path($cachedPhotoPath);

        if (!$forceRefresh && Storage::disk('public')->exists($cachedPhotoPath)) {
            Log::info('Usando foto em cache (sem chamar API)', ['student_id' => $studentId]);
            return ['path' => $fullCachedPath, 'error' => false];
        }

        // Try rembg API first, fallback to remove.bg if not configured
        $rembgKey = config('services.rembg.api_key') ?? env('REMBG_API_KEY');
        $rembgUrl = config('services.rembg.base_url', 'https://api.rembg.com');

        if ($rembgKey) {
            try {
                $response = Http::timeout(30)
                    ->withHeaders([
                        'x-api-key' => $rembgKey,
                    ])
                    ->attach('image', file_get_contents($imagePath), basename($imagePath))
                    ->post($rembgUrl . '/rmbg', [
                        'format' => 'png',
                        'w' => 500,  // resize to 500px width
                        'h' => 500,  // resize to 500px height
                        'exact_resize' => 'false', // maintain aspect ratio
                    ]);

                if ($response->successful()) {
                    // Save processed image to permanent storage
                    $cachedPhotoPath = 'processed_photos/' . $studentId . '_nobg.png';
                    Storage::disk('public')->put($cachedPhotoPath, $response->body());
                    $fullCachedPath = Storage::disk('public')->path($cachedPhotoPath);

                    Log::info('Fundo removido com sucesso e salvo em cache', ['student_id' => $studentId]);
                    return ['path' => $fullCachedPath, 'error' => false];
                }

                Log::warning('Serviço de processamento de imagem temporariamente indisponível', [
                    'status' => $response->status(),
                ]);
                $hasError = true;

            } catch (\Exception $e) {
                Log::warning('Erro ao processar imagem', [
                    'message' => 'Falha na comunicação com o serviço de processamento',
                ]);
                $hasError = true;
            }
        }

        // Fallback to remove.bg API if rembg fails or is not configured
        $removebgKey = config('services.removebg.api_key') ?? env('REMOVEBG_API_KEY');

        if (!$removebgKey) {
            Log::debug('Serviço de processamento: Nenhuma configuração encontrada, usando foto original');
            return ['path' => null, 'error' => false]; // No error if no API key configured
        }

        try {
            $response = Http::timeout(30)
                ->withHeaders([
                    'X-Api-Key' => $removebgKey,
                ])
                ->attach('image_file', file_get_contents($imagePath), basename($imagePath))
                ->post('https://api.remove.bg/v1.0/removebg', [
                    'size' => 'preview', // preview = free tier, auto = paid
                    'format' => 'png',
                ]);

            if ($response->successful()) {
                // Save processed image to permanent storage
                $cachedPhotoPath = 'processed_photos/' . $studentId . '_nobg.png';
                Storage::disk('public')->put($cachedPhotoPath, $response->body());
                $fullCachedPath = Storage::disk('public')->path($cachedPhotoPath);

                Log::info('Fundo removido com sucesso e salvo em cache (serviço alternativo)', ['student_id' => $studentId]);
                return ['path' => $fullCachedPath, 'error' => false];
            }

            Log::warning('Serviço de processamento de imagem temporariamente indisponível (alternativo)', [
                'status' => $response->status(),
            ]);

            return ['path' => null, 'error' => true];

        } catch (\Exception $e) {
            Log::warning('Erro ao processar imagem (alternativo)', [
                'message' => 'Falha na comunicação com o serviço de processamento',
            ]);
            return ['path' => null, 'error' => true];
        }
    }

    /**
     * Apply opacity to an image using GD library
     * Optimized to avoid memory exhaustion with large images
     *
     * @param string $imagePath Path to original image
     * @param float $opacity Opacity level (0.0 to 1.0)
     * @param string $studentId Student ID for temp file naming
     * @return string|null Path to processed image or null
     */
    private function applyImageOpacity(string $imagePath, float $opacity, string $studentId): ?string
    {
        try {
            // Get image info
            $imageInfo = getimagesize($imagePath);
            if (!$imageInfo) {
                return null;
            }

            $mime = $imageInfo['mime'];
            $width = $imageInfo[0];
            $height = $imageInfo[1];

            // Skip opacity processing for very large images to avoid memory issues
            // Max 500x500 = 250,000 pixels for pixel-by-pixel processing
            $maxPixels = 250000;
            $totalPixels = $width * $height;

            if ($totalPixels > $maxPixels) {
                Log::info('Imagem muito grande para aplicar opacidade, pulando processamento', [
                    'width' => $width,
                    'height' => $height,
                    'pixels' => $totalPixels,
                ]);
                return null; // Skip opacity, use original image
            }

            // Create image resource based on type
            $sourceImage = match ($mime) {
                'image/png' => imagecreatefrompng($imagePath),
                'image/jpeg', 'image/jpg' => imagecreatefromjpeg($imagePath),
                'image/gif' => imagecreatefromgif($imagePath),
                default => null,
            };

            if (!$sourceImage) {
                return null;
            }

            // Create new true color image with alpha
            $newImage = imagecreatetruecolor($width, $height);

            // Enable alpha blending and save alpha
            imagealphablending($newImage, false);
            imagesavealpha($newImage, true);

            // Fill with transparent background
            $transparent = imagecolorallocatealpha($newImage, 0, 0, 0, 127);
            imagefill($newImage, 0, 0, $transparent);

            // Enable alpha blending for copying
            imagealphablending($newImage, true);

            // Copy source image with opacity
            // Calculate opacity for GD (0 = opaque, 127 = transparent)
            $gdOpacity = (int) ((1 - $opacity) * 127);

            // Process pixel by pixel to apply opacity
            for ($x = 0; $x < $width; $x++) {
                for ($y = 0; $y < $height; $y++) {
                    $rgba = imagecolorat($sourceImage, $x, $y);
                    $alpha = ($rgba >> 24) & 0x7F;
                    $red = ($rgba >> 16) & 0xFF;
                    $green = ($rgba >> 8) & 0xFF;
                    $blue = $rgba & 0xFF;

                    // Apply additional opacity
                    $newAlpha = min(127, $alpha + $gdOpacity);

                    $newColor = imagecolorallocatealpha($newImage, $red, $green, $blue, $newAlpha);
                    imagesetpixel($newImage, $x, $y, $newColor);
                }
            }

            // Save to temp file
            $outputPath = sys_get_temp_dir() . '/student_photo_opacity_' . $studentId . '.png';
            imagepng($newImage, $outputPath);

            // Free memory
            imagedestroy($sourceImage);
            imagedestroy($newImage);

            return $outputPath;

        } catch (\Exception $e) {
            Log::warning('Erro ao aplicar opacidade na imagem', [
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }
}
