<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ReferenceGenerator
{
    /**
     * EMISSÃO COMPLETA (recomendado):
     * - Aplica regra do dia 5 (mês seguinte) como data de expiração
     * - Se hoje for >= dia 6 → calcula multa e gera NOVA referência com (base + multa)
     * - Retorna payload completo para persistir/exibir
     *
     * @param string               $month        ex: 'January'
     * @param int                  $year         ex: 2025
     * @param string|object        $student      student_id (string) OU objeto Student
     * @param float                $baseAmount   valor base da taxa
     * @param array<string,mixed>  $options      ['fine_type'=>'fixed|percentage','fine_value'=>float,'entity'=>string,'expires_at'=>Date|str]
     * @return array{
     *    entity:string, reference:string,
     *    base_amount:float, fine_amount:float, total_amount:float,
     *    expires_at:\Carbon\Carbon, ttl_days:int,
     *    status:'pending'|'expired',
     *    month:string, year:int,
     *    should_expire_previous:bool,
     * }
     */
    public function issue(string $month, int $year, $student, float $baseAmount, array $options = []): array
    {
        // Entity
        $entity = $options['entity'] ?? config('payments.entity', '11111');

        // Student number (7 dígitos)
        if (is_object($student)) {
            $studentNumber = $this->createStudentCode($student);
        } else {
            $studentNumber = $this->sanitizeStudentId((string)$student);
        }

        // Mês (2 dígitos) para o algoritmo BCI
        $monthNumber = str_pad((string)$this->getMonthNumber($month), 2, '0', STR_PAD_LEFT);

        // Cutoff: 5 do mês seguinte às 23:59:59
        $cutoff = $this->cutoffDay5OfNextMonth($month, $year);

        // Se foi passado expires_at explicitamente
        if (!empty($options['expires_at'])) {
            try {
                $cutoff = Carbon::parse($options['expires_at'])->setTime(23, 59, 59);
            } catch (\Throwable $e) {
                // mantém o cutoff padrão
            }
        }

        // Estamos depois do cutoff?
        $needsFine = now()->greaterThan($cutoff);
        $fineAmount = 0.0;

        // Cálculo de multa
        if ($needsFine) {
            $fineType = $options['fine_type'] ?? config('payments.fine.type', 'fixed');
            $fineValue = $options['fine_value'] ?? config('payments.fine.value', 0);
            $fineAmount = $this->calculateFine($baseAmount, $fineType, (float)$fineValue);
        }

        $totalAmount = round($baseAmount + $fineAmount, 2);

        // Se estamos a reemitir com multa, novo vencimento = próximo dia 5
        $expiresAt = $needsFine ? $this->nextCutoffFrom(now()) : $cutoff;

        // TTL em dias
        $ttlRaw = now()->diffInDays($expiresAt, false);
        $ttlDays = max(0, $ttlRaw);

        // Status
        $status = $ttlDays > 0 ? 'pending' : 'expired';

        // Geração da referência BCI com TOTAL (base+multa)
        $yearSuffix = substr((string)$year, -2); // Últimos 2 dígitos do ano
        $amountCents = (string)round($totalAmount * 100);
        $reference = $this->generateBCIReference($entity, $studentNumber, $monthNumber, $yearSuffix, $amountCents);

        // Flag para expirar referência anterior
        $shouldExpirePrevious = $needsFine;

        Log::info('Reference issued', [
            'entity' => $entity,
            'reference' => $reference,
            'month' => "$month $year",
            'amounts' => [
                'base' => $baseAmount,
                'fine' => $fineAmount,
                'total' => $totalAmount
            ],
            'expires_at' => $expiresAt->toDateTimeString(),
            'status' => $status
        ]);

        return [
            'entity' => $entity,
            'reference' => $reference,
            'base_amount' => $baseAmount,
            'fine_amount' => $fineAmount,
            'total_amount' => $totalAmount,
            'expires_at' => $expiresAt,
            'ttl_days' => $ttlDays,
            'status' => $status,
            'month' => $month,
            'year' => $year,
            'should_expire_previous' => $shouldExpirePrevious,
        ];
    }

    /**
     * Método simplificado para gerar apenas a referência
     * Garante unicidade adicionando contador se necessário
     */
    public function make(string $month, int $year, string $studentId, float $amount): string
    {
        $entity = config('payments.entity', '90013');
        $monthNumber = str_pad((string)$this->getMonthNumber($month), 2, '0', STR_PAD_LEFT);
        $yearSuffix = substr((string)$year, -2); // Últimos 2 dígitos do ano
        $amountCents = (string)round($amount * 100);
        $studentNumber = $this->sanitizeStudentId($studentId);

        // Gerar referência base
        $baseReference = $this->generateBCIReference($entity, $studentNumber, $monthNumber, $yearSuffix, $amountCents);

        // Verificar se já existe e adicionar variação no valor se necessário
        $reference = $baseReference;
        $attempt = 0;
        $maxAttempts = 100;

        while ($this->referenceExists($reference) && $attempt < $maxAttempts) {
            // Adicionar sufixo aos últimos 2 dígitos do número do estudante
            $attempt++;

            // Modificar os últimos 2 dígitos (permite até 100 variações)
            $lastTwoDigits = (int)substr($studentNumber, -2);
            $modifiedLastTwoDigits = str_pad((string)(($lastTwoDigits + $attempt) % 100), 2, '0', STR_PAD_LEFT);
            $modifiedStudentNumber = substr($studentNumber, 0, 5) . $modifiedLastTwoDigits;

            $reference = $this->generateBCIReference($entity, $modifiedStudentNumber, $monthNumber, $yearSuffix, $amountCents);

            Log::info('Reference collision detected, generating alternative', [
                'attempt' => $attempt,
                'original_reference' => $baseReference,
                'original_student_number' => $studentNumber,
                'modified_student_number' => $modifiedStudentNumber,
                'new_reference' => $reference
            ]);
        }

        if ($attempt >= $maxAttempts) {
            throw new \RuntimeException('Não foi possível gerar referência única após ' . $maxAttempts . ' tentativas');
        }

        return $reference;
    }

    /**
     * Gerar referência a partir de objeto Student
     * Garante unicidade mesmo para múltiplas compras no mesmo mês
     */
    public function makeFromStudent(string $month, int $year, $student, float $amount): string
    {
        $studentCode = $this->createStudentCode($student);

        // Usar método make() que já tem lógica de unicidade
        return $this->make($month, $year, $studentCode, $amount);
    }

    /**
     * Algoritmo BCI/BMEPS para gerar referência
     * Implementação atualizada com ano incluído para unicidade
     * @return string A referência gerada (13 dígitos: número + mês + ano + check digit)
     */
    private function generateBCIReference(string $entity, string $number, string $month, string $year, string $amountCents): string
    {
        // Limpar entrada - apenas dígitos
        $entity = preg_replace('/\D+/', '', $entity);
        $number = preg_replace('/\D+/', '', $number);
        $month = preg_replace('/\D+/', '', $month);
        $year = preg_replace('/\D+/', '', $year);
        $amountCents = preg_replace('/\D+/', '', $amountCents);

        // Formatação conforme documentação BCI (atualizada)
        // número: 7 dígitos, mês: 2 dígitos, ano: 2 dígitos
        $number = str_pad($number, 7, '0', STR_PAD_LEFT);
        $month = str_pad($month, 2, '0', STR_PAD_LEFT);
        $year = str_pad($year, 2, '0', STR_PAD_LEFT);

        // Concatenar: entidade + número + mês + ano + valor
        $concat = $entity . $number . $month . $year . $amountCents;

        // Algoritmo BCI para cálculo do check digit
        $Pi = 0;
        $n = strlen($concat);

        for ($i = 0; $i < $n; $i++) {
            $digit = (int)$concat[$i];
            $Si = $digit + $Pi;
            $Pi = ($Si * 10) % 97;
        }

        $Pn = ($Pi * 10) % 97;
        $checkDigit = 98 - $Pn;
        $checkDigitFormatted = str_pad((string)$checkDigit, 2, '0', STR_PAD_LEFT);

        // Referência final: número + mês + ano + check digit (13 dígitos)
        $reference = $number . $month . $year . $checkDigitFormatted;

        Log::debug('BCI Reference generated', [
            'entity' => $entity,
            'number' => $number,
            'month' => $month,
            'year' => $year,
            'amount_cents' => $amountCents,
            'check_digit' => $checkDigitFormatted,
            'reference' => $reference
        ]);

        return $reference;
    }

    /**
     * Calcular multa
     */
    private function calculateFine(float $baseAmount, string $type, float $value): float
    {
        if ($value <= 0) {
            return 0.0;
        }

        if (strtolower($type) === 'percentage') {
            return round($baseAmount * ($value / 100), 2);
        }

        // tipo fixed
        return round($value, 2);
    }

    /**
     * Dia 5 do mês SEGUINTE ao da taxa
     */
    private function cutoffDay5OfNextMonth(string $month, int $year): Carbon
    {
        $monthNum = $this->getMonthNumber($month);
        return Carbon::create($year, $monthNum, 5)
            ->addMonthNoOverflow()
            ->setTime(23, 59, 59);
    }

    /**
     * Próximo dia 5 a partir de uma data
     */
    private function nextCutoffFrom(Carbon $from): Carbon
    {
        $candidate = Carbon::create($from->year, $from->month, 5, 23, 59, 59);

        if ($from->day > 5) {
            $candidate = $candidate->addMonthNoOverflow();
        }

        return $candidate;
    }

    /**
     * Criar código único para o estudante (7 dígitos)
     */
    private function createStudentCode($student): string
    {
        // Extrair ID numérico
        $studentId = $student->student_id ?? $student->id ?? '';
        $numericPart = preg_replace('/[^0-9]/', '', (string)$studentId);
        
        // Extrair ano de nascimento
        $birthYear = $this->extractBirthYear($student);
        
        // Combinar: ID + últimos 2 dígitos do ano
        $yearPart = substr($birthYear, -2);
        $combined = $numericPart . $yearPart;

        // Ajustar para 7 dígitos
        if (strlen($combined) > 7) {
            $final = substr($combined, -7);
        } else {
            $final = str_pad($combined, 7, '0', STR_PAD_LEFT);
        }

        return $final;
    }

    /**
     * Extrair ano de nascimento do estudante
     */
    private function extractBirthYear($student): string
    {
        $dobFields = ['dob', 'date_of_birth', 'birth_date', 'birthday'];

        foreach ($dobFields as $field) {
            if (isset($student->$field) && !empty($student->$field)) {
                try {
                    $date = Carbon::parse($student->$field);
                    return $date->format('Y');
                } catch (\Exception $e) {
                    continue;
                }
            }
        }

        // Fallback para ano atual se não encontrar
        return date('Y');
    }

    /**
     * Sanitizar ID do estudante (garantir 7 dígitos)
     * Usa hash SHA-256 para evitar colisões em IDs não numéricos
     */
    private function sanitizeStudentId(string $studentId): string
    {
        // Extrair apenas números
        $numericOnly = preg_replace('/\D+/', '', $studentId);

        // Se não houver números, usar hash SHA-256 mais robusto
        if (empty($numericOnly)) {
            // Gerar hash SHA-256 do ID original
            $hash = hash('sha256', $studentId);

            // Converter primeiros 16 caracteres do hash de hex para decimal
            $numericOnly = substr(
                preg_replace('/\D/', '', base_convert(substr($hash, 0, 16), 16, 10)),
                0,
                7
            );

            Log::warning('Non-numeric student ID converted using SHA-256', [
                'original_id' => $studentId,
                'sanitized_id' => $numericOnly,
                'hash_prefix' => substr($hash, 0, 16)
            ]);
        }

        // Garantir 7 dígitos
        if (strlen($numericOnly) > 7) {
            $sanitized = substr($numericOnly, -7);
        } else {
            $sanitized = str_pad($numericOnly, 7, '0', STR_PAD_LEFT);
        }

        return $sanitized;
    }

    /**
     * Converter nome do mês para número
     */
    private function getMonthNumber(string $monthName): int
    {
        $months = [
            // English
            'january' => 1, 'february' => 2, 'march' => 3, 'april' => 4,
            'may' => 5, 'june' => 6, 'july' => 7, 'august' => 8,
            'september' => 9, 'october' => 10, 'november' => 11, 'december' => 12,
            // Portuguese
            'janeiro' => 1, 'fevereiro' => 2, 'março' => 3, 'abril' => 4,
            'maio' => 5, 'junho' => 6, 'julho' => 7, 'agosto' => 8,
            'setembro' => 9, 'outubro' => 10, 'novembro' => 11, 'dezembro' => 12,
            // Abbreviated
            'jan' => 1, 'feb' => 2, 'mar' => 3, 'apr' => 4,
            'may' => 5, 'jun' => 6, 'jul' => 7, 'aug' => 8,
            'sep' => 9, 'oct' => 10, 'nov' => 11, 'dec' => 12,
        ];

        $normalized = strtolower(trim($monthName));
        
        // Tentar encontrar o mês no array
        if (isset($months[$normalized])) {
            return $months[$normalized];
        }
        
        // Tentar interpretar como número
        if (is_numeric($monthName)) {
            $num = (int)$monthName;
            if ($num >= 1 && $num <= 12) {
                return $num;
            }
        }
        
        // Default para Janeiro se não reconhecer
        Log::warning('Month not recognized, defaulting to January', ['month' => $monthName]);
        return 1;
    }

    /**
     * Verificar se referência já existe no banco de dados
     */
    private function referenceExists(string $reference): bool
    {
        return \App\Models\PaymentReference::where('reference_number', $reference)->exists();
    }

    /**
     * Validar referência
     */
    public function validate(string $reference): bool
    {
        // Deve ter apenas dígitos
        if (!ctype_digit($reference)) {
            return false;
        }

        // Aceitar 11 dígitos (formato antigo) ou 13 dígitos (formato novo com ano)
        $length = strlen($reference);
        if ($length !== 11 && $length !== 13) {
            return false;
        }

        // Não pode ser tudo zeros
        if ($reference === str_repeat('0', $length)) {
            return false;
        }

        return true;
    }

    /**
     * Formatar referência para exibição
     */
    public function format(string $reference): string
    {
        $length = strlen($reference);

        if ($length === 13) {
            // Formato novo: XXX XXX XXX XX XX (7 número + 2 mês + 2 ano + 2 check)
            return substr($reference, 0, 3) . ' ' .
                   substr($reference, 3, 3) . ' ' .
                   substr($reference, 6, 3) . ' ' .
                   substr($reference, 9, 2) . ' ' .
                   substr($reference, 11, 2);
        }

        if ($length === 11) {
            // Formato antigo: XXX XXX XXX XX (7 número + 2 mês + 2 check)
            return substr($reference, 0, 3) . ' ' .
                   substr($reference, 3, 3) . ' ' .
                   substr($reference, 6, 3) . ' ' .
                   substr($reference, 9, 2);
        }

        if ($length === 9) {
            // Formato: XXX XXX XXX
            return substr($reference, 0, 3) . ' ' .
                   substr($reference, 3, 3) . ' ' .
                   substr($reference, 6, 3);
        }

        // Formato genérico para outros tamanhos
        $parts = [];
        for ($i = 0; $i < $length; $i += 3) {
            $parts[] = substr($reference, $i, 3);
        }

        return implode(' ', $parts);
    }

    /**
     * Teste com exemplo da documentação BCI (formato atualizado com ano)
     * Ano 2025 (25) incluído para unicidade
     */
    public function testWithDocumentExample(): array
    {
        $entity = '54321';
        $number = '1212245';
        $month = '07';
        $year = '25'; // 2025
        $amountCents = '200000'; // 2000,00 MT

        $reference = $this->generateBCIReference($entity, $number, $month, $year, $amountCents);

        return [
            'input' => [
                'entity' => $entity,
                'number' => $number,
                'month' => $month,
                'year' => $year,
                'amount_cents' => $amountCents,
            ],
            'reference' => $reference,
            'length' => strlen($reference),
            'expected_length' => 13,
            'formatted' => $this->format($reference),
            'is_valid' => $this->validate($reference),
            'breakdown' => [
                'student_number' => substr($reference, 0, 7),
                'month' => substr($reference, 7, 2),
                'year' => substr($reference, 9, 2),
                'check_digit' => substr($reference, 11, 2),
            ]
        ];
    }

    /**
     * Teste de unicidade: mesmo estudante, mesmo mês, anos diferentes
     */
    public function testUniquenessAcrossYears(): array
    {
        $entity = '11111';
        $number = '1234567';
        $month = '01'; // Janeiro
        $amount = '50000'; // 500 MT

        $ref2024 = $this->generateBCIReference($entity, $number, $month, '24', $amount);
        $ref2025 = $this->generateBCIReference($entity, $number, $month, '25', $amount);
        $ref2026 = $this->generateBCIReference($entity, $number, $month, '26', $amount);

        return [
            'student' => $number,
            'month' => 'Janeiro',
            'amount' => '500 MT',
            'references' => [
                '2024' => [
                    'reference' => $ref2024,
                    'formatted' => $this->format($ref2024),
                ],
                '2025' => [
                    'reference' => $ref2025,
                    'formatted' => $this->format($ref2025),
                ],
                '2026' => [
                    'reference' => $ref2026,
                    'formatted' => $this->format($ref2026),
                ],
            ],
            'all_unique' => (
                $ref2024 !== $ref2025 &&
                $ref2025 !== $ref2026 &&
                $ref2024 !== $ref2026
            ),
            'test_passed' => (
                $ref2024 !== $ref2025 &&
                $ref2025 !== $ref2026 &&
                $ref2024 !== $ref2026
            )
        ];
    }

    // ========================================================================
    // MÉTODOS V1 (LEGADO) - 11 DÍGITOS SEM ANO
    // Mantidos para compatibilidade com código existente
    // ========================================================================

    /**
     * Gera referência no formato V1 (11 dígitos SEM ano)
     * Garante unicidade incrementando o número base se necessário
     *
     * @param string $month Nome do mês
     * @param string $studentId ID do estudante
     * @param float $amount Valor
     * @return string Referência de 11 dígitos
     */
    public function makeV1(string $month, string $studentId, float $amount): string
    {
        $entity = config('payments.entity', '90013');
        $monthNumber = str_pad((string)$this->getMonthNumber($month), 2, '0', STR_PAD_LEFT);
        $amountCents = (string)round($amount * 100);
        $studentNumber = $this->sanitizeStudentId($studentId);

        // Gerar referência base
        $baseReference = $this->generateBCIReferenceV1($entity, $studentNumber, $monthNumber, $amountCents);

        // Verificar se já existe e adicionar variação se necessário
        $reference = $baseReference;
        $attempt = 0;
        $maxAttempts = 9999999; // 7 dígitos = até ~10 milhões de variações

        // Converter studentNumber para inteiro para incrementar
        $baseNumber = (int)$studentNumber;

        while ($this->referenceExists($reference) && $attempt < $maxAttempts) {
            $attempt++;

            // Incrementar o número do estudante e manter 7 dígitos
            $modifiedNumber = ($baseNumber + $attempt) % 10000000;
            $modifiedStudentNumber = str_pad((string)$modifiedNumber, 7, '0', STR_PAD_LEFT);

            $reference = $this->generateBCIReferenceV1($entity, $modifiedStudentNumber, $monthNumber, $amountCents);

            // Log apenas a cada 100 tentativas para não encher o log
            if ($attempt % 100 === 0) {
                Log::info('Reference V1 collision - still searching', [
                    'attempt' => $attempt,
                    'original_student_number' => $studentNumber,
                    'current_student_number' => $modifiedStudentNumber,
                ]);
            }
        }

        if ($attempt >= $maxAttempts) {
            throw new \RuntimeException('Não foi possível gerar referência única após ' . $maxAttempts . ' tentativas');
        }

        if ($attempt > 0) {
            Log::info('Reference V1 generated after collision resolution', [
                'attempts' => $attempt,
                'original_reference' => $baseReference,
                'final_reference' => $reference,
            ]);
        }

        return $reference;
    }

    /**
     * Gera referência V1 (11 dígitos) a partir de objeto Student
     * Formato compatível com BCI/BMEPS
     * Garante unicidade mesmo para múltiplas referências do mesmo estudante/mês
     *
     * @param string $month Nome do mês
     * @param object $student Objeto do estudante
     * @param float $amount Valor
     * @return string Referência de 11 dígitos
     */
    public function makeV1FromStudent(string $month, $student, float $amount): string
    {
        $studentCode = $this->createStudentCode($student);
        return $this->makeV1($month, $studentCode, $amount);
    }

    /**
     * Algoritmo BCI original - 11 dígitos SEM ano
     *
     * @param string $entity Código da entidade
     * @param string $number Número do estudante (7 dígitos)
     * @param string $month Mês (2 dígitos)
     * @param string $amountCents Valor em centavos
     * @return string Referência de 11 dígitos
     */
    private function generateBCIReferenceV1(string $entity, string $number, string $month, string $amountCents): string
    {
        // Limpar entrada - apenas dígitos
        $entity = preg_replace('/\D+/', '', $entity);
        $number = preg_replace('/\D+/', '', $number);
        $month = preg_replace('/\D+/', '', $month);
        $amountCents = preg_replace('/\D+/', '', $amountCents);

        // Formatação
        $number = str_pad($number, 7, '0', STR_PAD_LEFT);
        $month = str_pad($month, 2, '0', STR_PAD_LEFT);

        // Concatenar SEM ANO (formato original)
        $concat = $entity . $number . $month . $amountCents;

        // Algoritmo BCI para cálculo do check digit
        $Pi = 0;
        $n = strlen($concat);

        for ($i = 0; $i < $n; $i++) {
            $digit = (int)$concat[$i];
            $Si = $digit + $Pi;
            $Pi = ($Si * 10) % 97;
        }

        $Pn = ($Pi * 10) % 97;
        $checkDigit = 98 - $Pn;
        $checkDigitFormatted = str_pad((string)$checkDigit, 2, '0', STR_PAD_LEFT);

        // Referência final SEM ANO: número + mês + check digit (11 dígitos)
        $reference = $number . $month . $checkDigitFormatted;

        Log::debug('BCI Reference V1 generated (11 digits, no year)', [
            'entity' => $entity,
            'number' => $number,
            'month' => $month,
            'amount_cents' => $amountCents,
            'check_digit' => $checkDigitFormatted,
            'reference' => $reference,
            'version' => 'V1'
        ]);

        return $reference;
    }

    /**
     * Emissão V1 - 11 dígitos sem ano (formato legado)
     *
     * @deprecated Use issue() que já usa V2
     */
    public function issueV1(string $month, $student, float $baseAmount, array $options = []): array
    {
        $entity = $options['entity'] ?? config('payments.entity', '11111');

        if (is_object($student)) {
            $studentNumber = $this->createStudentCode($student);
        } else {
            $studentNumber = $this->sanitizeStudentId((string)$student);
        }

        $monthNumber = str_pad((string)$this->getMonthNumber($month), 2, '0', STR_PAD_LEFT);

        // Usar ano atual apenas para cálculo de cutoff, NÃO para referência
        $year = $options['year'] ?? now()->year;
        $cutoff = $this->cutoffDay5OfNextMonth($month, $year);

        if (!empty($options['expires_at'])) {
            try {
                $cutoff = Carbon::parse($options['expires_at'])->setTime(23, 59, 59);
            } catch (\Throwable $e) {
                // mantém o cutoff padrão
            }
        }

        $needsFine = now()->greaterThan($cutoff);
        $fineAmount = 0.0;

        if ($needsFine) {
            $fineType = $options['fine_type'] ?? config('payments.fine.type', 'fixed');
            $fineValue = $options['fine_value'] ?? config('payments.fine.value', 0);
            $fineAmount = $this->calculateFine($baseAmount, $fineType, (float)$fineValue);
        }

        $totalAmount = round($baseAmount + $fineAmount, 2);
        $expiresAt = $needsFine ? $this->nextCutoffFrom(now()) : $cutoff;
        $ttlRaw = now()->diffInDays($expiresAt, false);
        $ttlDays = max(0, $ttlRaw);
        $status = $ttlDays > 0 ? 'pending' : 'expired';

        // Gerar referência V1 (SEM ANO)
        $amountCents = (string)round($totalAmount * 100);
        $reference = $this->generateBCIReferenceV1($entity, $studentNumber, $monthNumber, $amountCents);

        $shouldExpirePrevious = $needsFine;

        Log::info('Reference V1 issued (11 digits, no year)', [
            'entity' => $entity,
            'reference' => $reference,
            'month' => $month,
            'year' => $year,
            'amounts' => [
                'base' => $baseAmount,
                'fine' => $fineAmount,
                'total' => $totalAmount
            ],
            'expires_at' => $expiresAt->toDateTimeString(),
            'status' => $status,
            'version' => 'V1'
        ]);

        return [
            'entity' => $entity,
            'reference' => $reference,
            'base_amount' => $baseAmount,
            'fine_amount' => $fineAmount,
            'total_amount' => $totalAmount,
            'expires_at' => $expiresAt,
            'ttl_days' => $ttlDays,
            'status' => $status,
            'month' => $month,
            'year' => $year,
            'should_expire_previous' => $shouldExpirePrevious,
            'version' => 'V1',
            'format' => '11_digits_no_year'
        ];
    }

    // ========================================================================
    // MÉTODOS V2 (NOVO) - 13 DÍGITOS COM ANO
    // Usa os métodos já implementados acima (make, issue, generateBCIReference)
    // ========================================================================

    /**
     * Alias explícito para make() - gera referência V2 (13 dígitos COM ano)
     */
    public function makeV2(string $month, int $year, string $studentId, float $amount): string
    {
        return $this->make($month, $year, $studentId, $amount);
    }

    /**
     * Alias explícito para issue() - emissão V2 (13 dígitos COM ano)
     */
    public function issueV2(string $month, int $year, $student, float $baseAmount, array $options = []): array
    {
        $result = $this->issue($month, $year, $student, $baseAmount, $options);
        $result['version'] = 'V2';
        $result['format'] = '13_digits_with_year';
        return $result;
    }

    /**
     * Teste comparativo V1 vs V2
     */
    public function testV1vsV2(): array
    {
        $studentId = '202401234';
        $month = 'Janeiro';
        $amount = 500.00;

        // V1 - 11 dígitos (mesmo estudante, mesmo mês, anos diferentes = MESMA referência)
        $refV1_2024 = $this->makeV1($month, $studentId, $amount);
        $refV1_2025 = $this->makeV1($month, $studentId, $amount);
        $refV1_2026 = $this->makeV1($month, $studentId, $amount);

        // V2 - 13 dígitos (mesmo estudante, mesmo mês, anos diferentes = referências DIFERENTES)
        $refV2_2024 = $this->makeV2($month, 2024, $studentId, $amount);
        $refV2_2025 = $this->makeV2($month, 2025, $studentId, $amount);
        $refV2_2026 = $this->makeV2($month, 2026, $studentId, $amount);

        return [
            'student_id' => $studentId,
            'month' => $month,
            'amount' => $amount,

            'v1_results' => [
                'format' => '11 digits (no year)',
                'references' => [
                    '2024' => $refV1_2024,
                    '2025' => $refV1_2025,
                    '2026' => $refV1_2026,
                ],
                'formatted' => [
                    '2024' => $this->format($refV1_2024),
                    '2025' => $this->format($refV1_2025),
                    '2026' => $this->format($refV1_2026),
                ],
                'all_same' => ($refV1_2024 === $refV1_2025 && $refV1_2025 === $refV1_2026),
                'problem' => 'DUPLICADAS - mesmo código para anos diferentes!'
            ],

            'v2_results' => [
                'format' => '13 digits (with year)',
                'references' => [
                    '2024' => $refV2_2024,
                    '2025' => $refV2_2025,
                    '2026' => $refV2_2026,
                ],
                'formatted' => [
                    '2024' => $this->format($refV2_2024),
                    '2025' => $this->format($refV2_2025),
                    '2026' => $this->format($refV2_2026),
                ],
                'all_unique' => (
                    $refV2_2024 !== $refV2_2025 &&
                    $refV2_2025 !== $refV2_2026 &&
                    $refV2_2024 !== $refV2_2026
                ),
                'solution' => 'ÚNICAS - ano incluído garante unicidade!'
            ],

            'recommendation' => 'Use V2 (makeV2, issueV2) para novas referências. V1 mantido apenas para compatibilidade.'
        ];
    }
}
