<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('payment_references', function (Blueprint $table) {
            // Data de início do período de pagamento (quando pode começar a pagar)
            $table->timestamp('payment_starts_at')->nullable()->after('valid_from')
                ->comment('When payment period starts');

            // Data de vencimento (quando começa a aplicar multa)
            $table->timestamp('due_date')->nullable()->after('payment_starts_at')
                ->comment('Payment due date - penalty applies after this date');

            // Tipo de multa (percentage ou fixed)
            $table->string('penalty_type', 20)->nullable()->after('due_date')
                ->comment('Penalty type: percentage or fixed');

            // Taxa/valor da multa
            $table->decimal('penalty_rate', 8, 2)->nullable()->after('penalty_type')
                ->comment('Penalty rate (percentage value or fixed amount)');

            // Indica se esta referência já tem multa aplicada
            $table->boolean('has_penalty')->default(false)->after('penalty_rate')
                ->comment('Indicates if this reference already has penalty applied');

            // ID da referência original (se esta for uma ref com multa)
            $table->unsignedBigInteger('original_reference_id')->nullable()->after('has_penalty')
                ->comment('Original reference ID if this is a penalty reference');

            // Foreign key para a referência original
            $table->foreign('original_reference_id')
                  ->references('id')
                  ->on('payment_references')
                  ->onDelete('set null');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('payment_references', function (Blueprint $table) {
            $table->dropForeign(['original_reference_id']);
            $table->dropColumn([
                'payment_starts_at',
                'due_date',
                'penalty_type',
                'penalty_rate',
                'has_penalty',
                'original_reference_id'
            ]);
        });
    }
};
