<?php

/**
 * Script de debug para investigar problema de certificados/declarações
 * Verifica os dados de um estudante específico
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use App\Models\User;
use App\Models\Classroom;
use App\Models\Exam;
use App\Models\TeacherSubject;

echo "=== DEBUG: CERTIFICADOS/DECLARAÇÕES ===\n\n";

// Pegar turma 9a A
$classroom = Classroom::where('class', 9)->where('name', 'like', '%A%')->first();

if (!$classroom) {
    echo "❌ Turma 9a A não encontrada!\n";
    exit;
}

echo "✓ Turma encontrada: {$classroom->name} (ID: {$classroom->id}, Classe: {$classroom->class})\n\n";

// Pegar primeiro estudante dessa turma
$student = User::where('classroom_id', $classroom->id)
    ->where('is_active', 1)
    ->first();

if (!$student) {
    echo "❌ Nenhum estudante ativo encontrado nesta turma!\n";
    exit;
}

echo "✓ Estudante: {$student->name} (ID: {$student->id})\n";
echo "  - Classroom ID: {$student->classroom_id}\n";
echo "  - Status: " . ($student->is_active ? 'Ativo' : 'Inativo') . "\n\n";

// Verificar disciplinas da turma
$teacherSubjects = TeacherSubject::where('classroom_id', $classroom->id)
    ->with('subject')
    ->get();

echo "📚 Disciplinas configuradas para esta turma: " . $teacherSubjects->count() . "\n";
$subjectIds = [];
foreach ($teacherSubjects as $ts) {
    if ($ts->subject) {
        echo "  - {$ts->subject->name} (ID: {$ts->subject->id})\n";
        $subjectIds[] = $ts->subject->id;
    }
}
echo "\n";

// Verificar exames do estudante
echo "📝 Verificando exames do estudante...\n";

// Verificar que anos existem
$yearsInDB = Exam::where('student_id', $student->id)
    ->distinct()
    ->pluck('year')
    ->sort();

echo "  Anos de exames na BD para este estudante: " . $yearsInDB->implode(', ') . "\n\n";

// Para cada ano, mostrar exames
foreach ($yearsInDB as $year) {
    echo "--- Ano {$year} ---\n";

    $examsForYear = Exam::where('student_id', $student->id)
        ->where('year', $year)
        ->get();

    echo "  Total de exames: {$examsForYear->count()}\n";

    // Agrupar por trimestre
    foreach ([1, 2, 3] as $trim) {
        $trimExams = $examsForYear->where('trimester_id', $trim);
        echo "  - Trimestre {$trim}: {$trimExams->count()} exames\n";

        if ($trimExams->count() > 0) {
            $firstExam = $trimExams->first();
            echo "    * Classroom ID dos exames: {$firstExam->classroom_id}\n";
            echo "    * Exemplo: Subject ID {$firstExam->subject_id}, ACS1a={$firstExam->ACS1a}, ACS2a={$firstExam->ACS2a}, AT={$firstExam->AT}\n";
        }
    }
    echo "\n";
}

// Testar a query que está sendo usada no CertificateService
echo "🔍 Testando query do CertificateService...\n";
$year = 2025;

$allExams = Exam::where('student_id', $student->id)
    ->where('classroom_id', $classroom->id)
    ->where(function($query) use ($year) {
        $query->where('year', $year)
              ->orWhere('year', $year - 1);
    })
    ->whereIn('subject_id', $subjectIds)
    ->get();

echo "  Query: student_id={$student->id}, classroom_id={$classroom->id}, year IN (2025, 2024)\n";
echo "  Subject IDs: " . implode(', ', $subjectIds) . "\n";
echo "  Resultado: {$allExams->count()} exames encontrados\n\n";

if ($allExams->count() == 0) {
    echo "⚠️ PROBLEMA IDENTIFICADO: A query não encontra exames!\n\n";

    echo "Possíveis causas:\n";
    echo "1. Os exames têm um ano diferente (verifique anos acima)\n";
    echo "2. Os exames estão associados a outro classroom_id\n";
    echo "3. Os subject_ids não correspondem\n\n";

    // Tentar query sem filtro de ano
    echo "Testando sem filtro de ano...\n";
    $examsNoYear = Exam::where('student_id', $student->id)
        ->where('classroom_id', $classroom->id)
        ->whereIn('subject_id', $subjectIds)
        ->count();
    echo "  Resultado: {$examsNoYear} exames\n\n";

    // Tentar query sem filtro de classroom
    echo "Testando sem filtro de classroom...\n";
    $examsNoClassroom = Exam::where('student_id', $student->id)
        ->where(function($query) use ($year) {
            $query->where('year', $year)
                  ->orWhere('year', $year - 1);
        })
        ->whereIn('subject_id', $subjectIds)
        ->count();
    echo "  Resultado: {$examsNoClassroom} exames\n\n";

    // Tentar query só com student_id
    echo "Testando só com student_id...\n";
    $examsOnlyStudent = Exam::where('student_id', $student->id)->count();
    echo "  Resultado: {$examsOnlyStudent} exames\n\n";

} else {
    echo "✓ Query encontrou exames!\n";
    echo "  Agrupados por disciplina/trimestre:\n";

    $grouped = $allExams->groupBy(function ($exam) {
        return $exam->subject_id . '_' . $exam->trimester_id;
    });

    foreach ($grouped as $key => $exams) {
        $exam = $exams->first();
        echo "  - Subject {$exam->subject_id}, Trimestre {$exam->trimester_id}: ACS1a={$exam->ACS1a}, ACS2a={$exam->ACS2a}, AT={$exam->AT}, NE={$exam->NE}\n";
    }
}

echo "\n=== FIM DEBUG ===\n";
