<?php

namespace App\Console\Commands;

use App\Models\Exam;
use App\Models\User;
use App\Models\Subject;
use App\Models\Trimester;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class AnalyzeDuplicateExams extends Command
{
    protected $signature = 'exams:analyze-duplicates {--year=} {--student-id=} {--subject-id=} {--trimester-id=} {--show-details}';
    protected $description = 'Analyze duplicate exam records and show which one has the most complete marks';

    public function handle()
    {
        $year = $this->option('year') ?: date('Y');
        $studentId = $this->option('student-id');
        $subjectId = $this->option('subject-id');
        $trimesterId = $this->option('trimester-id');
        $showDetails = $this->option('show-details');

        $this->info("Analyzing duplicate exam records for year: {$year}");
        $this->line("Grouping by: student_id + subject_id + trimester_id + year (regardless of classroom)");
        $this->newLine();

        // Build the query
        $query = Exam::where('year', $year)->whereNull('deleted_at');

        if ($studentId) {
            $query->where('student_id', $studentId);
        }
        if ($subjectId) {
            $query->where('subject_id', $subjectId);
        }
        if ($trimesterId) {
            $query->where('trimester_id', $trimesterId);
        }

        $exams = $query->with(['student', 'subject'])->get();

        // Group by student + subject + trimester + year (ignore classroom)
        $duplicateGroups = $exams->groupBy(function ($exam) {
            return $exam->student_id . '_' . $exam->subject_id . '_' . $exam->trimester_id . '_' . $exam->year;
        });

        $totalDuplicateGroups = 0;
        $totalDuplicateRecords = 0;
        $analysisResults = [];

        foreach ($duplicateGroups as $groupKey => $group) {
            if ($group->count() > 1) {
                $totalDuplicateGroups++;
                $totalDuplicateRecords += $group->count() - 1; // -1 because we keep one

                $analysis = $this->analyzeGroup($group);
                $analysisResults[] = $analysis;

                if ($showDetails) {
                    $this->displayGroupAnalysis($analysis);
                }
            }
        }

        // Summary
        $this->displaySummary($totalDuplicateGroups, $totalDuplicateRecords, $analysisResults);

        return 0;
    }

    private function analyzeGroup($group)
    {
        $firstExam = $group->first();
        $analysis = [
            'group_key' => $firstExam->student_id . '_' . $firstExam->subject_id . '_' . $firstExam->trimester_id . '_' . $firstExam->year,
            'student_name' => $firstExam->student->name ?? 'Unknown',
            'subject_name' => $firstExam->subject->name ?? 'Unknown',
            'trimester_id' => $firstExam->trimester_id,
            'year' => $firstExam->year,
            'total_records' => $group->count(),
            'records' => [],
            'recommended_action' => '',
            'best_record_id' => null,
            'reasoning' => ''
        ];

        foreach ($group as $exam) {
            $markCount = 0;
            $marks = [];

            // Count non-zero marks
            if ($exam->ACS1a > 0) {
                $markCount++;
                $marks['ACS1a'] = $exam->ACS1a;
            }
            if ($exam->ACS2a > 0) {
                $markCount++;
                $marks['ACS2a'] = $exam->ACS2a;
            }
            if ($exam->AT > 0) {
                $markCount++;
                $marks['AT'] = $exam->AT;
            }
            if ($exam->NE > 0) {
                $markCount++;
                $marks['NE'] = $exam->NE;
            }
            if ($exam->MACS > 0) {
                $marks['MACS'] = $exam->MACS;
            }
            if ($exam->MT > 0) {
                $marks['MT'] = $exam->MT;
            }

            $analysis['records'][] = [
                'id' => $exam->id,
                'classroom_id' => $exam->classroom_id,
                'mark_count' => $markCount,
                'marks' => $marks,
                'created_at' => $exam->created_at,
                'updated_at' => $exam->updated_at,
                'is_latest' => false // Will be set later
            ];
        }

        // Sort records by mark count (descending), then by updated_at (descending)
        usort($analysis['records'], function ($a, $b) {
            if ($a['mark_count'] == $b['mark_count']) {
                return $b['updated_at'] <=> $a['updated_at'];
            }
            return $b['mark_count'] <=> $a['mark_count'];
        });

        // Mark the latest record
        $latestRecord = collect($analysis['records'])->sortByDesc('updated_at')->first();
        foreach ($analysis['records'] as &$record) {
            if ($record['id'] == $latestRecord['id']) {
                $record['is_latest'] = true;
                break;
            }
        }

        // Determine best record and recommendation
        $bestRecord = $analysis['records'][0]; // Record with most marks
        $analysis['best_record_id'] = $bestRecord['id'];

        if ($bestRecord['mark_count'] > $latestRecord['mark_count']) {
            $analysis['recommended_action'] = 'KEEP_MOST_COMPLETE';
            $analysis['reasoning'] = "Record ID {$bestRecord['id']} has {$bestRecord['mark_count']} marks vs latest record ({$latestRecord['id']}) with {$latestRecord['mark_count']} marks";
        } elseif ($bestRecord['mark_count'] == $latestRecord['mark_count'] && $bestRecord['id'] == $latestRecord['id']) {
            $analysis['recommended_action'] = 'KEEP_LATEST';
            $analysis['reasoning'] = "Latest record also has the most complete marks ({$bestRecord['mark_count']} marks)";
        } else {
            $analysis['recommended_action'] = 'REVIEW_MANUALLY';
            $analysis['reasoning'] = "Multiple records with equal mark counts ({$bestRecord['mark_count']} marks) - manual review needed";
        }

        return $analysis;
    }

    private function displayGroupAnalysis($analysis)
    {
        $this->line("<fg=yellow>━━━ Duplicate Group Analysis ━━━</>");
        $this->line("Student: <fg=cyan>{$analysis['student_name']}</>");
        $this->line("Subject: <fg=cyan>{$analysis['subject_name']}</>");
        $this->line("Trimester: <fg=cyan>{$analysis['trimester_id']}</> | Year: <fg=cyan>{$analysis['year']}</>");
        $this->line("Total Records: <fg=red>{$analysis['total_records']}</>");
        $this->newLine();

        $this->line("<fg=magenta>Records Details:</>");
        $headers = ['ID', 'Classroom', 'Marks Count', 'Marks Detail', 'Created', 'Updated', 'Latest?'];
        $rows = [];

        foreach ($analysis['records'] as $record) {
            $marksDetail = [];
            foreach ($record['marks'] as $type => $value) {
                $marksDetail[] = "{$type}:{$value}";
            }

            $rows[] = [
                $record['id'],
                $record['classroom_id'],
                $record['mark_count'],
                implode(', ', $marksDetail),
                $record['created_at']->format('Y-m-d H:i'),
                $record['updated_at']->format('Y-m-d H:i'),
                $record['is_latest'] ? '✓' : ''
            ];
        }

        $this->table($headers, $rows);

        $this->line("<fg=green>Recommendation: {$analysis['recommended_action']}</>");
        $this->line("<fg=white>Reasoning: {$analysis['reasoning']}</>");
        $this->line("<fg=blue>Best Record ID: {$analysis['best_record_id']}</>");

        $this->newLine();
        $this->line("<fg=gray>━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━</>");
        $this->newLine();
    }

    private function displaySummary($totalGroups, $totalDuplicates, $analysisResults)
    {
        $this->newLine();
        $this->line("<fg=green>━━━ SUMMARY ━━━</>");
        $this->line("Total Duplicate Groups: <fg=red>{$totalGroups}</>");
        $this->line("Total Duplicate Records to Clean: <fg=red>{$totalDuplicates}</>");
        $this->newLine();

        // Count recommendations
        $recommendations = collect($analysisResults)->groupBy('recommended_action');

        $this->line("<fg=yellow>Cleanup Recommendations:</>");
        foreach ($recommendations as $action => $groups) {
            $count = $groups->count();
            $this->line("  {$action}: <fg=cyan>{$count}</> groups");
        }

        $this->newLine();

        // Show potential issues
        $manualReviewNeeded = $recommendations->get('REVIEW_MANUALLY', collect())->count();
        $keepMostComplete = $recommendations->get('KEEP_MOST_COMPLETE', collect())->count();

        if ($manualReviewNeeded > 0) {
            $this->warn("⚠️  {$manualReviewNeeded} groups need manual review due to equal mark counts");
        }

        if ($keepMostComplete > 0) {
            $this->info("ℹ️  {$keepMostComplete} groups should keep the most complete record (not the latest)");
        }

        $this->newLine();
        $this->line("<fg=blue>Suggested Commands:</>");
        $this->line("  • Run with --show-details to see detailed analysis");
        $this->line("  • Filter by specific student: --student-id=STUDENT_ID");
        $this->line("  • Filter by specific subject: --subject-id=SUBJECT_ID");
        $this->line("  • Filter by specific trimester: --trimester-id=TRIMESTER_ID");
    }
}
