<?php

namespace App\Console\Commands;

use App\Services\PaymentReconciliationService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Exception;

class ReconcileBMEPSPayments extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'payments:reconcile-bmeps 
                            {--file= : Process specific file}
                            {--dry-run : Run without making changes}
                            ';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Reconcile BMEPS payment files from SFTP server';

    /**
     * @var PaymentReconciliationService
     */
    protected $reconciliationService;

    /**
     * Create a new command instance.
     */
    public function __construct(PaymentReconciliationService $reconciliationService)
    {
        parent::__construct();
        $this->reconciliationService = $reconciliationService;
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $startTime = Carbon::now();
        
        $this->info('========================================');
        $this->info('BMEPS Payment Reconciliation');
        $this->info('Started at: ' . $startTime->format('Y-m-d H:i:s'));
        $this->info('========================================');
        
        try {
            // Check for dry run mode
            if ($this->option('dry-run')) {
                $this->warn('Running in DRY RUN mode - no changes will be made');
                Log::info('BMEPS reconciliation started in DRY RUN mode');
            }
            
            // Process specific file or all pending files
            if ($file = $this->option('file')) {
                $this->info("Processing specific file: $file");
                $stats = $this->reconciliationService->processFile($file);
            } else {
                $this->info('Processing all pending files from SFTP...');
                $stats = $this->reconciliationService->processAllPendingFiles();
            }
            
            // Display results
            $this->displayResults($stats, $startTime);
            
            return Command::SUCCESS;
            
        } catch (Exception $e) {
            $this->error('Reconciliation failed: ' . $e->getMessage());
            
            if ($this->option('verbose')) {
                $this->error('Stack trace:');
                $this->error($e->getTraceAsString());
            }
            
            Log::error('BMEPS reconciliation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Command::FAILURE;
        }
    }

    /**
     * Display reconciliation results
     */
    private function displayResults($stats, $startTime)
    {
        $endTime = Carbon::now();
        $duration = $endTime->diffInSeconds($startTime);
        
        $this->info('');
        $this->info('========================================');
        $this->info('Reconciliation Results');
        $this->info('========================================');
        
        $this->table(
            ['Metric', 'Value'],
            [
                ['Files Processed', $stats['total_files'] ?? 0],
                ['Total Records', $stats['total_records'] ?? 0],
                ['Successfully Reconciled', $stats['reconciled'] ?? 0],
                ['Failed', $stats['failed'] ?? 0],
                ['Duplicates', $stats['duplicates'] ?? 0],
                ['References Not Found', $stats['not_found'] ?? 0],
                ['Total Amount (MT)', number_format($stats['total_amount'] ?? 0, 2)],
                ['Processing Time', $duration . ' seconds'],
            ]
        );
        
        if (isset($stats['reconciled']) && $stats['reconciled'] > 0) {
            $this->info('');
            $this->info('✅ ' . $stats['reconciled'] . ' payments successfully reconciled!');
        }
        
        if (isset($stats['not_found']) && $stats['not_found'] > 0) {
            $this->warn('⚠️  ' . $stats['not_found'] . ' references not found - manual review required');
        }
        
        if (isset($stats['failed']) && $stats['failed'] > 0) {
            $this->error('❌ ' . $stats['failed'] . ' transactions failed - check logs for details');
        }
        
        $this->info('');
        $this->info('Completed at: ' . $endTime->format('Y-m-d H:i:s'));
        $this->info('========================================');
    }
}
