<?php

namespace App\Http\Livewire\Admin\Fee;

use App\Models\Admin;
use App\Models\FeeGroup;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;

class AddFeeGroup extends Component
{

    use FindGuard, LivewireAlert;

    // Declare public properties for the form fields
    public $name, $description;
    public $validatedData;
    public $updateMode = false;

    // List of listeners for confirmation dialogs
    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    // Validation rules for the form inputs
    protected array $rules = [
        'name' => ['required', 'string', 'max:255'],
        'description' => ['nullable', 'string'],
    ];

    /**
     * Perform real-time validation on updated properties
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName); // Validate each property as it is updated
    }

    /**
     * Validate and submit the form data
     */
    public function submit()
    {
        // Validate the form data
        $this->validatedData = $this->validate();

        // If the data is validated successfully, confirm before creating
        if ($this->validatedData) {
            $this->confirm('Tem certeza de que deseja prosseguir?', [
                'toast' => false,
                'position' => 'center',
                'showConfirmButton' => true,
                'confirmButtonText' => 'Sim, eu tenho certeza!',
                'cancelButtonText' => 'Não, cancelar!',
                'onConfirmed' => 'confirmed',
                'onDismissed' => 'cancelled'
            ]);
        }
    }

    /**
     * Create a new FeeGroup record in the database after confirmation
     */
    public function confirmed()
    {
        // Create a new FeeGroup entry in the database
        FeeGroup::create([
            'name' => $this->name,
            'description' => $this->description,
        ]);

        // Create a system notification for the Admin model
        Note::createSystemNotification(Admin::class, 'Novo Grupo de Propinas', 'Novo grupo de propinas criado com sucesso: ' . $this->name);

        // Emit event and alert to notify about the successful creation
        $this->emit('noteAdded');
        $this->alert('success', 'Novo grupo de propinas criado com sucesso: ' . $this->name);

        // Reset the form fields after successful submission
        $this->reset();
    }

    /**
     * Handle cancelled confirmation action
     */
    public function cancelled()
    {
        // Alert the user about the cancellation
        $this->alert('error', 'Você cancelou.');
    }

    /**
     * Render the Livewire component's view
     */

    public function render()
    {
        return view('livewire.admin.fee.add-fee-group');
    }
}
