<?php

//namespace App\Http\Livewire\User\Fee;
//
//use App\Models\Fee_assign;
//use App\Models\PaymentReference;
//use App\Models\User;
//use Livewire\Component;
//use App\Models\FeeStructure;
//use Carbon\Carbon;
//use LaravelMultipleGuards\Traits\FindGuard;
//use Barryvdh\DomPDF\Facade\Pdf;
//use Illuminate\Support\Facades\Log;
//use Illuminate\Support\Facades\Storage;
//use Illuminate\Support\Str;
//
//class FeeCollectionUser extends Component
//{
//    use FindGuard;
//    
//    public $studentId;
//    public $feeGroup;
//    public $date;
//    public $discountGroup;
//    public $discount = 0;
//    public $paymentMode = 'Cash';
//    public $note;
//    public $students = [];
//    public $selectedStudent;
//    public $selectedFee;
//    public $selectedFeeAssignId;
//    public $month;
//    public $year;
//    public $amount;
//    public $fine;
//
//    protected $listeners = [
//        'openModal' => 'preparePayment',
//        'collectPayment' => 'processPayment',
//        'generateReference' => 'generateReference'
//    ];
//
//    public function mount()
//    {
//        $user = $this->findGuardType()->user();
//        if ($user) {
//            $this->studentId = $user->id;
//            $this->selectedStudent = User::findOrFail($this->studentId);
//            $this->date = now()->format('d/m/Y');
//        } else {
//            abort(404, 'User not found');
//        }
//    }
//
//    /**
//     * Calcular data de vencimento da referência baseada na fee structure
//     */
//    private function calculateReferenceExpiryDate($month, $year)
//    {
//        $student = $this->selectedStudent;
//        $studentClassroom = $student->class->class ?? null;
//
//        // Buscar fee structure para obter dia de vencimento
//        $feeStructures = FeeStructure::where('active', 1)->get();
//        $filteredFeeStructures = $feeStructures->filter(function ($fee) use ($studentClassroom) {
//            $gradesArray = explode(',', $fee->grades);
//            return in_array($studentClassroom, $gradesArray);
//        });
//
//        $feeStructure = $filteredFeeStructures->first();
//        $dueDay = $feeStructure->payment_due_day ?? 5; // Default: dia 5
//
//        // Converter nome do mês para número
//        $monthNumber = $this->getMonthNumber($month);
//        
//        try {
//            // Criar data de vencimento
//            $expiryDate = Carbon::createFromDate($year, $monthNumber, $dueDay);
//            
//            // Se a data já passou no ano atual, não alterar (manter a data original)
//            if ($expiryDate->isPast() && $year == now()->year) {
//                // Para meses que já passaram, manter a data como estava
//                $expiryDate = $expiryDate; // Manter a data mesmo que seja passada
//            }
//            
//            Log::info('Calculated reference expiry date', [
//                'month' => $month,
//                'month_number' => $monthNumber,
//                'year' => $year,
//                'due_day' => $dueDay,
//                'expiry_date' => $expiryDate->toDateString(),
//                'is_past' => $expiryDate->isPast()
//            ]);
//            
//            return $expiryDate;
//            
//        } catch (\Exception $e) {
//            Log::error('Error calculating expiry date', [
//                'error' => $e->getMessage(),
//                'month' => $month,
//                'year' => $year,
//                'due_day' => $dueDay
//            ]);
//            
//            // Fallback: usar data atual + TTL days
//            $ttlDays = (int) config('payments.reference_ttl_days', 3);
//            return now()->addDays($ttlDays);
//        }
//    }
//    
//    public function generateReference($data)
//    {
//        Log::info('generateReference called', ['data' => $data]);
//        
//        try {
//            $validated = validator($data, [
//                'month' => 'required|string',
//                'year'  => 'required|integer|min:2020|max:' . (date('Y') + 1),
//                'amount'=> 'required|numeric|min:0.01',
//                'fine'  => 'nullable|numeric|min:0',
//                'discount' => 'nullable|numeric|min:0',
//                'paymentMode' => 'required|in:Reference,Entity_Reference',
//                'note' => 'nullable|string|max:500',
//            ])->validate();
//
//            $student = auth()->user();
//            if (!$student) {
//                throw new \RuntimeException('Sessão expirada');
//            }
//
//            // Verificar se já existe referência pendente para este mês/ano
//            $existingReference = PaymentReference::where([
//                'student_id' => $student->id,
//                'fee_month' => $validated['month'],
//                'fee_year' => $validated['year'],
//                'status' => 'pending'
//            ])->first();
//
//            if ($existingReference) {
//                throw new \RuntimeException('Já existe uma referência pendente para ' . $validated['month'] . '/' . $validated['year']);
//            }
//
//            $entity  = config('payments.entity', '11111');
//            $ttlDays = (int) config('payments.reference_ttl_days', 3);
//            $amount  = round(($validated['amount'] ?? 0), 2);
//            
//            // 1) calcula a validade (fim do dia) ANTES de gravar
//            $expiresAt = $this->calculateReferenceExpiryDate($validated['month'], (int)$validated['year']);
//
//
//            $referenceGenerator = app(\App\Services\ReferenceGenerator::class);
//            
//            // NOVO: Usar método que escolhe automaticamente o melhor campo do estudante
//            $reference = $referenceGenerator->makeFromStudent(
//                $validated['month'],
//                (int)$validated['year'],
//                $student,
//                $amount
//            );
//
//            if (!$referenceGenerator->validate($reference)) {
//                throw new \RuntimeException('Referência gerada é inválida');
//            }
//
//            Log::info('Reference generated using BCI algorithm', [
//                'student_id' => $student->id,
//                'month' => $validated['month'],
//                'year' => $validated['year'],
//                'amount' => $amount,
//                'reference' => $reference,
//                'reference_length' => strlen($reference),
//                'expires_at'       => $expiresAt->toDateTimeString()
//            ]);
//
//            // Salvar na tabela payment_references
//            $paymentReference = PaymentReference::create([
//                'student_id' => $student->id,
//                'entity_code' => $entity,
//                'reference_number' => $reference,
//                'amount' => $amount,
//                'fee_month' => $validated['month'],
//                'fee_year' => $validated['year'],
//                'expires_at' => $expiresAt,     // ✅ gravado
//                'status' => 'pending'
//            ]);
//
//            Log::info('Payment reference record created', ['payment_reference_id' => $paymentReference->id]);
//
//            
//            
//            Log::info('Reference expiry calculated', [
//                'month' => $validated['month'],
//                'year' => $validated['year'],
//                'expires_at' => $expiresAt->toDateString()
//            ]);
//            
//            try {
//                // Verificar se a view existe
//                if (!view()->exists('pdf.reference')) {
//                    Log::warning('PDF template not found, creating simple response');
//                    throw new \Exception('PDF template not available');
//                }
//
//                $pdf = Pdf::loadView('pdf.reference', [
//                    'student'    => $student,
//                    'payment'    => $paymentReference,
//                    'entity'     => $entity,
//                    'reference'  => $reference,
//                    'reference_formatted' => $referenceGenerator->format($reference),
//                    'amount'     => $amount,
//                    'expires_at' => $expiresAt,
//                ])
//                ->setOptions([
//                    'isRemoteEnabled' => false,
//                    'isHtml5ParserEnabled' => true,
//                    'isPhpEnabled' => false
//                ])
//                ->output();
//
//                // Nome do arquivo no formato SLUG_ID + REFERENCE + TIMESTAMP
//                $timestamp = now()->format('d-m-Y_H-i'); // Formato: dd-mm-yyyy_HH-mm
//                $studentSlug = Str::slug($student->name ?? $student->id);
//                $filename = "{$studentSlug}_{$reference}_{$timestamp}.pdf";
//                $path = "references/{$filename}";
//                
//                // Criar diretório se não existir
//                Storage::disk('public')->makeDirectory('references');
//                
//                // Salvar o arquivo
//                $saved = Storage::disk('public')->put($path, $pdf);
//                
//                if (!$saved) {
//                    throw new \Exception('Falhou ao salvar o arquivo PDF');
//                }
//                
//                // Verificar se o arquivo foi criado
//                if (!Storage::disk('public')->exists($path)) {
//                    throw new \Exception('Arquivo PDF não foi encontrado após criação');
//                }
//                
//                // Gerar URLs
//                $url = asset('storage/' . $path);
//                $urlAlt = url('storage/' . $path);
//                $urlManual = request()->getSchemeAndHttpHost() . '/storage/' . $path;
//                
//                $fileSize = Storage::disk('public')->size($path);
//                $fullPath = storage_path('app/public/' . $path);
//                
//                Log::info('PDF generated successfully', [
//                    'student_id' => $student->id,
//                    'student_slug' => $studentSlug,
//                    'reference' => $reference,
//                    'timestamp' => $timestamp,
//                    'filename' => $filename,
//                    'path' => $path,
//                    'url_asset' => $url,
//                    'url_alt' => $urlAlt,
//                    'url_manual' => $urlManual,
//                    'size' => $fileSize,
//                    'full_path' => $fullPath,
//                    'file_exists' => file_exists($fullPath),
//                    'public_path_exists' => file_exists(public_path('storage/' . $path))
//                ]);
//
//                $this->emit('referenceReady', [
//                    'pdf_url'   => $url,
//                    'pdf_url_alt' => $urlAlt,
//                    'pdf_url_manual' => $urlManual,
//                    'reference' => $reference,
//                    'reference_formatted' => $referenceGenerator->format($reference),
//                    'entity'    => $entity,
//                    'amount'    => $amount,
//                    'expires_at'=> $expiresAt->toDateString(),
//                    'expires_at_formatted' => $expiresAt->format('d/m/Y'),
//                    'payment_reference_id' => $paymentReference->id,
//                    'student_name' => $student->name,
//                    'file_info' => [
//                        'filename' => $filename,
//                        'path' => $path,
//                        'size' => $fileSize
//                    ]
//                ]);
//
//            } catch (\Exception $pdfError) {
//                Log::error('PDF generation failed', ['error' => $pdfError->getMessage()]);
//                
//                // Mesmo se o PDF falhar, emitir sucesso com dados da referência
//                $this->emit('referenceReady', [
//                    'pdf_url'   => null,
//                    'reference' => $reference,
//                    'reference_formatted' => $referenceGenerator->format($reference),
//                    'entity'    => $entity,
//                    'amount'    => $amount,
//                    'expires_at'=> $expiresAt->toDateString(),
//                    'expires_at_formatted' => $expiresAt->format('d/m/Y'),
//                    'payment_reference_id' => $paymentReference->id,
//                    'student_name' => $student->name,
//                    'manual_pdf' => true
//                ]);
//            }
//
//        } catch (\Illuminate\Validation\ValidationException $e) {
//            Log::error('Validation failed', ['errors' => $e->errors()]);
//            $this->emit('referenceError', 'Dados inválidos: ' . implode(', ', array_flatten($e->errors())));
//        } catch (\Throwable $e) {
//            Log::error('generateReference failed', [
//                'message' => $e->getMessage(),
//                'file' => $e->getFile(),
//                'line' => $e->getLine(),
//                'trace' => $e->getTraceAsString()
//            ]);
//            $this->emit('referenceError', 'Erro ao gerar a referência: ' . $e->getMessage());
//        }
//    }
//
//    private function calculateBaseFee()
//    {
//        $student = $this->selectedStudent;
//        $studentClassroom = $student->class->class ?? null;
//
//        $feeStructures = FeeStructure::where('active', 1)->get();
//        
//        $filteredFeeStructures = $feeStructures->filter(function ($fee) use ($studentClassroom) {
//            $gradesArray = explode(',', $fee->grades);
//            return in_array($studentClassroom, $gradesArray);
//        });
//
//        return $filteredFeeStructures->sum('monthly_fee');
//    }
//
//    private function calculateFine($month, $year)
//    {
//        $student = $this->selectedStudent;
//        $studentClassroom = $student->class->class ?? null;
//
//        $feeStructures = FeeStructure::where('active', 1)->get();
//        $filteredFeeStructures = $feeStructures->filter(function ($fee) use ($studentClassroom) {
//            $gradesArray = explode(',', $fee->grades);
//            return in_array($studentClassroom, $gradesArray);
//        });
//
//        $feeStructure = $filteredFeeStructures->first();
//        if (!$feeStructure) {
//            return 0;
//        }
//
//        $dueDay = $feeStructure->payment_due_day ?? 5;
//        $dueDate = Carbon::createFromDate($year, $month, $dueDay);
//        $currentDate = Carbon::now();
//
//        if ($currentDate->gt($dueDate)) {
//            $baseFee = $this->calculateBaseFee();
//            $latePenalty = $feeStructure->late_penalty_percentage ?? 0;
//            return ($baseFee * $latePenalty) / 100;
//        }
//
//        return 0;
//    }
//
//    private function validatePayment($month, $year)
//    {
//        $monthNumber = $this->getMonthNumber($month);
//        
//        $existingPayment = Fee_assign::where([
//            'student_id' => $this->studentId,
//            'month' => $month,
//            'year' => $year
//        ])->first();
//
//        if ($existingPayment) {
//            throw new \Exception("Já existe um pagamento registrado para {$month}/{$year}");
//        }
//
//        if ($monthNumber < 1 || $monthNumber > 12) {
//            throw new \Exception("Mês inválido");
//        }
//
//        if ($year < 2020 || $year > (date('Y') + 1)) {
//            throw new \Exception("Ano inválido");
//        }
//
//        return true;
//    }
//
//    private function getMonthNumber($monthName)
//    {
//        $months = [
//            'january' => 1, 'february' => 2, 'march' => 3, 'april' => 4,
//            'may' => 5, 'june' => 6, 'july' => 7, 'august' => 8,
//            'september' => 9, 'october' => 10, 'november' => 11, 'december' => 12
//        ];
//
//        return $months[strtolower($monthName)] ?? 0;
//    }
//
//    public function processPayment($paymentData)
//    {
//        try {
//            $month = $paymentData['month'];
//            $year = $paymentData['year'];
//            
//            $this->validatePayment($month, $year);
//            
//            $baseFee = $this->calculateBaseFee();
//            $fine = $this->calculateFine($this->getMonthNumber($month), $year);
//            
//            $discount = $this->validateDiscount($paymentData['discount'] ?? 0);
//            
//            $finalAmount = $baseFee + $fine - $discount;
//            
//            $finalAmount = max(0, $finalAmount);
//
//            $this->fill([
//                'month' => $month,
//                'year' => $year,
//                'discount' => $discount,
//                'paymentMode' => $paymentData['paymentMode'] ?? 'Cash',
//                'note' => $paymentData['note'] ?? null
//            ]);
//
//            $this->collectFee($finalAmount, $fine);
//
//        } catch (\Exception $e) {
//            $this->emit('paymentError', $e->getMessage());
//            session()->flash('error', $e->getMessage());
//        }
//    }
//
//    private function validateDiscount($requestedDiscount)
//    {
//        if ($requestedDiscount <= 0) {
//            return 0;
//        }
//
//        $student = $this->selectedStudent;
//        $maxDiscountAllowed = 0;
//        
//        if ($student->discount_group) {
//            $maxDiscountAllowed = $student->discount_group->max_discount ?? 0;
//        }
//        
//        return min($requestedDiscount, $maxDiscountAllowed);
//    }
//
//    public function preparePayment($feeId, $amount, $month, $year)
//    {
//        $this->selectedFeeAssignId = $feeId;
//        $this->month = $month;
//        $this->year = $year;
//        
//        $this->dispatchBrowserEvent('openModal');
//    }
//
//    public function collectFee($calculatedAmount = null, $calculatedFine = null)
//    {
//        if ($calculatedAmount === null) {
//            $calculatedAmount = $this->amount;
//        }
//        if ($calculatedFine === null) {
//            $calculatedFine = $this->fine;
//        }
//
//        $this->validate([
//            'month' => 'required|string',
//            'year' => 'required|numeric',
//            'discount' => 'numeric|min:0',
//            'paymentMode' => 'required',
//        ]);
//
//        try {
//            Fee_assign::create([
//                'student_id' => $this->studentId,
//                'amount' => $calculatedAmount,
//                'discount' => $this->discount,
//                'fine' => $calculatedFine,
//                'payment_mode' => $this->paymentMode,
//                'note' => $this->note,
//                'pay_day' => date("d"),
//                'pay_month' => date("m"),
//                'pay_year' => date("Y"),
//                'pay_type' => strtolower($this->paymentMode),
//                'status' => 'Paid',
//                'month' => $this->month,
//                'year' => $this->year,
//            ]);
//
//            $this->dispatchBrowserEvent('closeModal');
//            $this->emit('paymentSuccess');
//            
//            session()->flash('message', 'Pagamento registrado com sucesso!');
//            $this->reset(['discount', 'paymentMode', 'note', 'month', 'year']);
//
//        } catch (\Exception $e) {
//            $this->emit('paymentError', $e->getMessage());
//            session()->flash('error', $e->getMessage());
//        }
//    }
//
//    public function getPaymentDetails($month, $year)
//    {
//        $monthNumber = $this->getMonthNumber($month);
//        $baseFee = $this->calculateBaseFee();
//        $fine = $this->calculateFine($monthNumber, $year);
//        
//        return [
//            'baseFee' => $baseFee,
//            'fine' => $fine,
//            'total' => $baseFee + $fine,
//            'month' => $month,
//            'year' => $year
//        ];
//    }
//
//    public function render()
//    {
//        $currentYear = date('Y');
//        $student = $this->selectedStudent;
//        $studentClassroom = $student->class->class ?? null;
//
//        $feeStructures = FeeStructure::where('active', 1)->get();
//        $filteredFeeStructures = $feeStructures->filter(function ($fee) use ($studentClassroom) {
//            $gradesArray = explode(',', $fee->grades);
//            return in_array($studentClassroom, $gradesArray);
//        });
//
//        $fees = Fee_assign::where('student_id', $this->studentId)->get();
//        
//        $payment_references = PaymentReference::where('student_id', $this->studentId)
//            ->where('fee_year', $currentYear)
//            ->get();
//
//        $totalExpected = 0;
//        $totalPaid = 0;
//        $totalPending = 0;
//        $totalOverdue = 0;
//
//        $months = [
//            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
//            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
//            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
//        ];
//
//        $monthlyData = [];
//
//        foreach ($months as $num => $name) {
//            $baseFee = $this->calculateBaseFee();
//            $fine = $this->calculateFine($num, $currentYear);
//            
//            $feeStructure = $filteredFeeStructures->first();
//            $dueDay = $feeStructure->payment_due_day ?? 5;
//            $dueDate = Carbon::createFromDate($currentYear, $num, $dueDay);
//            $currentDate = Carbon::now();
//
//            $feeRecord = $fees->first(function ($f) use ($name, $currentYear) {
//                return strtolower($f->month) == strtolower($name) && $f->year == $currentYear;
//            });
//
//            $paid = $feeRecord && strtolower($feeRecord->status) == 'paid';
//            $amountPaid = $feeRecord->amount ?? 0;
//
//            $expected = $baseFee;
//            $totalExpected += $expected;
//
//            if ($paid) {
//                $totalPaid += $amountPaid;
//            } else {
//                if ($currentDate->gt($dueDate)) {
//                    $totalOverdue += $expected + $fine;
//                } else {
//                    $totalPending += $expected;
//                }
//            }
//
//            $monthlyData[] = [
//                'month' => "$name $currentYear",
//                'month_name' => $name,
//                'month_number' => $num,
//                'year' => $currentYear,
//                'due_date' => $dueDate->format('d/m/Y'),
//                'status' => $paid ? 'Pago' : ($currentDate->gt($dueDate) ? 'Em Atraso' : 'Não Pago'),
//                'expected' => $expected,
//                'fine' => $fine,
//                'paid' => $amountPaid,
//                'remaining' => max(0, $expected - $amountPaid),
//            ];
//        }
//
//        return view('livewire.user.fee.fee-collection', [
//            'student' => $student,
//            'fees' => $fees,
//            'feestructures' => $filteredFeeStructures,
//            'monthlyData' => $monthlyData,
//            'totalExpected' => $totalExpected,
//            'totalPaid' => $totalPaid,
//            'totalPending' => $totalPending,
//            'totalOverdue' => $totalOverdue,
//            'currentYear' => $currentYear,
//            'payment_references' => $payment_references,
//        ]);
//    }
//
//    public function openModal($feeAssignId, $amount, $month, $year)
//    {
//        $this->selectedFeeAssignId = $feeAssignId;
//        $this->month = (string) $month;
//        $this->year = (int) $year;
//        
//        $this->dispatchBrowserEvent('openModal');
//        $this->emit('openModal', $feeAssignId, $amount, $month, $year);
//    }
//}


namespace App\Http\Livewire\User\Fee;

use App\Models\Fee_assign;
use App\Models\PaymentReference;
use App\Models\User;
use App\Services\FeeCalculationService;
use Livewire\Component;
use App\Models\FeeStructure;
use Carbon\Carbon;
use LaravelMultipleGuards\Traits\FindGuard;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class FeeCollectionUser extends Component
{
    use FindGuard;
    
    public $studentId;
    public $feeGroup;
    public $date;
    public $discountGroup;
    public $discount = 0;
    public $paymentMode = 'Cash';
    public $note;
    public $students = [];
    public $selectedStudent;
    public $selectedFee;
    public $selectedFeeAssignId;
    public $month;
    public $year;
    public $amount;
    public $fine;

    protected $feeCalculationService;

    protected $listeners = [
        'openModal' => 'preparePayment',
        'collectPayment' => 'processPayment',
        'generateReference' => 'generateReference'
    ];

    public function boot(FeeCalculationService $feeCalculationService)
    {
        $this->feeCalculationService = $feeCalculationService;
    }

    public function mount()
    {
        $user = $this->findGuardType()->user();
        if ($user) {
            $this->studentId = $user->id;
            $this->selectedStudent = User::findOrFail($this->studentId);
            $this->date = now()->format('d/m/Y');
        } else {
            abort(404, 'User not found');
        }
    }

    /**
     * Calcular data de vencimento da referência baseada na fee structure
     */
    private function calculateReferenceExpiryDate($month, $year)
    {
        try {
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $this->selectedStudent,
                $month,
                (int)$year
            );

            // Se tem due_date calculada, usa ela
            if ($calculation['due_date']) {
                return $calculation['due_date'];
            }

            // Fallback para lógica original
            $student = $this->selectedStudent;
            $studentClassroom = $student->class->class ?? $student->classroom->class ?? null;

            $feeStructures = FeeStructure::where('active', 1)->get();
            $filteredFeeStructures = $feeStructures->filter(function ($fee) use ($studentClassroom) {
                $gradesArray = explode(',', $fee->grades);
                return in_array($studentClassroom, $gradesArray);
            });

            $feeStructure = $filteredFeeStructures->first();
            $dueDay = $feeStructure->payment_due_day ?? 5;

            $monthNumber = $this->getMonthNumber($month);
            $expiryDate = Carbon::createFromDate($year, $monthNumber, $dueDay);
            
            Log::info('Calculated reference expiry date', [
                'month' => $month,
                'month_number' => $monthNumber,
                'year' => $year,
                'due_day' => $dueDay,
                'expiry_date' => $expiryDate->toDateString(),
                'is_past' => $expiryDate->isPast()
            ]);
            
            return $expiryDate;
            
        } catch (\Exception $e) {
            Log::error('Error calculating expiry date', [
                'error' => $e->getMessage(),
                'month' => $month,
                'year' => $year
            ]);
            
            $ttlDays = (int) config('payments.reference_ttl_days', 3);
            return now()->addDays($ttlDays);
        }
    }
    
    public function generateReference($data)
    {
        Log::info('generateReference called', ['data' => $data]);
        
        try {
            $validated = validator($data, [
                'month' => 'required|string',
                'year'  => 'required|integer|min:2020|max:' . (date('Y') + 1),
                'amount'=> 'required|numeric|min:0.01',
                'fine'  => 'nullable|numeric|min:0',
                'discount' => 'nullable|numeric|min:0',
                'paymentMode' => 'required|in:Reference,Entity_Reference',
                'note' => 'nullable|string|max:500',
            ])->validate();

            $student = auth()->user();
            if (!$student) {
                throw new \RuntimeException('Sessão expirada');
            }

            // Verificar se já existe referência pendente para este mês/ano
            $existingReference = PaymentReference::where([
                'student_id' => $student->id,
                'fee_month' => $validated['month'],
                'fee_year' => $validated['year'],
                'status' => 'pending'
            ])->first();

            if ($existingReference) {
                throw new \RuntimeException('Já existe uma referência pendente para ' . $validated['month'] . '/' . $validated['year']);
            }

            // USAR O SERVICE UNIFICADO para calcular valores corretos
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $student,
                $validated['month'],
                (int)$validated['year']
            );

            $entity = config('payments.entity', '11111');
            
            // Valor total = base + multa calculada + multa manual - desconto
            $manualFine = (float)($validated['fine'] ?? 0);
            $discount = (float)($validated['discount'] ?? 0);
            $totalFine = max($calculation['fine_amount'], $manualFine);
            $totalAmount = round($calculation['base_amount'] + $totalFine - $discount, 2);

            // Calcula validade baseada no service
            $expiresAt = $this->calculateReferenceExpiryDate($validated['month'], (int)$validated['year']);

            $referenceGenerator = app(\App\Services\ReferenceGenerator::class);
            
            $reference = $referenceGenerator->makeFromStudent(
                $validated['month'],
                (int)$validated['year'],
                $student,
                $totalAmount
            );

            if (!$referenceGenerator->validate($reference)) {
                throw new \RuntimeException('Referência gerada é inválida');
            }

            Log::info('Reference generated with unified service', [
                'student_id' => $student->id,
                'month' => $validated['month'],
                'year' => $validated['year'],
                'base_amount' => $calculation['base_amount'],
                'calculated_fine' => $calculation['fine_amount'],
                'manual_fine' => $manualFine,
                'total_fine' => $totalFine,
                'discount' => $discount,
                'total_amount' => $totalAmount,
                'reference' => $reference,
                'expires_at' => $expiresAt->toDateTimeString(),
                'is_late_payment' => $calculation['is_late_payment']
            ]);

            // Salvar na tabela payment_references com valores corretos
            $paymentReference = PaymentReference::create([
                'student_id' => $student->id,
                'entity_code' => $entity,
                'reference_number' => $reference,
                'amount' => $totalAmount,
                'fine_amount' => $totalFine, // PRESERVA A MULTA
                'fee_month' => $validated['month'],
                'fee_year' => $validated['year'],
                'expires_at' => $expiresAt,
                'status' => 'pending'
            ]);

            try {
                if (!view()->exists('pdf.reference')) {
                    Log::warning('PDF template not found, creating simple response');
                    throw new \Exception('PDF template not available');
                }

                $pdf = Pdf::loadView('pdf.reference', [
                    'student'    => $student,
                    'payment'    => $paymentReference,
                    'entity'     => $entity,
                    'reference'  => $reference,
                    'reference_formatted' => $referenceGenerator->format($reference),
                    'amount'     => $totalAmount,
                    'base_amount' => $calculation['base_amount'],
                    'fine_amount' => $totalFine,
                    'discount_amount' => $discount,
                    'expires_at' => $expiresAt,
                    'calculation_details' => $calculation
                ])
                ->setOptions([
                    'isRemoteEnabled' => false,
                    'isHtml5ParserEnabled' => true,
                    'isPhpEnabled' => false
                ])
                ->output();

                $timestamp = now()->format('d-m-Y_H-i');
                $studentSlug = Str::slug($student->name ?? $student->id);
                $filename = "{$studentSlug}_{$reference}_{$timestamp}.pdf";
                $path = "references/{$filename}";
                
                Storage::disk('public')->makeDirectory('references');
                $saved = Storage::disk('public')->put($path, $pdf);
                
                if (!$saved || !Storage::disk('public')->exists($path)) {
                    throw new \Exception('Falhou ao salvar o arquivo PDF');
                }
                
                $url = asset('storage/' . $path);
                $fileSize = Storage::disk('public')->size($path);
                
                Log::info('PDF generated successfully with preserved fines', [
                    'student_id' => $student->id,
                    'reference' => $reference,
                    'filename' => $filename,
                    'base_amount' => $calculation['base_amount'],
                    'fine_amount' => $totalFine,
                    'total_amount' => $totalAmount,
                    'size' => $fileSize
                ]);

                $this->emit('referenceReady', [
                    'pdf_url' => $url,
                    'reference' => $reference,
                    'reference_formatted' => $referenceGenerator->format($reference),
                    'entity' => $entity,
                    'amount' => $totalAmount,
                    'base_amount' => $calculation['base_amount'],
                    'fine_amount' => $totalFine,
                    'discount_amount' => $discount,
                    'expires_at' => $expiresAt->toDateString(),
                    'expires_at_formatted' => $expiresAt->format('d/m/Y'),
                    'payment_reference_id' => $paymentReference->id,
                    'student_name' => $student->name,
                    'is_late_payment' => $calculation['is_late_payment'],
                    'file_info' => [
                        'filename' => $filename,
                        'path' => $path,
                        'size' => $fileSize
                    ]
                ]);

            } catch (\Exception $pdfError) {
                Log::error('PDF generation failed', ['error' => $pdfError->getMessage()]);
                
                $this->emit('referenceReady', [
                    'pdf_url' => null,
                    'reference' => $reference,
                    'reference_formatted' => $referenceGenerator->format($reference),
                    'entity' => $entity,
                    'amount' => $totalAmount,
                    'base_amount' => $calculation['base_amount'],
                    'fine_amount' => $totalFine,
                    'expires_at' => $expiresAt->toDateString(),
                    'expires_at_formatted' => $expiresAt->format('d/m/Y'),
                    'payment_reference_id' => $paymentReference->id,
                    'student_name' => $student->name,
                    'manual_pdf' => true
                ]);
            }

        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation failed', ['errors' => $e->errors()]);
            $this->emit('referenceError', 'Dados inválidos: ' . implode(', ', array_flatten($e->errors())));
        } catch (\Throwable $e) {
            Log::error('generateReference failed', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            $this->emit('referenceError', 'Erro ao gerar a referência: ' . $e->getMessage());
        }
    }

    public function processPayment($paymentData)
    {
        try {
            $month = $paymentData['month'];
            $year = $paymentData['year'];
            
            $this->validatePayment($month, $year);
            
            // USA O SERVICE UNIFICADO
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $this->selectedStudent,
                $month,
                (int)$year
            );
            
            $discount = $this->validateDiscount($paymentData['discount'] ?? 0);
            $manualFine = (float)($paymentData['fine'] ?? 0);
            
            // Valor final com multas preservadas
            $totalFine = max($calculation['fine_amount'], $manualFine);
            $finalAmount = max(0, $calculation['base_amount'] + $totalFine - $discount);

            $this->fill([
                'month' => $month,
                'year' => $year,
                'discount' => $discount,
                'paymentMode' => $paymentData['paymentMode'] ?? 'Cash',
                'note' => $paymentData['note'] ?? null
            ]);

            // Usa service unificado para criar pagamento
            $paymentInfo = [
                'month' => $month,
                'year' => $year,
                'amount' => $finalAmount,
                'fine' => $manualFine, // Multa manual adicional
                'discount' => $discount,
                'payment_mode' => $paymentData['paymentMode'] ?? 'Cash',
                'paymentMode' => $paymentData['paymentMode'] ?? 'Cash',
                'note' => $paymentData['note'] ?? null,
                'pay_type' => strtolower($paymentData['paymentMode'] ?? 'cash')
            ];

            $feeAssign = $this->feeCalculationService->createFeePayment(
                $this->selectedStudent,
                $paymentInfo,
                $calculation
            );

            Log::info('User fee payment processed with preserved fines', [
                'student_id' => $this->studentId,
                'fee_assign_id' => $feeAssign->id,
                'base_calculated' => $calculation['base_amount'],
                'fine_calculated' => $calculation['fine_amount'],
                'fine_manual' => $manualFine,
                'fine_preserved' => $feeAssign->fine,
                'total_amount' => $finalAmount,
                'is_late_payment' => $calculation['is_late_payment']
            ]);

            $this->emit('paymentSuccess');

        } catch (\Exception $e) {
            Log::error('User payment processing failed', [
                'student_id' => $this->studentId,
                'error' => $e->getMessage()
            ]);
            $this->emit('paymentError', $e->getMessage());
            session()->flash('error', $e->getMessage());
        }
    }

    private function validatePayment($month, $year)
    {
        $monthNumber = $this->getMonthNumber($month);
        
        // Usa service para verificar pagamento existente
        if ($this->feeCalculationService->hasPaymentForPeriod($this->selectedStudent, $month, $year)) {
            throw new \Exception("Já existe um pagamento registrado para {$month}/{$year}");
        }

        if ($monthNumber < 1 || $monthNumber > 12) {
            throw new \Exception("Mês inválido");
        }

        if ($year < 2020 || $year > (date('Y') + 1)) {
            throw new \Exception("Ano inválido");
        }

        return true;
    }

    private function getMonthNumber($monthName)
    {
        $months = [
            'january' => 1, 'february' => 2, 'march' => 3, 'april' => 4,
            'may' => 5, 'june' => 6, 'july' => 7, 'august' => 8,
            'september' => 9, 'october' => 10, 'november' => 11, 'december' => 12
        ];

        return $months[strtolower($monthName)] ?? 0;
    }

    private function validateDiscount($requestedDiscount)
    {
        if ($requestedDiscount <= 0) {
            return 0;
        }

        $student = $this->selectedStudent;
        $maxDiscountAllowed = 0;
        
        if ($student->discount_group) {
            $maxDiscountAllowed = $student->discount_group->max_discount ?? 0;
        }
        
        return min($requestedDiscount, $maxDiscountAllowed);
    }

    public function preparePayment($feeId, $amount, $month, $year)
    {
        $this->selectedFeeAssignId = $feeId;
        $this->month = $month;
        $this->year = $year;
        
        $this->dispatchBrowserEvent('openModal');
    }

    public function collectFee($calculatedAmount = null, $calculatedFine = null)
    {
        if ($calculatedAmount === null) {
            $calculatedAmount = $this->amount;
        }
        if ($calculatedFine === null) {
            $calculatedFine = $this->fine;
        }

        $this->validate([
            'month' => 'required|string',
            'year' => 'required|numeric',
            'discount' => 'numeric|min:0',
            'paymentMode' => 'required',
        ]);

        try {
            // Usa service unificado
            $paymentData = [
                'month' => $this->month,
                'year' => $this->year,
                'amount' => $calculatedAmount,
                'fine' => $calculatedFine,
                'discount' => $this->discount,
                'payment_mode' => $this->paymentMode,
                'paymentMode' => $this->paymentMode,
                'note' => $this->note,
                'pay_type' => strtolower($this->paymentMode)
            ];

            $feeAssign = $this->feeCalculationService->createFeePayment(
                $this->selectedStudent,
                $paymentData
            );

            $this->dispatchBrowserEvent('closeModal');
            $this->emit('paymentSuccess');
            
            session()->flash('message', 'Pagamento registrado com sucesso! Multa preservada no histórico.');
            $this->reset(['discount', 'paymentMode', 'note', 'month', 'year']);

        } catch (\Exception $e) {
            Log::error('User manual payment failed', [
                'student_id' => $this->studentId,
                'error' => $e->getMessage()
            ]);
            $this->emit('paymentError', $e->getMessage());
            session()->flash('error', $e->getMessage());
        }
    }

    public function getPaymentDetails($month, $year)
    {
        $calculation = $this->feeCalculationService->calculateFeeForStudent(
            $this->selectedStudent,
            $month,
            (int)$year
        );
        
        return [
            'baseFee' => $calculation['base_amount'],
            'fine' => $calculation['fine_amount'],
            'total' => $calculation['total_amount'],
            'month' => $month,
            'year' => $year,
            'isLate' => $calculation['is_late_payment'],
            'dueDate' => $calculation['due_date']?->format('d/m/Y')
        ];
    }

    public function render()
    {
        $currentYear = date('Y');
        $student = $this->selectedStudent;

        // USA SERVICE UNIFICADO para calcular resumo do ano
        $yearSummary = $this->feeCalculationService->calculateYearSummary($student, $currentYear);
        
        $fees = Fee_assign::where('student_id', $this->studentId)->get();
        
        $payment_references = PaymentReference::where('student_id', $this->studentId)
            ->where('fee_year', $currentYear)
            ->get();

        return view('livewire.user.fee.fee-collection', [
            'student' => $student,
            'fees' => $fees,
            'feestructures' => collect(), // Não usado mais diretamente
            'monthlyData' => $yearSummary['months_detail'],
            'totalExpected' => $yearSummary['total_expected'],
            'totalPaid' => $yearSummary['total_paid'],
            'totalPending' => $yearSummary['total_pending'],
            'totalOverdue' => $yearSummary['total_overdue'],
            'totalFines' => $yearSummary['total_fines'],
            'currentYear' => $currentYear,
            'payment_references' => $payment_references,
        ]);
    }

    public function openModal($feeAssignId, $amount, $month, $year)
    {
        $this->selectedFeeAssignId = $feeAssignId;
        $this->month = (string) $month;
        $this->year = (int) $year;
        
        $this->dispatchBrowserEvent('openModal');
        $this->emit('openModal', $feeAssignId, $amount, $month, $year);
    }
}
