<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Worksheet\PageSetup;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

class PaymentReportsExport implements FromCollection, WithHeadings, WithStyles, WithColumnWidths, WithTitle, WithEvents
{
    protected $data;
    protected $filters;
    protected $totalAmount;
    protected $totalCount;
    protected $totalFine;
    protected $subtotal;

    public function __construct($data, $filters, $totalAmount, $totalCount, $totalFine = 0, $subtotal = 0)
    {
        $this->data = $data;
        $this->filters = $filters;
        $this->totalAmount = $totalAmount;
        $this->totalCount = $totalCount;
        $this->totalFine = $totalFine;
        $this->subtotal = $subtotal;
    }

    public function collection()
    {
        $rows = collect([]);

        // Linhas do cabeçalho (título + data + espaço)
        $rows->push(['RELATÓRIO DE PAGAMENTOS']);
        $rows->push(['Gerado em: ' . now()->format('d/m/Y H:i:s')]);
        $rows->push(['']);

        // Filtros aplicados
        $rows->push(['FILTROS APLICADOS']);
        foreach ($this->filters as $filter) {
            $rows->push(['', $filter]);
        }
        $rows->push(['']);

        // Resumo financeiro
        $rows->push(['RESUMO FINANCEIRO']);
        $rows->push(['Total de Registros:', number_format($this->totalCount, 0, ',', '.')]);
        $rows->push(['Subtotal (sem multa):', number_format($this->subtotal, 2, ',', '.'), 'MT']);
        $rows->push(['Multas:', number_format($this->totalFine, 2, ',', '.'), 'MT']);
        $rows->push(['Total Geral:', number_format($this->totalAmount, 2, ',', '.'), 'MT']);
        $rows->push(['']);
        $rows->push(['']);

        // Cabeçalhos da tabela (serão estilizados separadamente)
        $rows->push([
            '#',
            'Estudante',
            'Código',
            'Turma',
            'Referência',
            'Mês/Ano',
            'Valor (MT)',
            'Multa (MT)',
            'Total (MT)',
            'Data/Hora'
        ]);

        // Dados dos pagamentos
        // Fórmula: VALOR BASE + MULTA - DESCONTO = TOTAL
        foreach ($this->data as $index => $payment) {
            $baseAmount = $payment->amount ?? 0;
            $fineAmount = $payment->fine ?? 0;
            $discountAmount = $payment->discount ?? 0;
            $totalAmount = $baseAmount + $fineAmount - $discountAmount;

            $rows->push([
                $index + 1,
                $payment->student->name ?? 'N/A',
                $payment->student->student_id ?? 'N/A',
                ($payment->student->classroom->class ?? 'N/A') . ' - ' . ($payment->student->classroom->name ?? ''),
                $payment->paymentReference->reference_number ?? ($payment->reference_number ?? 'N/A'),
                $payment->month . '/' . $payment->year,
                number_format($baseAmount, 2, ',', '.'),
                number_format($fineAmount, 2, ',', '.'),
                number_format($totalAmount, 2, ',', '.'),
                \Carbon\Carbon::parse($payment->created_at)->format('d/m/Y H:i')
            ]);
        }

        // Linha de totais
        $rows->push(['']);
        $rows->push([
            '',
            '',
            '',
            '',
            '',
            'SUBTOTAL:',
            number_format($this->subtotal, 2, ',', '.'),
            '',
            '',
            ''
        ]);
        $rows->push([
            '',
            '',
            '',
            '',
            '',
            'MULTAS:',
            number_format($this->totalFine, 2, ',', '.'),
            '',
            '',
            ''
        ]);
        $rows->push([
            '',
            '',
            '',
            '',
            '',
            'TOTAL GERAL:',
            number_format($this->totalAmount, 2, ',', '.'),
            '',
            '',
            ''
        ]);

        return $rows;
    }

    public function headings(): array
    {
        return [];
    }

    public function styles(Worksheet $sheet)
    {
        // Calcular linha onde começa a tabela de dados
        $headerRow = 4 + count($this->filters) + 1 + 6 + 2; // Título(3) + Filtros + Resumo(6) + Espaços(2)
        $dataStartRow = $headerRow + 1;
        $dataEndRow = $dataStartRow + $this->data->count() - 1;
        $totalsStartRow = $dataEndRow + 2;

        // ====================
        // TÍTULO PRINCIPAL
        // ====================
        $sheet->mergeCells('A1:J1');
        $sheet->getStyle('A1')->applyFromArray([
            'font' => [
                'bold' => true,
                'size' => 18,
                'color' => ['rgb' => 'FFFFFF']
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '1E40AF'] // Azul escuro profissional
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER
            ]
        ]);
        $sheet->getRowDimension(1)->setRowHeight(35);

        // ====================
        // DATA DE GERAÇÃO
        // ====================
        $sheet->mergeCells('A2:J2');
        $sheet->getStyle('A2')->applyFromArray([
            'font' => [
                'italic' => true,
                'size' => 9,
                'color' => ['rgb' => '6B7280']
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER
            ]
        ]);

        // ====================
        // SEÇÃO FILTROS
        // ====================
        $sheet->mergeCells('A4:J4');
        $sheet->getStyle('A4')->applyFromArray([
            'font' => [
                'bold' => true,
                'size' => 11,
                'color' => ['rgb' => 'FFFFFF']
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '6B7280']
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_LEFT,
                'vertical' => Alignment::VERTICAL_CENTER
            ]
        ]);
        $sheet->getRowDimension(4)->setRowHeight(25);

        // Estilos dos filtros
        $filtersEndRow = 4 + count($this->filters);
        for ($i = 5; $i <= $filtersEndRow; $i++) {
            $sheet->getStyle('B' . $i)->applyFromArray([
                'font' => ['size' => 9],
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_LEFT]
            ]);
        }

        // ====================
        // SEÇÃO RESUMO FINANCEIRO
        // ====================
        $resumoRow = $filtersEndRow + 2;
        $sheet->mergeCells('A' . $resumoRow . ':J' . $resumoRow);
        $sheet->getStyle('A' . $resumoRow)->applyFromArray([
            'font' => [
                'bold' => true,
                'size' => 11,
                'color' => ['rgb' => 'FFFFFF']
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '059669'] // Verde
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_LEFT,
                'vertical' => Alignment::VERTICAL_CENTER
            ]
        ]);
        $sheet->getRowDimension($resumoRow)->setRowHeight(25);

        // Estilizar linhas do resumo
        for ($i = $resumoRow + 1; $i <= $resumoRow + 5; $i++) {
            $sheet->getStyle('A' . $i)->applyFromArray([
                'font' => ['bold' => true, 'size' => 9],
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_RIGHT]
            ]);
            $sheet->getStyle('B' . $i)->applyFromArray([
                'font' => ['size' => 9],
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_RIGHT]
            ]);
        }

        // ====================
        // CABEÇALHO DA TABELA
        // ====================
        $sheet->getStyle('A' . $headerRow . ':J' . $headerRow)->applyFromArray([
            'font' => [
                'bold' => true,
                'size' => 10,
                'color' => ['rgb' => 'FFFFFF']
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '3B82F6'] // Azul
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_MEDIUM,
                    'color' => ['rgb' => '1E3A8A']
                ]
            ]
        ]);
        $sheet->getRowDimension($headerRow)->setRowHeight(30);

        // ====================
        // DADOS DA TABELA
        // ====================
        $sheet->getStyle('A' . $dataStartRow . ':J' . $dataEndRow)->applyFromArray([
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => ['rgb' => 'D1D5DB']
                ]
            ],
            'alignment' => [
                'vertical' => Alignment::VERTICAL_CENTER
            ]
        ]);

        // Alinhamento das colunas
        $sheet->getStyle('A' . $dataStartRow . ':A' . $dataEndRow)
            ->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // #
        $sheet->getStyle('B' . $dataStartRow . ':B' . $dataEndRow)
            ->getAlignment()->setHorizontal(Alignment::HORIZONTAL_LEFT); // Estudante
        $sheet->getStyle('C' . $dataStartRow . ':C' . $dataEndRow)
            ->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Código
        $sheet->getStyle('D' . $dataStartRow . ':D' . $dataEndRow)
            ->getAlignment()->setHorizontal(Alignment::HORIZONTAL_LEFT); // Turma
        $sheet->getStyle('E' . $dataStartRow . ':E' . $dataEndRow)
            ->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Referência
        $sheet->getStyle('F' . $dataStartRow . ':F' . $dataEndRow)
            ->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Mês/Ano
        $sheet->getStyle('G' . $dataStartRow . ':G' . $dataEndRow)
            ->getAlignment()->setHorizontal(Alignment::HORIZONTAL_RIGHT); // Valor
        $sheet->getStyle('H' . $dataStartRow . ':H' . $dataEndRow)
            ->getAlignment()->setHorizontal(Alignment::HORIZONTAL_RIGHT); // Multa
        $sheet->getStyle('I' . $dataStartRow . ':I' . $dataEndRow)
            ->getAlignment()->setHorizontal(Alignment::HORIZONTAL_RIGHT); // Total
        $sheet->getStyle('J' . $dataStartRow . ':J' . $dataEndRow)
            ->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Data

        // Zebra striping (linhas alternadas)
        for ($i = $dataStartRow; $i <= $dataEndRow; $i++) {
            if (($i - $dataStartRow) % 2 == 1) {
                $sheet->getStyle('A' . $i . ':J' . $i)->applyFromArray([
                    'fill' => [
                        'fillType' => Fill::FILL_SOLID,
                        'startColor' => ['rgb' => 'F9FAFB']
                    ]
                ]);
            }
        }

        // ====================
        // LINHAS DE TOTAIS
        // ====================
        for ($i = 0; $i < 3; $i++) {
            $row = $totalsStartRow + $i;
            $sheet->getStyle('F' . $row . ':G' . $row)->applyFromArray([
                'font' => [
                    'bold' => true,
                    'size' => 10
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_RIGHT
                ],
                'borders' => [
                    'top' => [
                        'borderStyle' => $i == 0 ? Border::BORDER_MEDIUM : Border::BORDER_THIN,
                        'color' => ['rgb' => '000000']
                    ]
                ]
            ]);

            // Destacar TOTAL GERAL
            if ($i == 2) {
                $sheet->getStyle('F' . $row . ':G' . $row)->applyFromArray([
                    'fill' => [
                        'fillType' => Fill::FILL_SOLID,
                        'startColor' => ['rgb' => 'DBEAFE']
                    ],
                    'font' => [
                        'bold' => true,
                        'size' => 11,
                        'color' => ['rgb' => '1E40AF']
                    ]
                ]);
            }
        }

        return [];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 15,   // #
            'B' => 30,   // Estudante
            'C' => 12,   // Código
            'D' => 18,   // Turma
            'E' => 15,   // Referência
            'F' => 12,   // Mês/Ano
            'G' => 13,   // Valor
            'H' => 13,   // Multa
            'I' => 13,   // Total
            'J' => 16,   // Data/Hora
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                // Configurar página para retrato (portrait)
                $event->sheet->getDelegate()->getPageSetup()
                    ->setOrientation(PageSetup::ORIENTATION_PORTRAIT)
                    ->setPaperSize(PageSetup::PAPERSIZE_A4)
                    ->setFitToWidth(1)
                    ->setFitToHeight(0);

                // Configurar margens
                $event->sheet->getDelegate()->getPageMargins()
                    ->setTop(0.5)
                    ->setRight(0.5)
                    ->setLeft(0.5)
                    ->setBottom(0.5);

                // Configurar impressão
                $event->sheet->getDelegate()->getPageSetup()
                    ->setHorizontalCentered(true);

                // Repetir cabeçalhos em todas as páginas
                $headerRow = 4 + count($this->filters) + 1 + 6 + 2;
                $event->sheet->getDelegate()->getPageSetup()
                    ->setRowsToRepeatAtTopByStartAndEnd($headerRow, $headerRow);
            },
        ];
    }

    public function title(): string
    {
        return 'Pagamentos';
    }
}
