<?php

namespace App\Http\Controllers;

use App\Models\Book;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\Student;
use App\Models\Classroom;
use App\Models\Exam;
use Illuminate\Support\Str;
use App\Models\Downloadsetings;
use App\Models\Downloadfile;
use App\Models\DownloadHistory;
use App\Models\Downloadcenter;
use PDF;
use App\Models\DeviceToken;
use Illuminate\Support\Carbon;
use App\Models\Noticeboard;
use App\Models\PaymentConfirmation;
use Illuminate\Support\Facades\Log;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;
use Livewire\Component;
use Kreait\Firebase\Exception\Messaging\NotFound;
use Illuminate\Support\Facades\Http;
use Google\Auth\Credentials\ServiceAccountCredentials;
use Google\Cloud\Storage\StorageClient;

class ApiController extends Controller
{
    
  
    public function login(Request $request)
    {
        // Validate the input data
        $validator = Validator::make($request->all(), [
            'student_id' => 'required',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'false',
                'errors' => $validator->errors(),
            ], 422);
        }

        if (Auth::attempt(['student_id' => $request->student_id, 'password' => $request->password])) {
            $user = Auth::user();

            if ($user->is_paid == 0) {
                return response()->json([
                    //'status' => false,
                    'error' => 'Sistema em manutencao, por favor tente mais tarde!'
                ], 403);
                
            }

            if ($user->is_active == 0) {
                return response()->json([
                    //'status' => 'false',
                    'error' => 'Sua conta encontra-se inactiva, contacte o suporte!'
                ], 403);
            }

            // Check if user already has an active token
            //$existingToken = $user->tokens()->where('revoked', false)->first();
            //if ($existingToken) {
            //    // Revoke the existing token
            //    $existingToken->revoke();
            // }

            // Generate a new token for the authenticated user
            $token = $user->createToken('API Token')->accessToken;

            return response()->json(['token' => $token], 200);
        } else {
            return response()->json([
                //'status' => 'false',
                'error' => 'Credenciais invalidas. Verifique seu ID e senha.'
            ], 401);
        }
    }

    




    public function examData(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }
        // Fetch exams with the related subject data using eager loading
        $examsQuery = Exam::with('subject') // Eager load the subject relationship
            ->where('student_id', $user->id);

        // Check if year is set in the request
        $examsQuery->when(isset($request['year']), function ($query) use ($request) {
            $year = $request['year'];
            return $query->where(function ($query) use ($year) {
                $query->where('year', $year)
                    ->orWhere('created_at', 'LIKE', '%' . $year . '%');
            });
        });


        // Execute the query
        $exams = $examsQuery->get();

        // Check if exams were found
        if ($exams->isEmpty()) {
            return response()->json([
                'status' => 'true',
                'exam' => 'No data found'
            ], 200);
        }

        // Group exams by trimester and subject
        $groupedExams = $exams->groupBy(['trimester_id', 'subject_id']);

        // Prepare the response data
        $responseData = [];

        foreach ($groupedExams as $trimesterId => $subjects) {
            $trimesterData = [];
            foreach ($subjects as $subjectId => $examGroup) {
                $subjectData = [
                    'subject_id' => $subjectId,
                    'subject_name' => $examGroup->first()->subject->name, // Get the subject name from the first exam in the group
                    'exams' => []
                ];

                // Add each exam in the subject group to the subject data
                foreach ($examGroup as $exam) {
                    $subjectData['exams'][] = [
                        'id' => $exam->id,
                        'ACS1' => $exam->ACS1a,
                        'ACS2' => $exam->ACS2a,
                        'MACS' => $exam->MACS,
                        'AT' => $exam->AT,
                        'MT' => $exam->MT,
                        'NE' => $exam->NE,
                        'is_active' => $exam->is_active,
                        'created_at' => $exam->created_at,
                        'updated_at' => $exam->updated_at,
                        'year' => $exam->year,
                        // Add other fields as necessary
                    ];
                }

                // Add the subject data to the trimester data
                $trimesterData[] = $subjectData;
            }

            // Add the trimester data to the response data
            $responseData[] = [
                'trimester_id' => $trimesterId,
                'subjects' => $trimesterData
            ];
        }

        // Return the grouped exam data in the response
        return response()->json([
            'status' => 'true',
            'exams' => $responseData
        ], 200);
    }


    public function listarMatrizes(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        // Verificar se é admin
        if ($user->role === 0) {
            // Admin pode ver tudo
            $downloads = Downloadsetings::orderByDesc('created_at')->get();
        } else {
            // Usuário comum só pode ver matrizes da sua turma
            $downloads = Downloadsetings::where(function ($query) use ($user) {
                    $query->where('type', 'all')
                          ->where('class_id', $user->classroom_id); // Garante que 'all' ainda respeita a turma do usuário
                })
                ->orWhere('class_id', $user->classroom_id)
                ->orderByDesc('created_at')
                ->get();
        }

        if ($downloads->isEmpty()) {
            return response()->json([
                'status' => 'true',
                'downloads' => "No data found"
            ], 200);
        }

        // Montar o array de resposta corretamente
        $data = [];
        foreach ($downloads as $model) {
            $downloadFiles = Downloadfile::where('download_id', $model->download_id)->get();
            foreach ($downloadFiles as $downloadFile) {
                $downloadCenter = Downloadcenter::where('slug', $model->download_id)->first();

                $item = [
                    'id' => $model->id,
                    'type' => $model->type,
                    'class_id' => $model->class_id,
                    'student_id' => $model->student_id,
                    'active' => $model->active,
                    'download_id' => $model->download_id,
                    'media_name' => $downloadFile->media_name,
                    'media_url' => $downloadFile->media_url,
                    'author' => $downloadCenter ? $downloadCenter->author : null,
                    'title' => $downloadCenter ? $downloadCenter->title : null,
                    'description' => $downloadCenter ? $downloadCenter->description : null,
                    'created_at' => $model->created_at,
                    'updated_at' => $model->updated_at,
                ];

                $data[] = $item;
            }
        }

        return response()->json([
            'status' => 'success',
            'downloads' => $data
        ], 200);
    }

    


    public function generatePdfReport(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);

        } else {
            // Get the current date
            $currentYear = Carbon::now()->year;

            // Calculate the last year
            $lastYear = $currentYear - 1;

            // Fetch exams for the current year
            $exams = Exam::query()
                ->where('student_id', $user->id)
                ->where('year', $currentYear)
                ->get();

            // If no exams found for the current year, fetch for the last year
            if ($exams->isEmpty()) {
                $exams = Exam::query()
                    ->where('student_id', $user->id)
                    ->where('year', $lastYear)
                    ->get();
            }

            $data = [
                'title' => 'COLÉGIO POLITÉCNICO DE MOÇAMBIQUE',
                'date' => date('d/m/Y'),
                'models' => $exams,
                'name' => $user->name,
                'id' => $user->student_id,
            ];

            // Generate the PDF
            $pdf = PDF::loadView('exports.userreport', $data);

            // Set options for the PDF (e.g., paper size, orientation, etc.)
            $pdf->setPaper('A4', 'portrait');

            // Add a page number
            $pdf->setOptions(['isPhpEnabled' => true]);

            // Return the PDF file as a response
            return $pdf->stream();
        }
    }



public function listarComunicados(Request $request)
{
    // Retrieve the authenticated user
    $user = Auth::guard('api')->user();

    if (!$user) {
        return response()->json([
            'status' => '401',
            'message' => 'Unauthorized! Invalid token.'
        ], 401);
    } else {
        // Retrieve the comunicados based on user role or permissions
        $comunicados = Noticeboard::where('type', 'all')
            ->orWhere('class_id', $user->classroom_id)
            ->orWhere('student_id', $user->id)
            ->orderByDesc('created_at')
            ->get();

        // Count the unseen and seen comunicados
        $unseenCount = Noticeboard::where('status', 0)->count();
        $seenCount = Noticeboard::where('status', 1)->count();

        if ($comunicados->isEmpty()) {
            return response()->json([
                'status' => 'true',
                'comunicados' => "No data found",
                'unseen_count' => $unseenCount,
                'seen_count' => $seenCount
            ], 200);
        } else {
            return response()->json([
                'status' => 'true',
                'comunicados' => $comunicados,
                'unseen_count' => $unseenCount,
                'seen_count' => $seenCount
            ], 200);
        }
    }
}

    public function ListarConfirmacoesPagamento(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        } else {
            // Fetch payment confirmations based on the user's role or permissions
            if ($user->role === 0) {
                // Admin user can access all payment confirmations
                $confirmations = PaymentConfirmation::orderByDesc('created_at')->get();

            } else {
                // Non-admin user's access based on role or permissions
                $confirmations = PaymentConfirmation::where('student_id', $user->id)
                    ->orWhere('student_id', $user->id)
                    ->orderByDesc('created_at')
                    ->get();
            }

            if ($confirmations->isEmpty()) {
                return response()->json([
                    'status' => 'true',
                    'confirmations' => "No data found"
                ], 200);
            }

            $data = [];
            foreach ($confirmations as $confirmation) {
                $item = [
                    'id' => $confirmation->id,
                    'uuid' => $confirmation->uuid,
                    'title' => $confirmation->title,
                    'type' => $confirmation->type,
                    'class_id' => $confirmation->class_id,
                    'student_id' => $confirmation->student_id,
                    'slug' => $confirmation->slug,
                    'media_name' => $confirmation->media_name,
                    'media_url' => $confirmation->media_url,
                    'created_at' => $confirmation->created_at,
                    'updated_at' => $confirmation->updated_at,
                    // Add more fields as needed
                ];
                $data[] = $item;
            }

            return response()->json([
                'status' => 'success',
                'confirmations' => $data
            ], 200);
        }
    }



    public function getUserClassName(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        // Get the user's classroom_id
        $classroomId = $user->classroom_id;

        // Fetch the classroom information based on the classroom_id
        $classroom = Classroom::find($classroomId);

        // Check if the classroom was found
        if (!$classroom) {
            return response()->json([
                'status' => '404',
                'message' => 'Classroom not found!'
            ], 404);
        }

        // Return the user's class name as JSON response
        return response()->json([
            'status' => 'success',
            'class' => $classroom->class,
            'turma' => $classroom->name
        ], 200);
    }


    public function getUserInfo(Request $request)
    {
        // Retrieve the authenticated user
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }
        // Return the user information as a JSON response
        return response()->json([
            'status' => 'success',
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone_number' => $user->phone_number,
                'avatar' => $user->avatar,
                // Add more fields as needed (e.g., address, date of birth, etc.)
            ]
        ], 200);
    }

   

public function listBooks(Request $request)
{
    // Retrieve the authenticated user
    $user = Auth::guard('api')->user();

    if (!$user) {
        return response()->json([
            'status' => '401',
            'message' => 'Unauthorized! Invalid token.'
        ], 401);
    }

    // Fetch active books
    $books = Book::where('is_active', 1)
        ->orderByDesc('created_at')
        ->get();

    // Count active and inactive books
    $activeCount = Book::where('is_active', 1)->count();
    $inactiveCount = Book::where('is_active', 0)->count();

    // Check if there are no active books
    if ($books->isEmpty()) {
        return response()->json([
            'status' => 'success',
            'books' => "No data found",
            'active_count' => $activeCount,
            'inactive_count' => $inactiveCount
        ], 200);
    }

    // Get the total number of students
    $totalStudents = Student::count();

    // Fetch all records in download history, including trashed ones
    $downloadHistory = DownloadHistory::withTrashed()->get();

    // Extract book IDs from download history
    $downloadedBookIds = $downloadHistory->pluck('book_id')->toArray();

    // Prepare the books data and categorize them
    $data = [];
    $downloadedBooks = [];
    $notDownloadedBooks = [];

    foreach ($books as $book) {
        $item = [
            'id' => $book->id,
            'title' => $book->title,
            'author' => $book->author,
            'description' => $book->description,
            'category_id' => $book->category_id,
            'slug' => $book->slug,
            'media_name' => $book->media_name,
            'media_url' => $book->media_url, // Cover image URL
            'pdf_name' => $book->pdf_name,
            'pdf_url' => $book->pdf_url, // PDF URL
            'created_at' => $book->created_at,
            'updated_at' => $book->updated_at,
            // Add more fields as needed
        ];

        $data[] = $item;

        if (in_array($book->id, $downloadedBookIds)) {
            $downloadedBooks[] = $item;
        } else {
            $notDownloadedBooks[] = $item;
        }
    }

    // Return JSON response with book data, categorized books, download counts, and other counts
    return response()->json([
        'status' => 'success',
        'books' => $data,
        'active_count' => $activeCount,
        'totalStudents' => $totalStudents,
        'inactive_count' => $inactiveCount,
        'downloaded_files_count' => count($downloadedBooks),
        'not_downloaded_files_count' => count($notDownloadedBooks)
    ], 200);
}	

public function downloadBooks(Request $request) {
    // Validate the incoming request
    $validator = Validator::make($request->all(), [
        'file_url' => 'required|string',
        'student_id' => 'nullable|string',
        'book_id' => 'required|string',
    ]);

    // Check if validation fails
    if ($validator->fails()) {
        return response()->json(['error' => $validator->errors()->first()], 422);
    }

    // Create a new instance of DownloadHistory and save the data
    $download = new DownloadHistory();
    $download->file_url = $request->input('file_url');
    $download->student_id = $request->input('student_id');
    $download->book_id = $request->input('book_id');
    $download->save();

    // Return a success response
    return response()->json(['message' => 'Download history saved successfully'], 200);
	}
        
        
        
public function storeDeviceTokenOld(Request $request, $userId)
{
    // Validate the incoming request
    $validator = Validator::make($request->all(), [
        'device_token' => 'required|string',
        'device_model' => 'nullable|string', // Allowing null if device_model is not provided
    ]);

    if ($validator->fails()) {
        return response()->json(['error' => $validator->errors()->first()], 422);
    }

    // Find the user by their student ID
    $user = User::where('student_id', $userId)->first();

    if (!$user) {
        return response()->json(['error' => 'User not found'], 404);
    }

    // Check if the token already exists for this user
    $existingToken = $user->deviceTokens()->where('token', $request->device_token)->first();

    if (!$existingToken) {
        // Check if the user has reached the maximum number of tokens
        if ($user->deviceTokens()->count() >= 50000) {
            return response()->json(['error' => 'Maximum device tokens reached'], 422);
        }

        // Create a new device token record with the model of the device
        $user->deviceTokens()->create([
            'token' => $request->device_token,
            'device_model' => $request->device_model ?? 'Desconhecido', // Default to 'Unknown' if device_model is null
        ]);
    }

    return response()->json(['message' => 'Device token and model saved successfully'], 200);
}
    
    public function storeDeviceToken(Request $request, $userId)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'device_token' => 'required|string',
            'device_model' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 422);
        }

        // Find the user
        $user = User::where('student_id', $userId)->first();
        if (!$user) {
            return response()->json(['error' => 'User not found'], 404);
        }

        // Check if this device token is already stored
        $existingToken = $user->deviceTokens()->where('token', $request->device_token)->first();

        if ($existingToken) {
            // Refresh the existing token's timestamp and model
            $existingToken->update([
                'device_model' => $request->device_model ?? 'Desconhecido',
                'updated_at' => now(),
            ]);
        } else {
            // ✅ Validate existing tokens before deleting
            $this->validateAndCleanTokens($user);

            // ✅ Ensure max 5 active tokens per user
            if ($user->deviceTokens()->count() >= 5) {
                $user->deviceTokens()->oldest()->first()->delete(); // Remove the oldest token
            }

            // Store the new token
            $user->deviceTokens()->create([
                'token' => $request->device_token,
                'device_model' => $request->device_model ?? 'Desconhecido',
            ]);
        }

        return response()->json(['message' => 'Device token updated successfully'], 200);
    }

    /**
     * 🔹 Validate existing tokens before deleting (FCM Test)
     */
    private function validateAndCleanTokens($user)
    {
        $tokens = $user->deviceTokens;

        foreach ($tokens as $token) {
            if (!$this->checkFCMToken($token->token)) {
                $token->delete(); // Remove invalid token
                Log::info('Token inválido removido: ' . $token->token);
            }
        }
    }

    
    private function checkFCMToken($deviceToken)
    {
        // Log the incoming device token for debugging purposes
            Log::info('Checking FCM Token: ' . $deviceToken);
        // 🔥 Hardcoded Firebase credentials (service account JSON as an array)
        $firebaseCredentials = [
            "type" => "service_account",
            "project_id" => "copmoz-3x",
            "private_key_id" => "bb9e045f0e2345c069b299a14636203509899ef8",
            "private_key" => "-----BEGIN PRIVATE KEY-----\nMIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDYfKrqJJyR6/8w\nP0NI75wA0OMzBMq53JD0GX/2G488VD2gE+Wbxt90e/6PUGdrOEwo7C+ilCNKsoX7\nzyy6l6kOt6MgVjO39bc7/qwaKYcA05/48grMrLMUxb6SNAMmKmj3CTQ9dU/zAmqA\n96HleqAkHt22wRSQ8hy4qg44CC1f8faBJxGb68YeMm9TmoYcdUhc7QT13JoMRZjl\nbi5gNDVIFWDyUhXeVpvw13Z3Sfo+EgkiT2Tb2Nr8//rptIabmrhzUsAFTEKnHblh\nY+1nsxBBwImu1NRtr5aQUCNLUZXcbWpuoym6EZtCLe3u6/mL3wI05407q1YqpUOq\nppfDwuHdAgMBAAECggEAMuv0Kfotp/dKgddHpObTm6+G2a/VXa2fnQNIgaPO5+Jq\n0nYLqr43PoeSA7TmdqZ556BSvhu5R29Bq876FvPojLEuWkNci/dsv9xyL/83pb6h\noOWSG729q/OtlWdNnFMaeI0+L/yhrygQ1EArYTacZ5KpFT1KcKO6lMdPa+ekgK4T\nPuwGVD/3VDFG9AWCEm+v3w6bQAhpKbdZiBPpGvTrfFHYov97/Yzv/iC3599VnW+i\no3GweFGaqsOn4PxuyIyYujzgoIc6kBrivUV+ddxZqLufrsQL1uE78SuWjhRNS8/k\n5rUD3Qvaj3BOQzkqCnttclfPZYSmbjMtVyLOMjMcQQKBgQDkE1SnrhY/dZF5QCFg\nIMaf7wkPLLCSStugOeYMGx2oeD0MxaY9L9N1ffRLekngffzANrO/ybE7AlRzgBJB\n2NYFd7fueihmrWuTpMafkBivG+mf+HdTjkxF0aQfYUL+MMW7viBBzGDbwckRSTX5\n5pSqzEHmRvctiLTfHqqgUnDeIQKBgQDy/hzbdteqvgqRok7KoXy3ciij4SQbDOQZ\n19KcmMTFgMiT/xm/3uni924Qw82VVSYQjDbAoSltTl7cpvtkXzSB2LIg00YEhfcE\nj/atYfswDIBYBAGCUZDalOn6s/8TWuTGVfCkQVwHA2Xa3rKqIrI1gHTuu5N4J/P4\niCmL3Vd0PQKBgQC1mirC5Rvz5ZIywHySxSZCfJtzCLuDkLV1zAQ/yWuBFFRQEiS+\n/ZDbMbJOUw32AXs6NZREdr125fEGkoh3A8fOTLgY3A3FS/qncgFxVdRBwfDxHm8t\nCdXzleyfy9sC+STIy6d1nN3WvWNzLx8aX54qrT8fs4vnJr4WFp330Azs4QKBgQCc\nnDj/Hb5sdmWbbW+425HlLfeRf+bkZE+TdG1yrmeMH6+m6zCuYD6AIRbYSUp0J9gL\nrEMRrg1kPLGZJyo5i77svTw5OdIT0j/dueez4cWiNzx0/cf3NRjWOEoBYgdRczCl\nv/gD9XgZsQ/xm7ytOQWAxBUZVN38AEwW77NPSM43xQKBgCgCmFCO57u8Fc6d+4G9\nBy/Vfmy3tZpwGCTGqxkwhFvG+qXZTERJOwRg4jejIM/W6iQlOPmfsis2CWhCZvYL\nSimfvuVeWSVixdnsxVpjVyyT7ci9jMhPXeL7LtMCjNF4hB34qgejqpkwh62Zqg0h\nNuy5ZdoARi0VwY/spf6HK03b\n-----END PRIVATE KEY-----\n",
            "client_email" => "firebase-adminsdk-as4sm@copmoz-3x.iam.gserviceaccount.com",
            "client_id" => "112030584520787629755",
            "auth_uri" => "https://accounts.google.com/o/oauth2/auth",
            "token_uri" => "https://oauth2.googleapis.com/token",
            "auth_provider_x509_cert_url" => "https://www.googleapis.com/oauth2/v1/certs",
            "client_x509_cert_url" => "https://www.googleapis.com/robot/v1/metadata/x509/firebase-adminsdk-as4sm%40copmoz-3x.iam.gserviceaccount.com",
            "universe_domain" => "googleapis.com"
        ];

        // Initialize Firebase with the credentials
        $factory = (new Factory)->withServiceAccount($firebaseCredentials);

        // 🔥 Definição correta dos escopos
        $scopes = ['https://www.googleapis.com/auth/firebase.messaging'];

        // 🔥 Passa o JSON e os escopos corretamente
        $credentials = new ServiceAccountCredentials($scopes, $firebaseCredentials);

        // Get the OAuth 2.0 token using the service account
        $token = $credentials->fetchAuthToken();

        // Set up the Authorization header with the Bearer token
        $authHeader = 'Bearer ' . $token['access_token'];

        // Prepare the FCM request data
        $data = [
            'message' => [
                'token' => $deviceToken,
                'notification' => [
                    'title' => 'COPMOZ',
                    'body' => 'Fique atento às novidades e melhorias. 🚀🔔'
                ],
                'android' => [
                    'priority' => 'high'
                ]
            ]
        ];
        // Log the request data for debugging purposes
        Log::info('Sending FCM request with data: ' . json_encode($data));

        // Send the request to FCM API v1
        $response = Http::withHeaders([
            'Authorization' => $authHeader,
            'Content-Type' => 'application/json',
        ])->post('https://fcm.googleapis.com/v1/projects/copmoz-3x/messages:send', $data);
        
        // Log the response from the API for debugging purposes
        Log::info('FCM response: ' . $response->body());

        // Handle the response
        $result = $response->json();
        // Log the result to check if there are any errors
        Log::info('FCM response JSON: ' . json_encode($result));

        // Return whether the token is valid
        return !(isset($result['error']) && in_array($result['error']['status'], ['INVALID_ARGUMENT', 'RESOURCE_EXHAUSTED']));
    }


    
public function removeDeviceToken(Request $request)
{
    try {
        $userId = $request->input('user_id');
        $deviceToken = $request->input('token');
        
        Log::info('Received userId: ' . $userId);
        Log::info('Received deviceToken: ' . $deviceToken);

        if (!$userId || !$deviceToken) {
            return response()->json(['error' => 'User ID and token are required'], 400);
        }

        // Log the SQL query for debugging
        $query = DeviceToken::where('user_id', $userId)
                             ->where('token', $deviceToken);
        Log::info('SQL Query: ' . $query->toSql());

        // Verificar se o token realmente existe
        $exists = $query->exists();
        Log::info('Token exists: ' . ($exists ? 'Yes' : 'No'));

        // Find and remove the token associated with this userId
        $deleted = $query->delete();
        
        if ($deleted) {
            return response()->json(['message' => 'Device token removed successfully'], 200);
        } else {
            return response()->json(['error' => 'Token not found'], 404);
        }
    } catch (\Exception $e) {
        Log::error('Error removing device token: '.$e->getMessage());
        return response()->json(['error' => 'Internal Server Error'], 500);
    }
}
    
    public function sendPushNotification(Request $request)
        {
            try {
                // Validate request data
                $request->validate([
                    'token' => 'required|string',
                    'message' => 'required|string',
                    'title' => 'required|string',
                    'sound' => 'nullable|string',
                    'badge' => 'nullable|integer',
                    'data' => 'nullable|array',
                ]);

                // Extract notification data
                $token = $request->input('token');
                $message = $request->input('message');
                $title = $request->input('title');
                $sound = $request->input('sound', 'default'); // Default sound if not provided
                $badge = $request->input('badge', 1); // Default badge if not provided
                $data = $request->input('data', []);

                // Code to send the push notification to Firebase (FCM)
                // Use Firebase SDK or custom logic to send the notification
                // Example: Send to Firebase Cloud Messaging (FCM) API or other service

                // Here you would typically send the request to FCM
                // Example using Firebase Admin SDK (PHP):
                // $fcm->sendTo($token, $title, $message, $data, $sound, $badge);

                return response()->json(['message' => 'Notification sent successfully.'], 200);

            } catch (\Exception $e) {
                Log::error('Error sending push notification: ' . $e->getMessage());
                return response()->json(['error' => 'Failed to send notification.'], 500);
            }
        }



    

    


//------------------------------------------------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------------------------------------------------

    protected $perPage = 30;

    /**
     * Authinticate the connection for pusher
     *
     * @param Request $request
     * @return void
     */
    public function pusherAuth(Request $request)
    {
        return Chatify::pusherAuth(
            $request->user(),
            Auth::user(),
            $request['channel_name'],
            $request['socket_id']
        );
    }

    /**
     * Fetch data by id for (user/group)
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function idFetchData(Request $request)
    {
        return auth()->user();
        // Favorite
        $favorite = Chatify::inFavorite($request['id']);

        // User data
        if ($request['type'] == 'user') {
            $fetch = User::where('id', $request['id'])->first();
            if ($fetch) {
                $userAvatar = Chatify::getUserWithAvatar($fetch)->avatar;
            }
        }

        // send the response
        return Response::json([
            'favorite' => $favorite,
            'fetch' => $fetch ?? null,
            'user_avatar' => $userAvatar ?? null,
        ]);
    }

    /**
     * This method to make a links for the attachments
     * to be downloadable.
     *
     * @param string $fileName
     * @return \Illuminate\Http\JsonResponse
     */
    public function download($fileName)
    {
        $path = config('chatify.attachments.folder') . '/' . $fileName;
        if (Chatify::storage()->exists($path)) {
            return response()->json([
                'file_name' => $fileName,
                'download_path' => Chatify::storage()->url($path)
            ], 200);
        } else {
            return response()->json([
                'message' => "Sorry, File does not exist in our server or may have been deleted!"
            ], 404);
        }
    }

    /**
     * Send a message to database
     *
     * @param Request $request
     * @return JSON response
     */
    public function send(Request $request)
    {
        // default variables
        $error = (object) [
            'status' => 0,
            'message' => null
        ];
        $attachment = null;
        $attachment_title = null;

        // if there is attachment [file]
        if ($request->hasFile('file')) {
            // allowed extensions
            $allowed_images = Chatify::getAllowedImages();
            $allowed_files = Chatify::getAllowedFiles();
            $allowed = array_merge($allowed_images, $allowed_files);

            $file = $request->file('file');
            // check file size
            if ($file->getSize() < Chatify::getMaxUploadSize()) {
                if (in_array(strtolower($file->extension()), $allowed)) {
                    // get attachment name
                    $attachment_title = $file->getClientOriginalName();
                    // upload attachment and store the new name
                    $attachment = Str::uuid() . "." . $file->extension();
                    $file->storeAs(config('chatify.attachments.folder'), $attachment, config('chatify.storage_disk_name'));
                } else {
                    $error->status = 1;
                    $error->message = "File extension not allowed!";
                }
            } else {
                $error->status = 1;
                $error->message = "File size you are trying to upload is too large!";
            }
        }

        if (!$error->status) {
             // send to database
             $message = Chatify::newMessage([
                'type' => $request['type'],
                'from_id' => Auth::user()->id,
                'to_channel_id' => $request['channel_id'],
                'body' => trim($request['message']),  // Removed htmlentities
                'attachment' => ($attachment) ? json_encode((object)[
                    'new_name' => $attachment,
                    'old_name' => trim($attachment_title),  // Removed htmlentities
                ]) : null,
            ]);
        
            // Processa a mensagem para preparar a resposta
            $messageData = Chatify::parseMessage($message);
        
            // Envia a mensagem para o usuário destinatário usando Pusher
            // Verifica se o remetente não é o destinatário
            if (Auth::user()->id != $request['id']) {
                Chatify::push("private-chatify." . $request['id'], 'messaging', [
                    'from_id' => Auth::user()->id, // ID do remetente
                    'to_id' => $request['id'], // ID do destinatário
                    'message' => $messageData // Dados da mensagem processados
                ]);
            }

            
            // Send notification after successfully saving the message
            $this->sendNotification($request['channel_id'], $message, $attachment);
        }
        

        // send the response
        return Response::json([
            'status' => '200',
            'error' => $error,
            'message' => $messageData ?? [],
            'tempID' => $request['temporaryMsgId'],
        ]);
    }

    /**
     * fetch [user/group] messages from database
     *
     * @param Request $request
     * @return JSON response
     */
    public function fetch(Request $request)
    {
        $query = Chatify::fetchMessagesQuery($request['id'])->latest();
        $messages = $query->paginate($request->per_page ?? $this->perPage);
        $totalMessages = $messages->total();
        $lastPage = $messages->lastPage();
        $response = [
            'total' => $totalMessages,
            'last_page' => $lastPage,
            'last_message_id' => collect($messages->items())->last()->id ?? null,
            'messages' => $messages->items(),
        ];
        return Response::json($response);
    }

    /**
     * Make messages as seen
     *
     * @param Request $request
     * @return void
     */
    public function seen(Request $request)
    {
        try {
            $userId = Auth::user()->id;
            // Aceitar tanto 'id' quanto 'channel_id' (JS da web envia 'channel_id')
            $channelId = $request['channel_id'] ?? $request['id'];

            Log::info('📩 Marcando mensagens como vistas', [
                'user_id' => $userId,
                'channel_or_user_id' => $channelId,
                'request_data' => $request->all(),
            ]);

            // Tentar marcar como visto usando canal primeiro
            $updatedCount = Message::where('to_channel_id', $channelId)
                ->where('from_id', '!=', $userId) // Não marcar mensagens próprias
                ->where('seen', 0)
                ->update([
                    'seen' => 1,
                    'updated_at' => now()
                ]);

            // Se não atualizou nada, tentar método antigo (conversas diretas sem canal)
            if ($updatedCount === 0) {
                $updatedCount = Message::where('from_id', $channelId)
                    ->where('to_id', $userId)
                    ->where('seen', 0)
                    ->update([
                        'seen' => 1,
                        'updated_at' => now()
                    ]);
            }

            Log::info('✅ Mensagens marcadas como vistas', [
                'channel_or_user_id' => $channelId,
                'updated_count' => $updatedCount,
            ]);

            return Response::json([
                'status' => 1,
                'updated_count' => $updatedCount,
            ], 200);

        } catch (\Exception $e) {
            Log::error('❌ Erro ao marcar mensagens como vistas: ' . $e->getMessage());
            return Response::json([
                'status' => 0,
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get contacts list
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse response
     */
    public function getContacts(Request $request)
    {
        // get all users that received/sent message from/to [Auth user]
        $users = Message::join('users', function ($join) {
            $join->on('ch_messages.from_id', '=', 'users.id')
                ->orOn('ch_messages.to_id', '=', 'users.id');
        })
            ->where(function ($q) {
                $q->where('ch_messages.from_id', Auth::user()->id)
                    ->orWhere('ch_messages.to_id', Auth::user()->id);
            })
            ->where('users.id', '!=', Auth::user()->id)
            ->select('users.*', DB::raw('MAX(ch_messages.created_at) max_created_at'))
            ->orderBy('max_created_at', 'desc')
            ->groupBy('users.id')
            ->paginate($request->per_page ?? $this->perPage);

        return response()->json([
            'contacts' => $users->items(),
            'total' => $users->total() ?? 0,
            'last_page' => $users->lastPage() ?? 1,
        ], 200);
    }

    /**
     * Put a user in the favorites list
     *
     * @param Request $request
     * @return void
     */
    public function favorite(Request $request)
    {
        $userId = $request['user_id'];
        // check action [star/unstar]
        $favoriteStatus = Chatify::inFavorite($userId) ? 0 : 1;
        Chatify::makeInFavorite($userId, $favoriteStatus);

        // send the response
        return Response::json([
            'status' => @$favoriteStatus,
        ], 200);
    }

    /**
     * Get favorites list
     *
     * @param Request $request
     * @return void
     */
    public function getFavorites(Request $request)
    {
        $favorites = Favorite::where('user_id', Auth::user()->id)->get();
        foreach ($favorites as $favorite) {
            $favorite->user = User::where('id', $favorite->favorite_id)->first();
        }
        return Response::json([
            'total' => count($favorites),
            'favorites' => $favorites ?? [],
        ], 200);
    }

    /**
     * Search in messenger
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function search(Request $request)
    {
        $input = trim(filter_var($request['input']));
        $records = User::where('id', '!=', Auth::user()->id)
            ->where('name', 'LIKE', "%{$input}%")
            ->paginate($request->per_page ?? $this->perPage);

        foreach ($records->items() as $index => $record) {
            $records[$index] += Chatify::getUserWithAvatar($record);
        }

        return Response::json([
            'records' => $records->items(),
            'total' => $records->total(),
            'last_page' => $records->lastPage()
        ], 200);
    }

    /**
     * Get shared photos
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function sharedPhotos(Request $request)
    {
        $images = Chatify::getSharedPhotos($request['user_id']);

        foreach ($images as $image) {
            $image = asset(config('chatify.attachments.folder') . $image);
        }
        // send the response
        return Response::json([
            'shared' => $images ?? [],
        ], 200);
    }

    /**
     * Delete conversation
     *
     * @param Request $request
     * @return void
     */
    public function deleteConversation(Request $request)
    {
        // delete
        $delete = Chatify::deleteConversation($request['id']);

        // send the response
        return Response::json([
            'deleted' => $delete ? 1 : 0,
        ], 200);
    }

    public function updateSettings(Request $request)
    {
        $msg = null;
        $error = $success = 0;

        // dark mode
        if ($request['dark_mode']) {
            $request['dark_mode'] == "dark"
                ? User::where('id', Auth::user()->id)->update(['dark_mode' => 1])  // Make Dark
                : User::where('id', Auth::user()->id)->update(['dark_mode' => 0]); // Make Light
        }

        // If messenger color selected
        if ($request['messengerColor']) {
            $messenger_color = trim(filter_var($request['messengerColor']));
            User::where('id', Auth::user()->id)
                ->update(['messenger_color' => $messenger_color]);
        }
        // if there is a [file]
        if ($request->hasFile('avatar')) {
            // allowed extensions
            $allowed_images = Chatify::getAllowedImages();

            $file = $request->file('avatar');
            // check file size
            if ($file->getSize() < Chatify::getMaxUploadSize()) {
                if (in_array(strtolower($file->extension()), $allowed_images)) {
                    // delete the older one
                    if (Auth::user()->avatar != config('chatify.user_avatar.default')) {
                        $path = Chatify::getUserAvatarUrl(Auth::user()->avatar);
                        if (Chatify::storage()->exists($path)) {
                            Chatify::storage()->delete($path);
                        }
                    }
                    // upload
                    $avatar = Str::uuid() . "." . $file->extension();
                    $update = User::where('id', Auth::user()->id)->update(['avatar' => $avatar]);
                    $file->storeAs(config('chatify.user_avatar.folder'), $avatar, config('chatify.storage_disk_name'));
                    $success = $update ? 1 : 0;
                } else {
                    $msg = "File extension not allowed!";
                    $error = 1;
                }
            } else {
                $msg = "File size you are trying to upload is too large!";
                $error = 1;
            }
        }

        // send the response
        return Response::json([
            'status' => $success ? 1 : 0,
            'error' => $error ? 1 : 0,
            'message' => $error ? $msg : 0,
        ], 200);
    }

    /**
     * Set user's active status
     *
     * @param Request $request
     * @return void
     */
    public function setActiveStatus(Request $request)
    {
        $activeStatus = $request['status'] > 0 ? 1 : 0;
        $status = User::where('id', Auth::user()->id)->update(['active_status' => $activeStatus]);
        return Response::json([
            'status' => $status,
        ], 200);
    }

  protected function sendNotification($channelId, $message, $attachment)
    {
        try {
            // Retrieve user IDs associated with the channel from the ch_channel_user table
            $userIds = DB::table('ch_channel_user')
                ->where('channel_id', $channelId)
                ->pluck('user_id');
    
            if ($userIds->isEmpty()) {
                throw new \Exception("No users found for channel_id: " . $channelId);
            }
    
            $firebaseCredentials = [
                "type" => "service_account",
                "project_id" => "copmoz-3x",
                "private_key_id" => "bb9e045f0e2345c069b299a14636203509899ef8",
                "private_key" => "-----BEGIN PRIVATE KEY-----\nMIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDYfKrqJJyR6/8w\nP0NI75wA0OMzBMq53JD0GX/2G488VD2gE+Wbxt90e/6PUGdrOEwo7C+ilCNKsoX7\nzyy6l6kOt6MgVjO39bc7/qwaKYcA05/48grMrLMUxb6SNAMmKmj3CTQ9dU/zAmqA\n96HleqAkHt22wRSQ8hy4qg44CC1f8faBJxGb68YeMm9TmoYcdUhc7QT13JoMRZjl\nbi5gNDVIFWDyUhXeVpvw13Z3Sfo+EgkiT2Tb2Nr8//rptIabmrhzUsAFTEKnHblh\nY+1nsxBBwImu1NRtr5aQUCNLUZXcbWpuoym6EZtCLe3u6/mL3wI05407q1YqpUOq\nppfDwuHdAgMBAAECggEAMuv0Kfotp/dKgddHpObTm6+G2a/VXa2fnQNIgaPO5+Jq\n0nYLqr43PoeSA7TmdqZ556BSvhu5R29Bq876FvPojLEuWkNci/dsv9xyL/83pb6h\noOWSG729q/OtlWdNnFMaeI0+L/yhrygQ1EArYTacZ5KpFT1KcKO6lMdPa+ekgK4T\nPuwGVD/3VDFG9AWCEm+v3w6bQAhpKbdZiBPpGvTrfFHYov97/Yzv/iC3599VnW+i\no3GweFGaqsOn4PxuyIyYujzgoIc6kBrivUV+ddxZqLufrsQL1uE78SuWjhRNS8/k\n5rUD3Qvaj3BOQzkqCnttclfPZYSmbjMtVyLOMjMcQQKBgQDkE1SnrhY/dZF5QCFg\nIMaf7wkPLLCSStugOeYMGx2oeD0MxaY9L9N1ffRLekngffzANrO/ybE7AlRzgBJB\n2NYFd7fueihmrWuTpMafkBivG+mf+HdTjkxF0aQfYUL+MMW7viBBzGDbwckRSTX5\n5pSqzEHmRvctiLTfHqqgUnDeIQKBgQDy/hzbdteqvgqRok7KoXy3ciij4SQbDOQZ\n19KcmMTFgMiT/xm/3uni924Qw82VVSYQjDbAoSltTl7cpvtkXzSB2LIg00YEhfcE\nj/atYfswDIBYBAGCUZDalOn6s/8TWuTGVfCkQVwHA2Xa3rKqIrI1gHTuu5N4J/P4\niCmL3Vd0PQKBgQC1mirC5Rvz5ZIywHySxSZCfJtzCLuDkLV1zAQ/yWuBFFRQEiS+\n/ZDbMbJOUw32AXs6NZREdr125fEGkoh3A8fOTLgY3A3FS/qncgFxVdRBwfDxHm8t\nCdXzleyfy9sC+STIy6d1nN3WvWNzLx8aX54qrT8fs4vnJr4WFp330Azs4QKBgQCc\nnDj/Hb5sdmWbbW+425HlLfeRf+bkZE+TdG1yrmeMH6+m6zCuYD6AIRbYSUp0J9gL\nrEMRrg1kPLGZJyo5i77svTw5OdIT0j/dueez4cWiNzx0/cf3NRjWOEoBYgdRczCl\nv/gD9XgZsQ/xm7ytOQWAxBUZVN38AEwW77NPSM43xQKBgCgCmFCO57u8Fc6d+4G9\nBy/Vfmy3tZpwGCTGqxkwhFvG+qXZTERJOwRg4jejIM/W6iQlOPmfsis2CWhCZvYL\nSimfvuVeWSVixdnsxVpjVyyT7ci9jMhPXeL7LtMCjNF4hB34qgejqpkwh62Zqg0h\nNuy5ZdoARi0VwY/spf6HK03b\n-----END PRIVATE KEY-----\n",
                "client_email" => "firebase-adminsdk-as4sm@copmoz-3x.iam.gserviceaccount.com",
                "client_id" => "112030584520787629755",
                "auth_uri" => "https://accounts.google.com/o/oauth2/auth",
                "token_uri" => "https://oauth2.googleapis.com/token",
                "auth_provider_x509_cert_url" => "https://www.googleapis.com/oauth2/v1/certs",
                "client_x509_cert_url" => "https://www.googleapis.com/robot/v1/metadata/x509/firebase-adminsdk-as4sm%40copmoz-3x.iam.gserviceaccount.com",
                "universe_domain" => "googleapis.com"
            ];
    
            $factory = (new Factory)->withServiceAccount($firebaseCredentials);
            $messaging = $factory->createMessaging();
            Log::info('Firebase Messaging initialized');
    
            // For each user in the channel, send a notification
            foreach ($userIds as $userId) {
                $user = User::where('id', $userId)->first();
                if (!$user) {
                    Log::error("User not found for ID: $userId");
                    continue;
                }
    
                // Get device tokens for the user
                $tokens = DeviceToken::where('user_id', $user->id)->pluck('token');
                if ($tokens->isEmpty()) {
                    Log::error("No device tokens found for user: " . $user->name);
                    continue;
                }
    
                Log::info('Device tokens retrieved for user: ' . $user->name, ['tokens' => $tokens]);
    
                // Send notification for each device token
                foreach ($tokens as $token) {
                    try {
                        $notification = Notification::create(
                            "Nova mensagem de " . Auth::user()->name,
                            ($attachment) ? "Ficheiro anexado" : $message->body
                        );

                        $firebaseMessage = CloudMessage::withTarget('token', $token)
                            ->withNotification($notification)
                            ->withData([
                                'channel_id' => (string) $channelId,
                                'sender_id' => (string) Auth::user()->id,
                                'sender_name' => Auth::user()->name ?? 'Usuário',
                                'message_type' => 'chat',
                                'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                            ])
                            ->withHighestPossiblePriority()
                            ->withDefaultSounds();

                        $messaging->send($firebaseMessage);
                        Log::info('Notification sent to token: ' . $token);
    
                    } catch (\Kreait\Firebase\Exception\Messaging\InvalidMessage $e) {
                        Log::error("Invalid token: {$token}. Error: {$e->getMessage()}");
                        DeviceToken::where('token', $token)->delete();  // Remove invalid token
                    } catch (\Exception $e) {
                        Log::error("Error sending notification: " . $e->getMessage());
                    }
                }
            }
    
        } catch (\Exception $e) {
            Log::error('Error sending notifications: ' . $e->getMessage());
        }
    }

 	public function updateCommunicadoSeen($id)
{
    // Validate the id
    if (!$id) {
        return response()->json(['error' => 'No ID provided'], 400);
    }

    // Check if the message exists
    $message = \DB::table('noticeboards')->where('id', $id)->first();

    // Return the message for debugging purposes (you can remove this line later)
    // return response()->json($message);

    if (!$message) {
        return response()->json(['error' => 'Message not found'], 404);
    }

    // Check if the status is already 1
    if ($message->status == 1) {
        return response()->json(['info' => 'Message is already viewed'], 200);
    }

    // Update the message status
    $updated = \DB::table('noticeboards')->where('id', $id)->update(['status' => 1]);

    if ($updated) {
        return response()->json(['success' => 'Message viewed successfully'], 200);
    } else {
        return response()->json(['error' => 'Failed to update message status'], 500);
    }
}


//====================================================================================================
//====================================================================================================
//
//  MÓDULO DE PAGAMENTOS API - SEÇÃO CONSOLIDADA
//  Consolidado de FeeApiController e PaymentApiController
//  Data: 2025
//
//====================================================================================================
//====================================================================================================

    /**
     * Get student payment summary for current year
     * GET /api/v2/student/payments?student_id=123&year=2025
     */
    public function getStudentPayments(Request $request)
    {
        try {
            $studentId = $request->get('student_id');

            if (!$studentId) {
                return response()->json([
                    'success' => false,
                    'message' => 'student_id é obrigatório'
                ], 400);
            }

            $user = \App\Models\User::find($studentId);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'Estudante não encontrado'
                ], 404);
            }

            $year = $request->get('year', now()->year);

            // Use FeeCalculationService para obter dados unificados
            $feeService = app(\App\Services\FeeCalculationService::class);
            $summary = $feeService->calculateYearSummary($user, (int)$year);

            return response()->json([
                'success' => true,
                'data' => [
                    'student' => [
                        'id' => $user->id,
                        'student_id' => $user->student_id,
                        'name' => $user->name,
                        'class' => $user->class->class ?? null,
                        'classroom' => $user->classroom->name ?? null,
                    ],
                    'year' => $year,
                    'summary' => [
                        'total_expected' => $summary['total_expected'],
                        'total_paid' => $summary['total_paid'],
                        'total_pending' => $summary['total_pending'],
                        'total_overdue' => $summary['total_overdue'],
                        'total_fines' => $summary['total_fines'],
                        'total_discounts' => $summary['total_discounts'] ?? 0,
                    ],
                    'payment_progress' => [
                        'months_total' => 12,
                        'months_paid' => count(array_filter($summary['months_detail'], fn($m) => $m['is_paid'])),
                        'percentage' => $summary['total_expected'] > 0
                            ? round(($summary['total_paid'] / $summary['total_expected']) * 100, 1)
                            : 0,
                    ],
                    'months' => array_map(function($month) use ($user) {
                        // Buscar referência pendente para este mês
                        $pendingReference = \App\Models\PaymentReference::where([
                            'student_id' => $user->id,
                            'fee_month' => $month['month'],
                            'fee_year' => $month['year'],
                            'status' => 'pending'
                        ])
                        ->where('expires_at', '>', now()) // Apenas não expiradas
                        ->first();

                        return [
                            'month' => $month['month'],
                            'year' => $month['year'],
                            'base_amount' => (float) $month['base_amount'],
                            'fine_amount' => (float) $month['fine_amount'],
                            'discount_applied' => (float) ($month['discount_applied'] ?? 0),
                            'total_expected' => (float) $month['total_expected'],
                            'is_paid' => (bool) $month['is_paid'],
                            'paid_amount' => (float) ($month['paid_amount'] ?? 0),
                            'is_overdue' => (bool) $month['is_overdue'],
                            'due_date' => $month['due_date']?->format('Y-m-d'),
                            'payment_date' => $month['payment_date']?->format('Y-m-d H:i:s'),
                            'fee_names' => $month['fee_names'] ?? 'Taxa Escolar',
                            'status' => $month['is_paid'] ? 'paid' : ($month['is_overdue'] ? 'overdue' : 'pending'),
                            'payment_id' => $month['existing_payment']?->id ?? null,
                            'has_receipt' => !empty($month['existing_payment']),
                            'pending_reference' => $pendingReference ? [
                                'id' => $pendingReference->id,
                                'reference_number' => $pendingReference->reference_number,
                                'entity_code' => $pendingReference->entity_code,
                                'amount' => (float) $pendingReference->amount,
                                'expires_at' => \Carbon\Carbon::parse($pendingReference->expires_at)->format('Y-m-d H:i:s'),
                                'days_until_expiry' => now()->diffInDays(\Carbon\Carbon::parse($pendingReference->expires_at), false),
                            ] : null,
                        ];
                    }, $summary['months_detail']),
                ],
            ], 200);

        } catch (\Exception $e) {
            Log::error('API: Error getting payments', [
                'student_id' => $request->get('student_id'),
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erro ao buscar pagamentos',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get payment history (all years)
     * GET /api/v2/student/payment-history?student_id=123
     */
    public function getStudentPaymentHistory(Request $request)
    {
        try {
            $studentId = $request->get('student_id');

            if (!$studentId) {
                return response()->json([
                    'success' => false,
                    'message' => 'student_id é obrigatório'
                ], 400);
            }

            $user = \App\Models\User::find($studentId);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'Estudante não encontrado'
                ], 404);
            }

            // Buscar todos os pagamentos do estudante
            $payments = \App\Models\Fee_assign::where('student_id', $user->id)
                ->where('status', 'paid')
                ->orderBy('year', 'desc')
                ->orderByRaw("FIELD(month, 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro')")
                ->get();

            // Buscar números de recibo para cada pagamento
            $history = $payments->map(function($payment) {
                $receiptNumber = \App\Models\ReceiptNumber::where('receipt_type', 'fee_assign')
                    ->where('record_id', $payment->id)
                    ->first();

                return [
                    'payment_id' => $payment->id,
                    'receipt_number' => $receiptNumber ? $receiptNumber->receipt_number : null,
                    'month' => $payment->month,
                    'year' => $payment->year,
                    'amount' => (float) $payment->amount,
                    'fine' => (float) ($payment->fine ?? 0),
                    'discount' => (float) ($payment->discount ?? 0),
                    'total' => (float) ($payment->amount + ($payment->fine ?? 0) - ($payment->discount ?? 0)),
                    'payment_method' => $payment->pay_type === 'POS' ? 'POS' : 'Dinheiro',
                    'payment_date' => $payment->payment_date ?? $payment->created_at,
                    'transaction_id' => $payment->transaction_id ?? null,
                    'has_receipt' => (bool) $receiptNumber,
                ];
            });

            // Estatísticas
            $stats = [
                'total_payments' => $payments->count(),
                'total_paid' => $payments->sum(function($p) {
                    return $p->amount + ($p->fine ?? 0) - ($p->discount ?? 0);
                }),
                'years' => $payments->pluck('year')->unique()->values(),
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'student' => [
                        'id' => $user->id,
                        'student_id' => $user->student_id,
                        'name' => $user->name,
                    ],
                    'statistics' => $stats,
                    'payments' => $history,
                ],
            ], 200);

        } catch (\Exception $e) {
            Log::error('API: Error getting payment history', [
                'student_id' => $request->get('student_id'),
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erro ao buscar histórico de pagamentos',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get payment receipt details
     * GET /api/v2/student/payments/{id}/receipt?student_id=123
     */
    public function getStudentReceiptDetails(Request $request, $paymentId)
    {
        try {
            $studentId = $request->get('student_id');

            if (!$studentId) {
                return response()->json([
                    'success' => false,
                    'message' => 'student_id é obrigatório'
                ], 400);
            }

            $user = \App\Models\User::find($studentId);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'Estudante não encontrado'
                ], 404);
            }

            // Buscar pagamento
            $payment = \App\Models\Fee_assign::where('id', $paymentId)
                ->where('student_id', $user->id)
                ->first();

            if (!$payment) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pagamento não encontrado',
                ], 404);
            }

            // Verificar se tem número de recibo
            $receiptNumber = \App\Models\ReceiptNumber::where('receipt_type', 'fee_assign')
                ->where('record_id', $payment->id)
                ->first();

            // Se não tem, gerar agora
            if (!$receiptNumber) {
                $receiptNumberStr = \App\Services\ReceiptNumberService::getOrGenerateReceiptNumber(
                    'fee_assign',
                    $payment->id,
                    'mobile',
                    $user->id
                );
            } else {
                $receiptNumberStr = $receiptNumber->receipt_number;
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'receipt_number' => $receiptNumberStr,
                    'payment_id' => $payment->id,
                    'student' => [
                        'name' => $user->name,
                        'student_id' => $user->student_id,
                        'class' => $user->class->class ?? null,
                        'classroom' => $user->classroom->name ?? null,
                    ],
                    'payment_details' => [
                        'month' => $payment->month,
                        'year' => $payment->year,
                        'base_amount' => (float) $payment->amount,
                        'fine_amount' => (float) ($payment->fine ?? 0),
                        'discount_amount' => (float) ($payment->discount ?? 0),
                        'total_paid' => (float) ($payment->amount + ($payment->fine ?? 0) - ($payment->discount ?? 0)),
                        'payment_method' => $payment->pay_type === 'POS' ? 'POS' : 'Dinheiro',
                        'payment_date' => $payment->payment_date ?? $payment->created_at,
                        'transaction_id' => $payment->transaction_id ?? null,
                    ],
                    'institution' => [
                        'name' => 'COLÉGIO POLITÉCNICO DE MOÇAMBIQUE',
                        'location' => 'Nampula - Moçambique',
                    ],
                ],
            ], 200);

        } catch (\Exception $e) {
            Log::error('API: Error getting receipt details', [
                'payment_id' => $paymentId,
                'student_id' => $request->get('student_id'),
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erro ao buscar detalhes do recibo',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Download receipt PDF
     * GET /api/v2/student/payments/{id}/receipt/download?student_id=123
     */
    public function downloadStudentReceipt(Request $request, $paymentId)
    {
        try {
            $studentId = $request->get('student_id');

            if (!$studentId) {
                return response()->json([
                    'success' => false,
                    'message' => 'student_id é obrigatório'
                ], 400);
            }

            $user = \App\Models\User::find($studentId);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'Estudante não encontrado'
                ], 404);
            }

            $payment = \App\Models\Fee_assign::where('id', $paymentId)
                ->where('student_id', $user->id)
                ->first();

            if (!$payment) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pagamento não encontrado',
                ], 404);
            }

            // Gerar ou obter número de recibo
            $receiptNumber = \App\Services\ReceiptNumberService::getOrGenerateReceiptNumber(
                'fee_assign',
                $payment->id,
                'mobile',
                $user->id
            );

            // Preparar dados
            $baseAmount = $payment->amount;
            $fineAmount = $payment->fine ?? 0;
            $discountAmount = $payment->discount ?? 0;
            $totalPaid = $baseAmount + $fineAmount - $discountAmount;

            $paymentData = (object)[
                'id' => $payment->id,
                'fee_month' => $payment->month,
                'fee_year' => $payment->year,
            ];

            $paymentMethod = ($payment->payment_mode === 'POS' || strtolower($payment->pay_type ?? '') === 'pos')
                ? 'POS'
                : 'Dinheiro';

            // Gerar PDF
            $pdf = PDF::loadView('pdf.receipt-official', [
                'payment' => $paymentData,
                'student' => $user,
                'base_amount' => $baseAmount,
                'fine_amount' => $fineAmount,
                'discount_amount' => $discountAmount,
                'amount_paid' => $totalPaid,
                'paid_at' => Carbon::parse($payment->payment_date ?? $payment->created_at),
                'payment_method' => $paymentMethod,
                'transaction_id' => $payment->transaction_id ?? null,
                'receipt_number' => $receiptNumber,
            ]);

            return $pdf->download("recibo_{$receiptNumber}.pdf");

        } catch (\Exception $e) {
            Log::error('API: Error downloading receipt', [
                'payment_id' => $paymentId,
                'student_id' => $request->get('student_id'),
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erro ao gerar recibo',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get receipt statistics
     * GET /api/v2/student/receipts/statistics?student_id=123
     */
    public function getStudentReceiptStatistics(Request $request)
    {
        try {
            $studentId = $request->get('student_id');

            if (!$studentId) {
                return response()->json([
                    'success' => false,
                    'message' => 'student_id é obrigatório'
                ], 400);
            }

            $user = \App\Models\User::find($studentId);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'Estudante não encontrado'
                ], 404);
            }

            // Buscar todos os recibos do estudante
            $receipts = \App\Models\ReceiptNumber::where('receipt_type', 'fee_assign')
                ->whereIn('record_id', function($query) use ($user) {
                    $query->select('id')
                        ->from('fee_assign')
                        ->where('student_id', $user->id);
                })
                ->orderBy('issued_at', 'desc')
                ->get();

            $stats = [
                'total_receipts' => $receipts->count(),
                'by_year' => $receipts->groupBy(function($r) {
                    return explode('/', $r->receipt_number)[1] ?? date('Y');
                })->map->count(),
                'latest_receipt' => $receipts->first() ? [
                    'number' => $receipts->first()->receipt_number,
                    'issued_at' => $receipts->first()->issued_at,
                ] : null,
            ];

            return response()->json([
                'success' => true,
                'data' => $stats,
            ], 200);

        } catch (\Exception $e) {
            Log::error('API: Error getting receipt statistics', [
                'student_id' => $request->get('student_id'),
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erro ao buscar estatísticas de recibos',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

            /**
         * Generate payment reference
         * POST /api/v2/fees/generate-reference
         *
         * Comportamento melhorado:
         * - Se existe referência pendente VÁLIDA → retorna ela (success: true, is_reused: true)
         * - Se existe referência EXPIRADA → gera nova e expira a antiga
         * - Se não existe → gera nova
         */
        public function generatePaymentReference(Request $request)
        {
            $user = Auth::guard('api')->user();

            if (!$user) {
                return response()->json([
                    'status' => '401',
                    'message' => 'Unauthorized! Invalid token.'
                ], 401);
            }

            $validator = Validator::make($request->all(), [
                'month' => 'required|string',
                'year' => 'required|integer|min:2020|max:' . (date('Y') + 1),
                'amount' => 'required|numeric|min:0.01',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'false',
                    'errors' => $validator->errors(),
                ], 422);
            }

            try {
                // Verificar se já existe referência pendente
                $existingReference = \App\Models\PaymentReference::where([
                    'student_id' => $user->id,
                    'fee_month' => $request->month,
                    'fee_year' => $request->year,
                    'status' => 'pending'
                ])->first();

                // Se existe e ainda está válida, retornar ela
                if ($existingReference) {
                    $expiresAt = Carbon::parse($existingReference->expires_at);
                    $isExpired = $expiresAt->isPast();

                    if (!$isExpired) {
                        // Referência ainda válida - retornar success com flag is_reused
                        Log::info('Reusing existing valid payment reference', [
                            'user_id' => $user->id,
                            'reference' => $existingReference->reference_number,
                            'expires_at' => $expiresAt->toDateTimeString()
                        ]);

                        return response()->json([
                            'status' => 'true',
                            'message' => 'Referência existente ainda válida',
                            'is_reused' => true,
                            'data' => [
                                'id' => $existingReference->id,
                                'entity_code' => $existingReference->entity_code,
                                'reference_number' => $existingReference->reference_number,
                                'reference_formatted' => $this->formatReference($existingReference->reference_number),
                                'amount' => (float) $existingReference->amount,
                                'base_amount' => (float) ($existingReference->amount - $existingReference->fine_amount),
                                'fine_amount' => (float) $existingReference->fine_amount,
                                'discount_amount' => 0.00,
                                'fee_month' => $existingReference->fee_month,
                                'fee_year' => $existingReference->fee_year,
                                'status' => 'pending',
                                'expires_at' => $expiresAt->format('Y-m-d H:i:s'),
                                'days_until_expiry' => now()->diffInDays($expiresAt, false),
                                'payment_instructions' => [
                                    'mpesa' => "Menu → Pagar Conta → Entidade: {$existingReference->entity_code} → Referência: {$existingReference->reference_number}",
                                    'emola' => "Pagar Serviços → Entidade: {$existingReference->entity_code} → Referência: {$existingReference->reference_number}",
                                    'atm' => "Pagar Serviços → Entidade: {$existingReference->entity_code} → Referência: {$existingReference->reference_number}",
                                ]
                            ]
                        ], 200);
                    } else {
                        // Referência expirada - marcar como expired e gerar nova
                        Log::info('Expiring old payment reference', [
                            'user_id' => $user->id,
                            'reference' => $existingReference->reference_number,
                            'expired_at' => $expiresAt->toDateTimeString()
                        ]);

                        $existingReference->update(['status' => 'expired']);
                    }
                }

                // Obter cálculo de fee para o mês
                $feeService = app(\App\Services\FeeCalculationService::class);
                $calculation = $feeService->calculateFeeForStudent($user, $request->month, $request->year);

                if (!$calculation) {
                    return response()->json([
                        'status' => 'false',
                        'message' => 'Não foi possível calcular o valor da taxa para este mês'
                    ], 400);
                }

                // Calcular valores
                $baseAmount = $calculation['base_amount'];
                $fineAmount = $calculation['fine_amount'] ?? 0;
                $discount = $request->input('discount', 0);
                $totalAmount = $baseAmount + $fineAmount - $discount;

                // Configurações
                $entity = config('payments.entity', '11111');

                // Calcular data de expiração usando configuração de prazos
                $expiresAt = \App\Models\FeeMonthDeadline::calculateExpiresAt($request->month, $request->year);

                // Se a data calculada já passou, verificar se permite pagamento atrasado
                if ($expiresAt->isPast() && !\App\Models\FeeMonthDeadline::allowsLatePayment($request->month, $request->year)) {
                    return response()->json([
                        'status' => 'false',
                        'message' => 'Prazo para gerar referência deste mês já expirou',
                    ], 400);
                }

                // Gerar referência única
                $referenceGenerator = app(\App\Services\ReferenceGenerator::class);
                $reference = $referenceGenerator->makeFromStudent(
                    $request->month,      // string - mês primeiro
                    (int) $request->year, // int - ano segundo
                    $user,                // objeto User - estudante terceiro
                    $totalAmount          // float - valor total quarto
                );

                // Criar registro de payment reference com tratamento de duplicação
                try {
                    $paymentReference = \App\Models\PaymentReference::create([
                        'student_id' => $user->id,
                        'entity_code' => $entity,
                        'reference_number' => $reference,
                        'amount' => $totalAmount,
                        'fine_amount' => $fineAmount,
                        'fee_month' => $request->month,
                        'fee_year' => $request->year,
                        'expires_at' => $expiresAt,
                        'status' => 'pending'
                    ]);

                    Log::info('Payment reference generated via API', [
                        'user_id' => $user->id,
                        'reference' => $reference,
                        'month' => $request->month,
                        'year' => $request->year,
                        'amount' => $totalAmount,
                        'replaced_expired' => isset($existingReference)
                    ]);
                } catch (\Illuminate\Database\QueryException $e) {
                    // Se erro de duplicação (1062), buscar referência existente
                    if ($e->getCode() == 23000 || strpos($e->getMessage(), '1062') !== false) {
                        Log::warning('Duplicate reference detected, fetching existing', [
                            'user_id' => $user->id,
                            'reference' => $reference,
                            'month' => $request->month,
                            'year' => $request->year
                        ]);

                        // Buscar referência existente pelo número
                        $paymentReference = \App\Models\PaymentReference::where('reference_number', $reference)
                            ->where('status', 'pending')
                            ->first();

                        // Se não encontrar, relançar exceção
                        if (!$paymentReference) {
                            throw $e;
                        }

                        // Usar expiresAt da referência existente
                        $expiresAt = $paymentReference->expires_at ?
                            \Carbon\Carbon::parse($paymentReference->expires_at) :
                            \App\Models\FeeMonthDeadline::calculateExpiresAt($request->month, $request->year);
                    } else {
                        // Outro tipo de erro, relançar
                        throw $e;
                    }
                }

                // Verificar se foi criado novo ou reutilizado
                $wasReused = $paymentReference->wasRecentlyCreated === false;

                return response()->json([
                    'status' => 'true',
                    'message' => $wasReused ? 'Referência existente retornada' : 'Referência gerada com sucesso',
                    'is_reused' => $wasReused,
                    'data' => [
                        'id' => $paymentReference->id,
                        'entity_code' => $paymentReference->entity_code,
                        'reference_number' => $paymentReference->reference_number,
                        'reference_formatted' => $this->formatReference($paymentReference->reference_number),
                        'amount' => (float) $paymentReference->amount,
                        'base_amount' => (float) ($paymentReference->amount - $paymentReference->fine_amount),
                        'fine_amount' => (float) $paymentReference->fine_amount,
                        'discount_amount' => 0.00,
                        'fee_month' => $paymentReference->fee_month,
                        'fee_year' => $paymentReference->fee_year,
                        'status' => 'pending',
                        'expires_at' => $expiresAt->format('Y-m-d H:i:s'),
                        'days_until_expiry' => now()->diffInDays($expiresAt, false),
                        'payment_instructions' => [
                            'mpesa' => "Menu → Pagar Conta → Entidade: {$paymentReference->entity_code} → Referência: {$paymentReference->reference_number}",
                            'emola' => "Pagar Serviços → Entidade: {$paymentReference->entity_code} → Referência: {$paymentReference->reference_number}",
                            'atm' => "Pagar Serviços → Entidade: {$paymentReference->entity_code} → Referência: {$paymentReference->reference_number}",
                        ]
                    ]
                ], 200);

            } catch (\Exception $e) {
                Log::error('Error generating payment reference API', [
                    'user_id' => $user->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);

                return response()->json([
                    'status' => 'false',
                    'message' => 'Erro ao gerar referência de pagamento',
                    'error' => config('app.debug') ? $e->getMessage() : 'Erro interno do servidor'
                ], 500);
            }
        }

    /**
     * Get payment references for student
     * GET /api/v2/fees/references
     */
    public function getPaymentReferences(Request $request)
    {
        $user = Auth::guard('api')->user();

        if (!$user) {
            return response()->json([
                'status' => '401',
                'message' => 'Unauthorized! Invalid token.'
            ], 401);
        }

        try {
            $year = $request->input('year', date('Y'));
            $status = $request->input('status'); // paid, pending, expired

            $query = \App\Models\PaymentReference::where('student_id', $user->id)
                ->where('fee_year', $year);

            if ($status) {
                $query->where('status', $status);
            }

            $references = $query->orderBy('created_at', 'desc')->get();

            $formattedReferences = $references->map(function($ref) {
                return [
                    'id' => $ref->id,
                    'entity_code' => $ref->entity_code,
                    'reference_number' => $ref->reference_number,
                    'reference_formatted' => $this->formatReference($ref->reference_number),
                    'amount' => (float) $ref->amount,
                    'fee_month' => $ref->fee_month,
                    'fee_year' => $ref->fee_year,
                    'status' => $ref->status,
                    'expires_at' => $ref->expires_at ? Carbon::parse($ref->expires_at)->format('Y-m-d H:i:s') : null,
                    'paid_at' => $ref->paid_at ? Carbon::parse($ref->paid_at)->format('Y-m-d H:i:s') : null,
                    'created_at' => $ref->created_at->format('Y-m-d H:i:s'),
                    'is_expired' => $ref->expires_at && Carbon::parse($ref->expires_at)->isPast() && $ref->status === 'pending',
                    'days_until_expiry' => $ref->expires_at && $ref->status === 'pending'
                        ? now()->diffInDays(Carbon::parse($ref->expires_at), false)
                        : null,
                ];
            });

            return response()->json([
                'status' => 'true',
                'data' => $formattedReferences
            ], 200);

        } catch (\Exception $e) {
            Log::error('Error in payment references API', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 'false',
                'message' => 'Error fetching payment references',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper: Format reference number
     */
    private function formatReference($reference)
    {
        $ref = preg_replace('/\D/', '', $reference);

        if (strlen($ref) === 11) {
            return substr($ref, 0, 3) . ' ' . substr($ref, 3, 3) . ' ' . substr($ref, 6, 3) . ' ' . substr($ref, 9, 2);
        }

        return $reference;
    }

}
