<?php

namespace App\Http\Livewire\Admin\Fee;

use App\Models\Admin;
use App\Models\FeeType;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;

class EditFeeType extends Component
{
    use FindGuard, LivewireAlert;

    public $name, $fees_code, $description, $user, $model;
    public $validatedData;
    public $updateMode = false;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function mount($id)
    {
        $this->model = FeeType::findOrFail($id); // Load FeeType by ID
        $this->name = $this->model->name;
        $this->fees_code = $this->model->fees_code;
        $this->description = $this->model->description;
    }

    protected array $rules = [
        'name' => ['required', 'string', 'max:255'],
        'fees_code' => ['required', 'string', 'max:255'],
        'description' => ['nullable', 'string', 'max:1000'],
    ];

    /**
     * Real-time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function submit()
    {
        $this->validatedData = $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        $this->model->fill($this->validatedData);
        if ($this->model->isClean()) {
            $this->alert('warning', 'Pelo menos um valor deve ser alterado.');
            return redirect()->back();
        }

        $this->model->save();
        Note::createSystemNotification(Admin::class, 'Detalhes', 'Atualizado com sucesso: ' . $this->name);
        $this->emit('noteAdded');
        $this->alert('success', 'Atualizado com sucesso: ' . $this->name);
        $this->reset();
        return redirect()->route('admin.list.feetypes'); // Update route to FeeType list
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function render()
    {
        return view('livewire.admin.fee.edit-fee-type');
    }
}
