<?php

//namespace App\Http\Livewire\Admin\Finance;
//
//use App\Models\PaymentReference;
//use App\Models\User;
//use App\Models\Fee_assign;
//use App\Services\PaymentReconciliationService;
//use App\Services\FeeCalculationService;
//use Livewire\Component;
//use Livewire\WithFileUploads;
//use Livewire\WithPagination;
//use Carbon\Carbon;
//use Illuminate\Support\Facades\DB;
//use Illuminate\Support\Facades\Log;
//use Illuminate\Support\Facades\Storage;
//use Maatwebsite\Excel\Facades\Excel;
//
//class PaymentReconciliationDashboard extends Component
//{
//    use WithFileUploads, WithPagination;
//
//    // Upload de ficheiros
//    public $uploadedFile;
//    public $uploadProgress = 0;
//
//    // Filtros
//    public $filterStatus = 'all';
//    public $filterDateFrom;
//    public $filterDateTo;
//    public $filterFileType = 'all';
//    public $searchTerm = '';
//
//    // Reconciliação manual
//    public $showManualModal = false;
//    public $manualReference;
//    public $manualAmount;
//    public $manualDate;
//    public $manualTransactionId;
//    public $manualTerminalId;
//    public $manualNote;
//
//    // Estatísticas em tempo real
//    public $stats = [];
//    public $recentActivity = [];
//
//    // Configurações
//    public $autoRefresh = false;
//    public $refreshInterval = 30; // segundos
//
//    protected $reconciliationService;
//    protected $feeCalculationService;
//
//    protected $rules = [
//        'uploadedFile' => 'required|file|mimes:txt,TXT|max:10240',
//        'manualReference' => 'required|string|size:11',
//        'manualAmount' => 'required|numeric|min:0.01',
//        'manualDate' => 'required|date',
//        'manualTransactionId' => 'nullable|string|max:50',
//        'manualTerminalId' => 'nullable|string|max:20',
//        'manualNote' => 'nullable|string|max:500',
//    ];
//
//    public function boot(
//        PaymentReconciliationService $reconciliationService,
//        FeeCalculationService $feeCalculationService
//    ) {
//        $this->reconciliationService = $reconciliationService;
//        $this->feeCalculationService = $feeCalculationService;
//    }
//
//    public function mount()
//    {
//        $this->filterDateFrom = now()->subDays(30)->format('Y-m-d');
//        $this->filterDateTo = now()->format('Y-m-d');
//        $this->loadStats();
//        $this->loadRecentActivity();
//    }
//
//    /* ==================== PROCESSAMENTO DE FICHEIROS ==================== */
//
//    public function processUploadedFile()
//    {
//        $this->validate(['uploadedFile' => 'required|file|mimes:txt,TXT|max:10240']);
//
//        try {
//            $this->uploadProgress = 10;
//
//            // Move o arquivo para o diretório de processamento
//            $fileName = 'manual_' . now()->format('Ymd_His') . '_' . $this->uploadedFile->getClientOriginalName();
//            $incomingPath = public_path('bci/bmeps/incoming');
//            
//            if (!is_dir($incomingPath)) {
//                mkdir($incomingPath, 0775, true);
//            }
//
//            $filePath = $incomingPath . '/' . $fileName;
//            $this->uploadedFile->storeAs('', $fileName, ['disk' => 'reconciliation_incoming']);
//            
//            $this->uploadProgress = 50;
//
//            // Processa o arquivo
//            $results = $this->reconciliationService->processAllPendingFiles();
//            
//            $this->uploadProgress = 100;
//
//            // Registra log detalhado
//            $this->logReconciliationActivity([
//                'type' => 'file_upload',
//                'file_name' => $fileName,
//                'results' => $results,
//                'user_id' => auth()->id(),
//                'ip_address' => request()->ip(),
//            ]);
//
//            $this->dispatchBrowserEvent('file-processed', [
//                'message' => 'Ficheiro processado com sucesso!',
//                'stats' => $results
//            ]);
//
//            $this->reset(['uploadedFile', 'uploadProgress']);
//            $this->loadStats();
//            $this->loadRecentActivity();
//
//            session()->flash('success',
//                "Ficheiro processado: {$results['reconciled']} pagamentos reconciliados, " .
//                "{$results['failed']} falharam, {$results['duplicates']} duplicados."
//            );
//
//        } catch (\Exception $e) {
//            Log::error('Erro no processamento manual de ficheiro', [
//                'error' => $e->getMessage(),
//                'user_id' => auth()->id(),
//                'file_name' => $this->uploadedFile ? $this->uploadedFile->getClientOriginalName() : 'unknown'
//            ]);
//
//            $this->uploadProgress = 0;
//            session()->flash('error', 'Erro ao processar ficheiro: ' . $e->getMessage());
//        }
//    }
//
//    /* ==================== RECONCILIAÇÃO MANUAL ==================== */
//
//    public function openManualReconciliation()
//    {
//        $this->reset(['manualReference', 'manualAmount', 'manualDate', 'manualTransactionId', 'manualTerminalId', 'manualNote']);
//        $this->manualDate = now()->format('Y-m-d');
//        $this->showManualModal = true;
//    }
//
//    public function processManualReconciliation()
//    {
//        $this->validate();
//
//        DB::beginTransaction();
//        try {
//            // Verifica se a referência existe
//            $paymentReference = PaymentReference::where('reference_number', $this->manualReference)
//                ->whereIn('status', ['pending'])
//                ->first();
//
//            if (!$paymentReference) {
//                throw new \Exception("Referência {$this->manualReference} não encontrada ou já processada.");
//            }
//
//            // Busca o estudante
//            $student = User::find($paymentReference->student_id);
//            if (!$student) {
//                throw new \Exception("Estudante não encontrado para a referência {$this->manualReference}.");
//            }
//
//            // Calcula os valores usando o serviço unificado
//            $paymentDate = Carbon::parse($this->manualDate . ' ' . now()->format('H:i:s'));
//            
//            $calculation = $this->feeCalculationService->calculateFeeForStudent(
//                $student,
//                $paymentReference->fee_month,
//                $paymentReference->fee_year,
//                $paymentDate,
//                $paymentReference
//            );
//
//            // Valida o montante com tolerância
//            $tolerance = 1.00;
//            if (abs($this->manualAmount - $calculation['total_amount']) > $tolerance) {
//                $this->dispatchBrowserEvent('show-amount-warning', [
//                    'paid' => $this->manualAmount,
//                    'expected' => $calculation['total_amount'],
//                    'difference' => abs($this->manualAmount - $calculation['total_amount'])
//                ]);
//            }
//
//            // Atualiza a PaymentReference
//            $paymentReference->update([
//                'status' => 'paid',
//                'paid_at' => $paymentDate,
//                'transaction_id' => $this->manualTransactionId ?: ('MANUAL-' . now()->format('YmdHis') . '-' . substr(md5($this->manualReference), 0, 6)),
//                'terminal_id' => $this->manualTerminalId ?: 'MANUAL',
//                'fine_amount' => $calculation['fine_amount'],
//            ]);
//
//            // Cria o pagamento usando o serviço unificado
//            $paymentData = [
//                'month' => $paymentReference->fee_month,
//                'year' => $paymentReference->fee_year,
//                'amount' => (float) $this->manualAmount,
//                'fine' => (float) $calculation['fine_amount'],
//                'discount' => 0,
//                'payment_mode' => 'Manual',
//                'paymentMode' => 'Manual',
//                'pay_type' => 'manual',
//                'transaction_id' => $paymentReference->transaction_id,
//                'reference_number' => $this->manualReference,
//                'payment_date' => $paymentDate,
//                'note' => 'Reconciliação manual: ' . ($this->manualNote ?: 'Processado via dashboard administrativo')
//            ];
//
//            $feeAssign = $this->feeCalculationService->createFeePayment(
//                $student,
//                $paymentData,
//                $calculation
//            );
//
//            // Registra na tabela de reconciliação
//            DB::table('payment_reconciliations')->insert([
//                'transaction_id' => $paymentReference->transaction_id,
//                'reference' => $this->manualReference,
//                'amount' => $this->manualAmount,
//                'payment_date' => $paymentDate,
//                'terminal_id' => $this->manualTerminalId ?: 'MANUAL',
//                'terminal_location' => 'Manual Dashboard',
//                'processed_at' => now(),
//                'created_at' => now(),
//                'updated_at' => now(),
//            ]);
//
//            // Log da atividade
//            $this->logReconciliationActivity([
//                'type' => 'manual_reconciliation',
//                'reference' => $this->manualReference,
//                'amount' => $this->manualAmount,
//                'student_id' => $student->id,
//                'transaction_id' => $paymentReference->transaction_id,
//                'user_id' => auth()->id(),
//                'note' => $this->manualNote,
//                'fee_assign_id' => $feeAssign->id,
//            ]);
//
//            DB::commit();
//
//            $this->showManualModal = false;
//            $this->loadStats();
//            $this->loadRecentActivity();
//
//            session()->flash('success',
//                "Reconciliação manual concluída: Referência {$this->manualReference}, " .
//                "Estudante: {$student->name}, Valor: MT " . number_format($this->manualAmount, 2)
//            );
//
//            Log::info('Manual reconciliation completed', [
//                'reference' => $this->manualReference,
//                'amount' => $this->manualAmount,
//                'student_id' => $student->id,
//                'user_id' => auth()->id(),
//                'fee_assign_id' => $feeAssign->id,
//            ]);
//
//        } catch (\Exception $e) {
//            DB::rollback();
//            
//            Log::error('Manual reconciliation failed', [
//                'reference' => $this->manualReference,
//                'error' => $e->getMessage(),
//                'user_id' => auth()->id(),
//            ]);
//
//            session()->flash('error', 'Erro na reconciliação manual: ' . $e->getMessage());
//        }
//    }
//
//    /* ==================== PROCESSAMENTO AUTOMÁTICO ==================== */
//
//    public function runAutomaticReconciliation()
//    {
//        try {
//            $results = $this->reconciliationService->processAllPendingFiles();
//            
//            $this->logReconciliationActivity([
//                'type' => 'automatic_processing',
//                'results' => $results,
//                'user_id' => auth()->id(),
//                'trigger' => 'manual_dashboard',
//            ]);
//
//            $this->loadStats();
//            $this->loadRecentActivity();
//
//            if ($results['total_files'] > 0) {
//                session()->flash('success',
//                    "Processamento automático concluído: {$results['total_files']} ficheiros, " .
//                    "{$results['reconciled']} reconciliados, {$results['failed']} falharam."
//                );
//            } else {
//                session()->flash('info', 'Nenhum ficheiro pendente encontrado para processamento.');
//            }
//
//        } catch (\Exception $e) {
//            Log::error('Automatic reconciliation failed', [
//                'error' => $e->getMessage(),
//                'user_id' => auth()->id(),
//            ]);
//
//            session()->flash('error', 'Erro no processamento automático: ' . $e->getMessage());
//        }
//    }
//
//    /* ==================== GESTÃO DE FICHEIROS ==================== */
//
//    public function getIncomingFilesProperty()
//    {
//        $incomingPath = public_path('bci/bmeps/incoming');
//        if (!is_dir($incomingPath)) return collect();
//
//        $files = collect(glob($incomingPath . '/*.{txt,TXT}', GLOB_BRACE))
//            ->map(function ($filePath) {
//                $fileName = basename($filePath);
//                $fileStats = stat($filePath);
//                
//                return [
//                    'name' => $fileName,
//                    'size' => $fileStats['size'],
//                    'size_formatted' => $this->formatFileSize($fileStats['size']),
//                    'created_at' => Carbon::createFromTimestamp($fileStats['ctime']),
//                    'modified_at' => Carbon::createFromTimestamp($fileStats['mtime']),
//                    'is_processing' => false, // TODO: implementar verificação se está sendo processado
//                ];
//            })
//            ->sortByDesc('modified_at');
//
//        return $files;
//    }
//
//    public function deleteIncomingFile($fileName)
//    {
//        try {
//            $filePath = public_path('bci/bmeps/incoming/' . $fileName);
//            if (file_exists($filePath)) {
//                unlink($filePath);
//                
//                Log::info('Incoming file deleted manually', [
//                    'file_name' => $fileName,
//                    'user_id' => auth()->id(),
//                ]);
//
//                session()->flash('success', "Ficheiro {$fileName} eliminado com sucesso.");
//            } else {
//                session()->flash('error', "Ficheiro {$fileName} não encontrado.");
//            }
//        } catch (\Exception $e) {
//            Log::error('Failed to delete incoming file', [
//                'file_name' => $fileName,
//                'error' => $e->getMessage(),
//                'user_id' => auth()->id(),
//            ]);
//
//            session()->flash('error', 'Erro ao eliminar ficheiro: ' . $e->getMessage());
//        }
//    }
//
//    /* ==================== RELATÓRIOS E LOGS ==================== */
//
//    public function getReconciliationLogsProperty()
//    {
//        $query = DB::table('reconciliation_logs')
//            ->orderBy('started_at', 'desc');
//
//        if ($this->filterStatus !== 'all') {
//            $query->where('status', $this->filterStatus);
//        }
//
//        if ($this->filterDateFrom) {
//            $query->whereDate('started_at', '>=', $this->filterDateFrom);
//        }
//
//        if ($this->filterDateTo) {
//            $query->whereDate('started_at', '<=', $this->filterDateTo);
//        }
//
//        if ($this->searchTerm) {
//            $query->where(function ($q) {
//                $q->where('file_name', 'like', '%' . $this->searchTerm . '%')
//                  ->orWhere('file_id', 'like', '%' . $this->searchTerm . '%');
//            });
//        }
//
//        return $query->paginate(20);
//    }
//
//    public function getUnmatchedPaymentsProperty()
//    {
//        return DB::table('unmatched_payments')
//            ->orderBy('created_at', 'desc')
//            ->limit(50)
//            ->get()
//            ->map(function ($payment) {
//                $payment->can_reconcile_manually = !empty($payment->reference);
//                $payment->formatted_amount = 'MT ' . number_format($payment->amount, 2);
//                $payment->age_days = Carbon::parse($payment->created_at)->diffInDays(now());
//                return $payment;
//            });
//    }
//
//    public function exportReconciliationReport()
//    {
//        try {
//            $data = $this->getReconciliationReportData();
//            
//            return Excel::download(
//                new ReconciliationReportExport($data),
//                'relatorio_reconciliacao_' . now()->format('Ymd_His') . '.xlsx'
//            );
//        } catch (\Exception $e) {
//            Log::error('Failed to export reconciliation report', [
//                'error' => $e->getMessage(),
//                'user_id' => auth()->id(),
//            ]);
//
//            session()->flash('error', 'Erro ao exportar relatório: ' . $e->getMessage());
//        }
//    }
//
//    /* ==================== MÉTODOS AUXILIARES ==================== */
//
//    private function loadStats()
//    {
//        $todayStart = now()->startOfDay();
//        $todayEnd = now()->endOfDay();
//        $weekStart = now()->startOfWeek();
//        $monthStart = now()->startOfMonth();
//
//        $this->stats = [
//            'today' => [
//                'processed_files' => DB::table('reconciliation_logs')->whereDate('started_at', today())->count(),
//                'reconciled_payments' => DB::table('payment_reconciliations')->whereDate('processed_at', today())->count(),
//                'total_amount' => DB::table('payment_reconciliations')->whereDate('processed_at', today())->sum('amount'),
//                'failed_transactions' => DB::table('reconciliation_logs')->whereDate('started_at', today())->sum('failed'),
//            ],
//            'week' => [
//                'processed_files' => DB::table('reconciliation_logs')->whereBetween('started_at', [$weekStart, now()])->count(),
//                'reconciled_payments' => DB::table('payment_reconciliations')->whereBetween('processed_at', [$weekStart, now()])->count(),
//                'total_amount' => DB::table('payment_reconciliations')->whereBetween('processed_at', [$weekStart, now()])->sum('amount'),
//                'success_rate' => $this->calculateSuccessRate($weekStart, now()),
//            ],
//            'month' => [
//                'processed_files' => DB::table('reconciliation_logs')->whereBetween('started_at', [$monthStart, now()])->count(),
//                'reconciled_payments' => DB::table('payment_reconciliations')->whereBetween('processed_at', [$monthStart, now()])->count(),
//                'total_amount' => DB::table('payment_reconciliations')->whereBetween('processed_at', [$monthStart, now()])->sum('amount'),
//                'average_processing_time' => $this->calculateAverageProcessingTime($monthStart, now()),
//            ],
//            'pending' => [
//                'incoming_files' => count($this->incoming_files),
//                'unmatched_payments' => DB::table('unmatched_payments')->where('status', 'unmatched')->count(),
//                'pending_references' => PaymentReference::where('status', 'pending')->count(),
//                'expired_references' => PaymentReference::where('status', 'pending')->where('expires_at', '<', now())->count(),
//            ],
//        ];
//    }
//
//    private function loadRecentActivity()
//    {
//        $this->recentActivity = DB::table('reconciliation_activity_log')
//            ->orderBy('created_at', 'desc')
//            ->limit(10)
//            ->get()
//            ->map(function ($activity) {
//                $activity->formatted_created_at = Carbon::parse($activity->created_at)->diffForHumans();
//                $activity->icon = $this->getActivityIcon($activity->type);
//                $activity->color = $this->getActivityColor($activity->type);
//                return $activity;
//            });
//    }
//
//    private function calculateSuccessRate($from, $to)
//    {
//        $total = DB::table('reconciliation_logs')->whereBetween('started_at', [$from, $to])->sum('total_records');
//        $successful = DB::table('reconciliation_logs')->whereBetween('started_at', [$from, $to])->sum('reconciled');
//        
//        return $total > 0 ? round(($successful / $total) * 100, 1) : 0;
//    }
//
//    private function calculateAverageProcessingTime($from, $to)
//    {
//        $logs = DB::table('reconciliation_logs')
//            ->whereBetween('started_at', [$from, $to])
//            ->whereNotNull('completed_at')
//            ->select(DB::raw('AVG(TIMESTAMPDIFF(SECOND, started_at, completed_at)) as avg_seconds'))
//            ->first();
//
//        return $logs && $logs->avg_seconds ? round($logs->avg_seconds, 1) : 0;
//    }
//
//    private function logReconciliationActivity(array $data)
//    {
//        try {
//            DB::table('reconciliation_activity_log')->insert([
//                'type' => $data['type'],
//                'description' => $this->generateActivityDescription($data),
//                'data' => json_encode($data),
//                'user_id' => $data['user_id'] ?? null,
//                'ip_address' => $data['ip_address'] ?? request()->ip(),
//                'created_at' => now(),
//            ]);
//        } catch (\Exception $e) {
//            Log::warning('Failed to log reconciliation activity', [
//                'error' => $e->getMessage(),
//                'data' => $data,
//            ]);
//        }
//    }
//
//    private function generateActivityDescription(array $data)
//    {
//        switch ($data['type']) {
//            case 'file_upload':
//                return "Ficheiro '{$data['file_name']}' carregado e processado - {$data['results']['reconciled']} reconciliados";
//            case 'manual_reconciliation':
//                return "Reconciliação manual da referência {$data['reference']} - MT " . number_format($data['amount'], 2);
//            case 'automatic_processing':
//                return "Processamento automático executado - {$data['results']['total_files']} ficheiros processados";
//            default:
//                return "Atividade de reconciliação: {$data['type']}";
//        }
//    }
//
//    private function getActivityIcon($type)
//    {
//        $icons = [
//            'file_upload' => 'fas fa-upload',
//            'manual_reconciliation' => 'fas fa-edit',
//            'automatic_processing' => 'fas fa-cogs',
//            'error' => 'fas fa-exclamation-triangle',
//            'success' => 'fas fa-check-circle',
//        ];
//
//        return $icons[$type] ?? 'fas fa-info-circle';
//    }
//
//    private function getActivityColor($type)
//    {
//        $colors = [
//            'file_upload' => 'text-info',
//            'manual_reconciliation' => 'text-warning',
//            'automatic_processing' => 'text-primary',
//            'error' => 'text-danger',
//            'success' => 'text-success',
//        ];
//
//        return $colors[$type] ?? 'text-secondary';
//    }
//
//    private function formatFileSize($bytes)
//    {
//        $units = ['B', 'KB', 'MB', 'GB'];
//        $bytes = max($bytes, 0);
//        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
//        $pow = min($pow, count($units) - 1);
//        
//        $bytes /= pow(1024, $pow);
//        
//        return round($bytes, 2) . ' ' . $units[$pow];
//    }
//
//    private function getReconciliationReportData()
//    {
//        return [
//            'period' => [
//                'from' => $this->filterDateFrom,
//                'to' => $this->filterDateTo,
//                'generated_at' => now(),
//                'generated_by' => auth()->user()->name,
//            ],
//            'summary' => $this->stats,
//            'logs' => $this->reconciliation_logs,
//            'unmatched' => $this->unmatched_payments,
//            'activity' => $this->recentActivity,
//        ];
//    }
//
//    public function render()
//    {
//        return view('livewire.admin.finance.payment-reconciliation-dashboard');
//    }
//}


namespace App\Http\Livewire\Admin\Fee;

use App\Models\PaymentReference;
use App\Models\User;
use App\Models\Fee_assign;
use App\Services\PaymentReconciliationService;
use App\Services\FeeCalculationService;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Collection;

class PaymentReconciliationDashboard extends Component
{
    use WithFileUploads, WithPagination;

    // Upload de ficheiros
    public $uploadedFile;
    public $uploadProgress = 0;

    // Filtros
    public $filterStatus = 'all';
    public $filterDateFrom;
    public $filterDateTo;
    public $searchTerm = '';

    // Reconciliação manual
    public $showManualModal = false;
    public $manualReference;
    public $manualAmount;
    public $manualDate;
    public $manualTransactionId;
    public $manualTerminalId;
    public $manualNote;

    // Estatísticas em tempo real
    public $stats = [];

    // Caches para evitar múltiplas consultas
    private $cachedIncomingFiles = null;
    private $cachedUnmatchedPayments = null;
    private $cachedRecentActivity = null;

    protected $reconciliationService;
    protected $feeCalculationService;

    protected $rules = [
        'uploadedFile' => 'required|file|mimes:txt,TXT|max:10240',
        'manualReference' => 'required|string|size:11',
        'manualAmount' => 'required|numeric|min:0.01',
        'manualDate' => 'required|date',
        'manualTransactionId' => 'nullable|string|max:50',
        'manualTerminalId' => 'nullable|string|max:20',
        'manualNote' => 'nullable|string|max:500',
    ];

    public function boot(
        PaymentReconciliationService $reconciliationService,
        FeeCalculationService $feeCalculationService
    ) {
        $this->reconciliationService = $reconciliationService;
        $this->feeCalculationService = $feeCalculationService;
    }

    public function mount()
    {
        $this->filterDateFrom = now()->subDays(30)->format('Y-m-d');
        $this->filterDateTo = now()->format('Y-m-d');
        $this->loadStats();
    }

    /* ==================== PROPERTIES SEGURAS ==================== */

    public function getIncomingFilesProperty(): Collection
    {
        if ($this->cachedIncomingFiles !== null) {
            return $this->cachedIncomingFiles;
        }

        try {
            $incomingPath = public_path('bci/bmeps/incoming');
            
            if (!is_dir($incomingPath)) {
                $this->cachedIncomingFiles = collect([]);
                return $this->cachedIncomingFiles;
            }

            $files = glob($incomingPath . '/*.{txt,TXT}', GLOB_BRACE);
            
            if (!is_array($files) || empty($files)) {
                $this->cachedIncomingFiles = collect([]);
                return $this->cachedIncomingFiles;
            }

            $this->cachedIncomingFiles = collect($files)
                ->map(function ($filePath) {
                    try {
                        $fileName = basename($filePath);
                        $fileStats = stat($filePath);
                        
                        if (!$fileStats) {
                            return null;
                        }
                        
                        return [
                            'name' => $fileName,
                            'size' => $fileStats['size'] ?? 0,
                            'size_formatted' => $this->formatFileSize($fileStats['size'] ?? 0),
                            'created_at' => Carbon::createFromTimestamp($fileStats['ctime'] ?? time()),
                            'modified_at' => Carbon::createFromTimestamp($fileStats['mtime'] ?? time()),
                        ];
                    } catch (\Exception $e) {
                        Log::warning('Error reading file stats', ['file' => $filePath ?? 'unknown', 'error' => $e->getMessage()]);
                        return null;
                    }
                })
                ->filter() // Remove nulls
                ->sortByDesc('modified_at')
                ->values(); // Reset keys

        } catch (\Exception $e) {
            Log::error('Error getting incoming files', ['error' => $e->getMessage()]);
            $this->cachedIncomingFiles = collect([]);
        }

        return $this->cachedIncomingFiles;
    }

    public function getUnmatchedPaymentsProperty(): Collection
    {
        if ($this->cachedUnmatchedPayments !== null) {
            return $this->cachedUnmatchedPayments;
        }

        try {
            $payments = DB::table('unmatched_payments')
                ->orderBy('created_at', 'desc')
                ->limit(50)
                ->get();

            if (!$payments) {
                $this->cachedUnmatchedPayments = collect([]);
                return $this->cachedUnmatchedPayments;
            }

            $this->cachedUnmatchedPayments = collect($payments)->map(function ($payment) {
                $payment->can_reconcile_manually = !empty($payment->reference);
                $payment->formatted_amount = 'MT ' . number_format($payment->amount ?? 0, 2);
                $payment->age_days = Carbon::parse($payment->created_at ?? now())->diffInDays(now());
                return $payment;
            });

        } catch (\Exception $e) {
            Log::error('Error getting unmatched payments', ['error' => $e->getMessage()]);
            $this->cachedUnmatchedPayments = collect([]);
        }

        return $this->cachedUnmatchedPayments;
    }

    public function getRecentActivityProperty(): Collection
    {
        if ($this->cachedRecentActivity !== null) {
            return $this->cachedRecentActivity;
        }

        try {
            $activities = collect([]);

            // Logs de reconciliation_logs
            $recentLogs = DB::table('reconciliation_logs')
                ->select([
                    'file_name as description',
                    'status',
                    'started_at as created_at',
                    DB::raw("'file_processing' as type"),
                    'reconciled',
                    'failed'
                ])
                ->orderBy('started_at', 'desc')
                ->limit(5)
                ->get();

            if ($recentLogs) {
                foreach ($recentLogs as $log) {
                    $activities->push([
                        'type' => 'file_processing',
                        'description' => "Ficheiro {$log->description} processado - {$log->reconciled} reconciliados",
                        'created_at' => $log->created_at,
                        'formatted_created_at' => Carbon::parse($log->created_at)->diffForHumans(),
                        'icon' => $this->getActivityIcon('file_processing'),
                        'color' => $this->getStatusColor($log->status),
                    ]);
                }
            }

            // Reconciliações manuais
            $recentManual = DB::table('payment_reconciliations')
                ->where('terminal_id', 'MANUAL')
                ->select([
                    'reference',
                    'amount',
                    'processed_at as created_at'
                ])
                ->orderBy('processed_at', 'desc')
                ->limit(5)
                ->get();

            if ($recentManual) {
                foreach ($recentManual as $manual) {
                    $activities->push([
                        'type' => 'manual_reconciliation',
                        'description' => "Reconciliação manual da referência {$manual->reference} - MT " . number_format($manual->amount ?? 0, 2),
                        'created_at' => $manual->created_at,
                        'formatted_created_at' => Carbon::parse($manual->created_at)->diffForHumans(),
                        'icon' => $this->getActivityIcon('manual_reconciliation'),
                        'color' => 'text-warning',
                    ]);
                }
            }

            $this->cachedRecentActivity = $activities->sortByDesc('created_at')->take(10)->values();

        } catch (\Exception $e) {
            Log::error('Error getting recent activity', ['error' => $e->getMessage()]);
            $this->cachedRecentActivity = collect([]);
        }

        return $this->cachedRecentActivity;
    }

    public function getReconciliationLogsProperty()
    {
        try {
            $query = DB::table('reconciliation_logs')
                ->orderBy('started_at', 'desc');

            if ($this->filterStatus !== 'all') {
                $query->where('status', $this->filterStatus);
            }

            if ($this->filterDateFrom) {
                $query->whereDate('started_at', '>=', $this->filterDateFrom);
            }

            if ($this->filterDateTo) {
                $query->whereDate('started_at', '<=', $this->filterDateTo);
            }

            if ($this->searchTerm) {
                $query->where(function ($q) {
                    $q->where('file_name', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('file_id', 'like', '%' . $this->searchTerm . '%');
                });
            }

            return $query->paginate(20);
        } catch (\Exception $e) {
            Log::error('Error getting reconciliation logs', ['error' => $e->getMessage()]);
            return collect([])->paginate(20);
        }
    }

    /* ==================== PROCESSAMENTO DE FICHEIROS ==================== */

    public function processUploadedFile()
    {
        $this->validate(['uploadedFile' => 'required|file|mimes:txt,TXT|max:10240']);

        try {
            $this->uploadProgress = 10;

            // Move o arquivo para o diretório de processamento
            $fileName = 'manual_' . now()->format('Ymd_His') . '_' . $this->uploadedFile->getClientOriginalName();
            $incomingPath = public_path('bci/bmeps/incoming');
            
            if (!is_dir($incomingPath)) {
                mkdir($incomingPath, 0775, true);
            }

            $filePath = $incomingPath . '/' . $fileName;
            copy($this->uploadedFile->getRealPath(), $filePath);
            
            $this->uploadProgress = 50;

            // Processa o arquivo
            $results = $this->reconciliationService->processAllPendingFiles();
            
            $this->uploadProgress = 100;

            // Log
            $this->logFileProcessing($fileName, $results);

            $this->dispatchBrowserEvent('file-processed', [
                'message' => 'Ficheiro processado com sucesso!',
                'stats' => $results
            ]);

            $this->reset(['uploadedFile', 'uploadProgress']);
            $this->clearCaches();
            $this->loadStats();

            session()->flash('success',
                "Ficheiro processado: " . ($results['reconciled'] ?? 0) . " pagamentos reconciliados, " .
                ($results['failed'] ?? 0) . " falharam, " . ($results['duplicates'] ?? 0) . " duplicados."
            );

        } catch (\Exception $e) {
            Log::error('Erro no processamento manual de ficheiro', [
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
                'file_name' => $this->uploadedFile ? $this->uploadedFile->getClientOriginalName() : 'unknown'
            ]);

            $this->uploadProgress = 0;
            session()->flash('error', 'Erro ao processar ficheiro: ' . $e->getMessage());
        }
    }

    /* ==================== RECONCILIAÇÃO MANUAL ==================== */

    public function openManualReconciliation()
    {
        $this->reset(['manualReference', 'manualAmount', 'manualDate', 'manualTransactionId', 'manualTerminalId', 'manualNote']);
        $this->manualDate = now()->format('Y-m-d');
        $this->showManualModal = true;
    }

    public function processManualReconciliation()
    {
        $this->validate();

        DB::beginTransaction();
        try {
            // Verifica se a referência existe
            $paymentReference = PaymentReference::where('reference_number', $this->manualReference)
                ->whereIn('status', ['pending'])
                ->first();

            if (!$paymentReference) {
                throw new \Exception("Referência {$this->manualReference} não encontrada ou já processada.");
            }

            // Busca o estudante
            $student = User::find($paymentReference->student_id);
            if (!$student) {
                throw new \Exception("Estudante não encontrado para a referência {$this->manualReference}.");
            }

            $paymentDate = Carbon::parse($this->manualDate . ' ' . now()->format('H:i:s'));
            
            // Calcula os valores usando o serviço unificado
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $student,
                $paymentReference->fee_month,
                $paymentReference->fee_year,
                $paymentDate,
                $paymentReference
            );

            // Atualiza a PaymentReference
            $paymentReference->update([
                'status' => 'paid',
                'paid_at' => $paymentDate,
                'transaction_id' => $this->manualTransactionId ?: ('MANUAL-' . now()->format('YmdHis') . '-' . substr(md5($this->manualReference), 0, 6)),
                'terminal_id' => $this->manualTerminalId ?: 'MANUAL',
                'fine_amount' => $calculation['fine_amount'],
            ]);

            // Cria o pagamento usando o serviço unificado
            $paymentData = [
                'month' => $paymentReference->fee_month,
                'year' => $paymentReference->fee_year,
                'amount' => (float) $this->manualAmount,
                'fine' => (float) $calculation['fine_amount'],
                'discount' => 0,
                'payment_mode' => 'Manual',
                'paymentMode' => 'Manual',
                'pay_type' => 'manual',
                'transaction_id' => $paymentReference->transaction_id,
                'reference_number' => $this->manualReference,
                'payment_date' => $paymentDate,
                'note' => 'Reconciliação manual: ' . ($this->manualNote ?: 'Processado via dashboard administrativo')
            ];

            $feeAssign = $this->feeCalculationService->createFeePayment(
                $student,
                $paymentData,
                $calculation
            );

            // Registra na tabela de reconciliação
            DB::table('payment_reconciliations')->insert([
                'transaction_id' => $paymentReference->transaction_id,
                'reference' => $this->manualReference,
                'amount' => $this->manualAmount,
                'payment_date' => $paymentDate,
                'terminal_id' => $this->manualTerminalId ?: 'MANUAL',
                'terminal_location' => 'Manual Dashboard',
                'processed_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Log
            Log::info('Manual reconciliation completed', [
                'type' => 'manual_reconciliation',
                'reference' => $this->manualReference,
                'amount' => $this->manualAmount,
                'student_id' => $student->id,
                'transaction_id' => $paymentReference->transaction_id,
                'user_id' => auth()->id(),
                'note' => $this->manualNote,
                'fee_assign_id' => $feeAssign->id,
            ]);

            DB::commit();

            $this->showManualModal = false;
            $this->clearCaches();
            $this->loadStats();

            session()->flash('success',
                "Reconciliação manual concluída: Referência {$this->manualReference}, " .
                "Estudante: {$student->name}, Valor: MT " . number_format($this->manualAmount, 2)
            );

        } catch (\Exception $e) {
            DB::rollback();
            
            Log::error('Manual reconciliation failed', [
                'reference' => $this->manualReference,
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
            ]);

            session()->flash('error', 'Erro na reconciliação manual: ' . $e->getMessage());
        }
    }

    /* ==================== PROCESSAMENTO AUTOMÁTICO ==================== */

    public function runAutomaticReconciliation()
    {
        try {
            $results = $this->reconciliationService->processAllPendingFiles();
            
            // Log
            Log::info('Automatic reconciliation completed', [
                'type' => 'automatic_processing',
                'results' => $results,
                'user_id' => auth()->id(),
                'trigger' => 'manual_dashboard',
            ]);

            $this->clearCaches();
            $this->loadStats();

            if (($results['total_files'] ?? 0) > 0) {
                session()->flash('success',
                    "Processamento automático concluído: " . ($results['total_files'] ?? 0) . " ficheiros, " .
                    ($results['reconciled'] ?? 0) . " reconciliados, " . ($results['failed'] ?? 0) . " falharam."
                );
            } else {
                session()->flash('info', 'Nenhum ficheiro pendente encontrado para processamento.');
            }

        } catch (\Exception $e) {
            Log::error('Automatic reconciliation failed', [
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
            ]);

            session()->flash('error', 'Erro no processamento automático: ' . $e->getMessage());
        }
    }

    /* ==================== GESTÃO DE FICHEIROS ==================== */

    public function deleteIncomingFile($fileName)
    {
        try {
            $filePath = public_path('bci/bmeps/incoming/' . $fileName);
            if (file_exists($filePath)) {
                unlink($filePath);
                
                Log::info('Incoming file deleted manually', [
                    'file_name' => $fileName,
                    'user_id' => auth()->id(),
                ]);

                $this->clearCaches();
                session()->flash('success', "Ficheiro {$fileName} eliminado com sucesso.");
            } else {
                session()->flash('error', "Ficheiro {$fileName} não encontrado.");
            }
        } catch (\Exception $e) {
            Log::error('Failed to delete incoming file', [
                'file_name' => $fileName,
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
            ]);

            session()->flash('error', 'Erro ao eliminar ficheiro: ' . $e->getMessage());
        }
    }

    /* ==================== MÉTODOS AUXILIARES ==================== */

    private function clearCaches()
    {
        $this->cachedIncomingFiles = null;
        $this->cachedUnmatchedPayments = null;
        $this->cachedRecentActivity = null;
    }

    private function loadStats()
    {
        try {
            $todayStart = now()->startOfDay();
            $weekStart = now()->startOfWeek();
            $monthStart = now()->startOfMonth();

            $this->stats = [
                'today' => [
                    'processed_files' => DB::table('reconciliation_logs')->whereDate('started_at', today())->count() ?? 0,
                    'reconciled_payments' => DB::table('payment_reconciliations')->whereDate('processed_at', today())->count() ?? 0,
                    'total_amount' => $this->calculateTotalWithFines(today(), today()),
                    'failed_transactions' => DB::table('reconciliation_logs')->whereDate('started_at', today())->sum('failed') ?? 0,
                ],
                'week' => [
                    'processed_files' => DB::table('reconciliation_logs')->whereBetween('started_at', [$weekStart, now()])->count() ?? 0,
                    'reconciled_payments' => DB::table('payment_reconciliations')->whereBetween('processed_at', [$weekStart, now()])->count() ?? 0,
                    'total_amount' => $this->calculateTotalWithFines($weekStart, now()),
                    'success_rate' => $this->calculateSuccessRate($weekStart, now()),
                ],
                'month' => [
                    'processed_files' => DB::table('reconciliation_logs')->whereBetween('started_at', [$monthStart, now()])->count() ?? 0,
                    'reconciled_payments' => DB::table('payment_reconciliations')->whereBetween('processed_at', [$monthStart, now()])->count() ?? 0,
                    'total_amount' => $this->calculateTotalWithFines($monthStart, now()),
                    'average_processing_time' => $this->calculateAverageProcessingTime($monthStart, now()),
                ],
                'pending' => [
                    'incoming_files' => $this->getIncomingFilesProperty()->count(),
                    'unmatched_payments' => DB::table('unmatched_payments')->where('status', 'unmatched')->count() ?? 0,
                    'pending_references' => PaymentReference::where('status', 'pending')->count() ?? 0,
                    'expired_references' => PaymentReference::where('status', 'pending')->where('expires_at', '<', now())->count() ?? 0,
                ],
            ];
        } catch (\Exception $e) {
            Log::error('Error loading stats', ['error' => $e->getMessage()]);
            $this->stats = $this->getDefaultStats();
        }
    }

    private function getDefaultStats(): array
    {
        return [
            'today' => [
                'processed_files' => 0,
                'reconciled_payments' => 0,
                'total_amount' => 0,
                'failed_transactions' => 0,
            ],
            'week' => [
                'processed_files' => 0,
                'reconciled_payments' => 0,
                'total_amount' => 0,
                'success_rate' => 0,
            ],
            'month' => [
                'processed_files' => 0,
                'reconciled_payments' => 0,
                'total_amount' => 0,
                'average_processing_time' => 0,
            ],
            'pending' => [
                'incoming_files' => 0,
                'unmatched_payments' => 0,
                'pending_references' => 0,
                'expired_references' => 0,
            ],
        ];
    }

    private function calculateSuccessRate($from, $to)
    {
        try {
            $total = DB::table('reconciliation_logs')->whereBetween('started_at', [$from, $to])->sum('total_records') ?? 0;
            $successful = DB::table('reconciliation_logs')->whereBetween('started_at', [$from, $to])->sum('reconciled') ?? 0;
            
            return $total > 0 ? round(($successful / $total) * 100, 1) : 0;
        } catch (\Exception $e) {
            Log::error('Error calculating success rate', ['error' => $e->getMessage()]);
            return 0;
        }
    }

    private function calculateAverageProcessingTime($from, $to)
    {
        try {
            $logs = DB::table('reconciliation_logs')
                ->whereBetween('started_at', [$from, $to])
                ->whereNotNull('completed_at')
                ->select(DB::raw('AVG(TIMESTAMPDIFF(SECOND, started_at, completed_at)) as avg_seconds'))
                ->first();

            return $logs && $logs->avg_seconds ? round($logs->avg_seconds, 1) : 0;
        } catch (\Exception $e) {
            Log::error('Error calculating average processing time', ['error' => $e->getMessage()]);
            return 0;
        }
    }

    /**
     * Calcula o total de pagamentos incluindo multas e descontos
     * Fórmula: VALOR BASE + MULTA - DESCONTO = TOTAL
     */
    private function calculateTotalWithFines($from, $to)
    {
        try {
            // Busca os transaction_ids dos pagamentos reconciliados no período
            $transactionIds = DB::table('payment_reconciliations')
                ->whereBetween('processed_at', [$from, $to])
                ->pluck('transaction_id');

            if ($transactionIds->isEmpty()) {
                return 0;
            }

            // Busca os fee_assign correspondentes e calcula o total correto
            $payments = Fee_assign::whereIn('transaction_id', $transactionIds)->get();

            return $payments->sum(function($payment) {
                return ($payment->amount ?? 0) + ($payment->fine ?? 0) - ($payment->discount ?? 0);
            });
        } catch (\Exception $e) {
            Log::error('Error calculating total with fines', ['error' => $e->getMessage()]);
            return 0;
        }
    }

    private function logFileProcessing($fileName, $results)
    {
        try {
            Log::info('File processing completed via dashboard', [
                'type' => 'file_upload',
                'file_name' => $fileName,
                'results' => $results,
                'user_id' => auth()->id(),
                'ip_address' => request()->ip(),
            ]);
        } catch (\Exception $e) {
            Log::error('Error logging file processing', ['error' => $e->getMessage()]);
        }
    }

    private function getActivityIcon($type)
    {
        $icons = [
            'file_processing' => 'fas fa-upload',
            'manual_reconciliation' => 'fas fa-edit',
            'automatic_processing' => 'fas fa-cogs',
            'error' => 'fas fa-exclamation-triangle',
            'success' => 'fas fa-check-circle',
        ];

        return $icons[$type] ?? 'fas fa-info-circle';
    }

    private function getStatusColor($status)
    {
        $colors = [
            'completed' => 'text-success',
            'failed' => 'text-danger',
            'processing' => 'text-primary',
        ];

        return $colors[$status] ?? 'text-secondary';
    }

    private function formatFileSize($bytes)
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        
        $bytes /= pow(1024, $pow);
        
        return round($bytes, 2) . ' ' . $units[$pow];
    }

    public function render()
    {
        // Garantir que todas as propriedades sejam collections válidas
        $data = [
            'incoming_files' => $this->getIncomingFilesProperty(),
            'unmatched_payments' => $this->getUnmatchedPaymentsProperty(),
            'recent_activity' => $this->getRecentActivityProperty(),
            'reconciliation_logs' => $this->getReconciliationLogsProperty(),
            'stats' => $this->stats,
        ];

        return view('livewire.admin.fee.payment-reconciliation-dashboard', $data);
    }
}
