<?php

namespace App\Http\Livewire\Admin\Reports;
use App\Models\Admin;
use App\Models\Classroom;
use App\Models\Exam;
use App\Models\User;
use App\Models\Subject;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Note\Note;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\StudentMarks;
use App\Models\TeacherSubject;

class GenerateSubjectCombine extends Component
{
    use FindGuard, LivewireAlert;

    public $trimester_id;
    public $class_id;
    public $user_id, $class, $year;
    public $subject_id;
    public $report = [];
    public $reports;
    public bool $readyToLoad = false;
    public $isTeacher = false;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function mount()
    {
        // Set default year to current year
        $this->year = date("Y");
    }

    public function loadReport()
    {
        $this->readyToLoad = true;
    }

    protected array $rules = [
        'user_id' => ['string', 'nullable'],
        'class_id' => ['string', 'required'],
        'class' => ['string', 'required'],
        'subject_id' => ['string', 'required'],
        'year' => ['required']
    ];

    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    /**
     * Reset fields when year is changed
     */
    public function updatedYear()
    {
        // Reset class, classroom and subject when year changes
        $this->class = null;
        $this->class_id = null;
        $this->subject_id = null;
    }

    /**
     * Reset fields when class is changed
     */
    public function updatedClass()
    {
        // Reset classroom and subject when class level changes
        $this->class_id = null;
        $this->subject_id = null;
    }

    /**
     * generate report
     * @throws Exception
     */
    public function generateReport()
    {
        $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        Note::createSystemNotification(Admin::class, 'Relatório final de cada disciplina', 'Relatório gerado com sucesso ');
        $this->emit('noteAdded');
        $this->alert('success', 'Relatório gerado com sucesso ');
        $classroom = Classroom::query()->where('id', $this->class_id)->first();
        $class_name = $classroom ? $classroom->name : 'Turma'; // Nome da turma (ex: "Turma B")
        $classe = $this->class ?? 'Classe'; // Nome da classe (ex: "6ª Classe")
        $disciplina = Subject::query()->where('id', $this->subject_id)->first()->name ?? 'Disciplina'; // Nome da disciplina
        $ano = $this->year ?? date('Y'); // Ano (ex: "2024")

        // Criar condição para exportar os dados
        $condition = [
            'class_id' => $this->class_id,
            'subject_id' => $this->subject_id,
            'year' => $this->year,
        ];

        // Nome formatado do arquivo
        $fileName = "{$classe} - {$class_name} - {$disciplina} ({$ano}).xlsx";

        return Excel::download(new StudentMarks($condition), $fileName);
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function render()
    {
        $this->user = $this->findGuardType()->user();
        $ccclass = ($this->class) ? $this->class : '1';
        $this->isTeacher = ($this->user->role == 2);

        // Get available class levels (numeric values) based on teacher's assignments
        $availableClassLevels = [];

        if ($this->isTeacher) {
            // Get all subjects assigned to this teacher
            $teacherSubjects = TeacherSubject::where('teacher_id', $this->user->id)
                ->whereNotNull('class')
                ->get();

            // Filter by year if needed
            if (!empty($this->year)) {
                // Also look at exams to see what classes this teacher has exams for in this year
                $examClasses = Exam::join('teacher_subjects', 'exams.subject_id', '=', 'teacher_subjects.subject_id')
                    ->join('classrooms', 'exams.classroom_id', '=', 'classrooms.id')
                    ->where('teacher_subjects.teacher_id', $this->user->id)
                    ->where('exams.year', $this->year)
                    ->select('classrooms.class')
                    ->distinct()
                    ->pluck('class')
                    ->toArray();

                // Combine with directly assigned classes
                $directClasses = $teacherSubjects->pluck('class')->toArray();
                $availableClassLevels = array_unique(array_merge($examClasses, $directClasses));
            } else {
                // If no year selected, just show all classes assigned to this teacher
                $availableClassLevels = $teacherSubjects->pluck('class')->unique()->toArray();
            }
        }

        // Get classrooms for the selected class level
        if ($this->isTeacher) {
            $classrooms = Classroom::where('class', $ccclass)
                ->where(function($query) use ($ccclass) {
                    // Classrooms directly assigned to this teacher
                    $query->whereIn('id', function($subquery) {
                        $subquery->select('classroom_id')
                                ->from('teacher_subjects')
                                ->where('teacher_id', $this->user->id)
                                ->whereNotNull('classroom_id');
                    });

                    // OR Classrooms where teacher has exams for this year (if year is selected)
                    if (!empty($this->year)) {
                        $query->orWhereIn('id', function($subquery) {
                            $subquery->select('exams.classroom_id')
                                    ->from('exams')
                                    ->join('teacher_subjects', 'exams.subject_id', '=', 'teacher_subjects.subject_id')
                                    ->where('teacher_subjects.teacher_id', '=', $this->user->id)
                                    ->where('exams.year', '=', $this->year)
                                    ->distinct();
                        });
                    }
                })
                ->orderBy('name')
                ->get();
        } else {
            // For admins, show all classrooms for the selected class level
            $classrooms = Classroom::where('class', $ccclass)
                ->orderBy('name')
                ->get();
        }

        // Get subjects based on selected class level and teacher
        if ($this->isTeacher) {
            // Get all teacher's subjects for this class level
            $subjects = Subject::join('teacher_subjects', 'subjects.id', '=', 'teacher_subjects.subject_id')
                ->select('subjects.id', 'subjects.name', 'subjects.slug')
                ->where('teacher_subjects.teacher_id', $this->user->id)
                ->where('teacher_subjects.class', $ccclass)
                ->orderBy('subjects.name')
                ->distinct()
                ->get();

            // If a classroom is selected, filter for that specific classroom if needed
            if (!empty($this->class_id)) {
                $classroomSubjects = Subject::join('teacher_subjects', 'subjects.id', '=', 'teacher_subjects.subject_id')
                    ->select('subjects.id', 'subjects.name', 'subjects.slug')
                    ->where('teacher_subjects.teacher_id', $this->user->id)
                    ->where('teacher_subjects.classroom_id', $this->class_id)
                    ->orderBy('subjects.name')
                    ->distinct()
                    ->get();

                // If there are specific subjects for this classroom, use those instead
                if ($classroomSubjects->count() > 0) {
                    $subjects = $classroomSubjects;
                }
            }
        } else {
            // For admins, show all subjects for the selected class level
            $subjects = Subject::join('teacher_subjects', 'subjects.id', '=', 'teacher_subjects.subject_id')
                ->select('subjects.id', 'subjects.name', 'subjects.slug')
                ->where('teacher_subjects.class', $ccclass)
                ->orderBy('subjects.name')
                ->distinct()
                ->get();
        }

        // Get all students for the selected classroom and sort by name
        $students = [];
        if ($this->class_id) {
            $classroom = Classroom::where('id', $this->class_id)->first();
            if ($classroom) {
                $students = $classroom->students()->get()->sortBy('name')->values();
            }
        }

        return view('livewire.admin.reports.generate-subject-combine', [
            'subjects' => $subjects,
            'classes' => $classrooms,
            'teacher' => TeacherSubject::where("teacher_id", $this->user->id)->get(),
            'students' => $students,
            'availableClassLevels' => $availableClassLevels,
            'isTeacher' => $this->isTeacher
        ]);
    }
}
