<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class FeeMonthDeadline extends Model
{
    use HasFactory;

    protected $fillable = [
        'month',
        'year',
        'deadline',
        'allow_late_payment',
        'notes'
    ];

    protected $casts = [
        'deadline' => 'date',
        'allow_late_payment' => 'boolean',
    ];

    /**
     * Buscar prazo configurado para um mês/ano
     *
     * @param string $month Nome do mês (Janeiro, Fevereiro, etc.)
     * @param int $year Ano (2025, 2026, etc.)
     * @return self|null
     */
    public static function getDeadline(string $month, int $year): ?self
    {
        return self::where('month', $month)
            ->where('year', $year)
            ->first();
    }

    /**
     * Calcular data de expiração para referência
     *
     * @param string $month
     * @param int $year
     * @return Carbon
     */
    public static function calculateExpiresAt(string $month, int $year): Carbon
    {
        $deadline = self::getDeadline($month, $year);

        if ($deadline && $deadline->deadline) {
            // Se tem deadline configurado, usar essa data às 23:59:59
            return Carbon::parse($deadline->deadline)->endOfDay();
        }

        // Fallback: usar configuração antiga (3 dias)
        $ttlDays = (int) config('payments.reference_ttl_days', 3);
        return now()->addDays($ttlDays)->endOfDay();
    }

    /**
     * Verificar se permite pagamento atrasado
     *
     * @param string $month
     * @param int $year
     * @return bool
     */
    public static function allowsLatePayment(string $month, int $year): bool
    {
        $deadline = self::getDeadline($month, $year);

        return $deadline ? $deadline->allow_late_payment : true;
    }

    /**
     * Verificar se prazo já passou
     *
     * @return bool
     */
    public function isExpired(): bool
    {
        if (!$this->deadline) {
            return false; // Sem prazo = nunca expira
        }

        return Carbon::parse($this->deadline)->endOfDay()->isPast();
    }
}
