<?php

namespace App\Services;

use App\Models\Exam;
use App\Models\User;
use App\Models\Subject;
use App\Models\Classroom;
use App\Models\Trimester;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class ExcelMarksImportService
{
    /**
     * Import marks from Excel file
     *
     * @param string $filePath
     * @param array $importData
     * @return array
     */
    public function importMarks(string $filePath, array $importData): array
    {
        try {
            DB::beginTransaction();

            // Validate import data
            $this->validateImportData($importData);

            // Load Excel file
            $spreadsheet = IOFactory::load($filePath);
            $worksheet = $spreadsheet->getActiveSheet();

            // Parse Excel data
            $excelData = $this->parseExcelData($worksheet, $importData);

            // Clean existing marks if requested
            if ($importData['clean_existing']) {
                $this->cleanExistingMarks($importData);
            }

            // Import marks with enhanced error reporting
            $result = $this->processMarksImportWithDetailedErrors($excelData, $importData);

            DB::commit();

            return [
                'success' => true,
                'message' => 'Marks imported successfully',
                'data' => $result
            ];

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Excel import failed: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => $e->getMessage(),
                'data' => null
            ];
        }
    }

    /**
     * Validate import data
     *
     * @param array $data
     * @throws Exception
     */
    private function validateImportData(array $data): void
    {
        $required = ['year', 'trimester_id', 'subject_id', 'classroom_id'];

        foreach ($required as $field) {
            if (empty($data[$field])) {
                throw new Exception("Field {$field} is required");
            }
        }

        // Validate entities exist
        if (!Subject::find($data['subject_id'])) {
            throw new Exception('Subject not found');
        }

        if (!Classroom::find($data['classroom_id'])) {
            throw new Exception('Classroom not found');
        }

        if (!in_array($data['trimester_id'], [1, 2, 3])) {
            throw new Exception('Invalid trimester');
        }
    }

    /**
     * Parse Excel data based on the specific structure
     *
     * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $worksheet
     * @param array $importData
     * @return array
     */
    private function parseExcelData($worksheet, array $importData): array
    {
        $data = [];
        $highestRow = $worksheet->getHighestRow();

        // Define the expected structure based on your Excel format
        $columnMapping = [
            'A' => 'number',        // N°
            'B' => 'student_name',  // NOME COMPLETO
            'C' => 'sex',           // SEXO
        ];

        // Determine which trimester we're importing for
        $trimester = $importData['trimester_id'];

        // Map trimester columns based on your Excel structure
        $trimesterColumns = $this->getTrimesterColumns($trimester);

        // Find the data start row (usually after headers)
        $dataStartRow = $this->findDataStartRow($worksheet);

        if (!$dataStartRow) {
            throw new Exception('Could not find data start row in Excel file');
        }

        // Parse data rows
        for ($row = $dataStartRow; $row <= $highestRow; $row++) {
            $rowData = $this->parseDataRowByStructure($worksheet, $row, $trimesterColumns, $trimester);

            if ($rowData && !empty($rowData['student_name'])) {
                $rowData['excel_row'] = $row; // Add row number for error reporting
                $data[] = $rowData;
            }
        }

        if (empty($data)) {
            throw new Exception('No valid data found in Excel file');
        }

        return $data;
    }

    /**
     * Get trimester column mapping based on Excel structure
     *
     * @param int $trimester
     * @return array
     */
    private function getTrimesterColumns(int $trimester): array
    {
        // Based on your Excel structure:
        // 1º Trimestre: columns D-H (ACS 1ª, 2ª, MACS, AT, MT)
        // 2º Trimestre: columns I-M (1ª, 2ª, MACS, AT, MT)
        // 3º Trimestre: columns N-R (ACS 1ª, 2ª, MACS, AT, MT)

        switch ($trimester) {
            case 1:
                return [
                    'ACS1a' => 'D',  // ACS 1ª
                    'ACS2a' => 'E',  // 2ª
                    'MACS' => 'F',   // MACS
                    'AT' => 'G',     // AT
                    'MT' => 'H',     // MT
                ];
            case 2:
                return [
                    'ACS1a' => 'I',  // 1ª
                    'ACS2a' => 'J',  // 2ª
                    'MACS' => 'K',   // MACS
                    'AT' => 'L',     // AT
                    'MT' => 'M',     // MT
                ];
            case 3:
                return [
                    'ACS1a' => 'N',  // ACS 1ª
                    'ACS2a' => 'O',  // 2ª
                    'MACS' => 'P',   // MACS
                    'AT' => 'Q',     // AT
                    'MT' => 'R',     // MT
                    'NE' => 'S',     // This might be NE column for 3rd trimester
                ];
            default:
                throw new Exception('Invalid trimester');
        }
    }

    /**
     * Find the row where actual data starts
     *
     * @param $worksheet
     * @return int|null
     */
    private function findDataStartRow($worksheet): ?int
    {
        $highestRow = $worksheet->getHighestRow();

        // Look for the first row that has a number in column A and a name in column B
        for ($row = 10; $row <= min(20, $highestRow); $row++) {
            $numberCell = trim($worksheet->getCell('A' . $row)->getValue() ?? '');
            $nameCell = trim($worksheet->getCell('B' . $row)->getValue() ?? '');

            // Check if this looks like a data row (number + name)
            if (is_numeric($numberCell) && !empty($nameCell) && strlen($nameCell) > 5) {
                return $row;
            }
        }

        return null;
    }

    /**
     * Parse a data row based on the Excel structure
     *
     * @param $worksheet
     * @param int $row
     * @param array $trimesterColumns
     * @param int $trimester
     * @return array|null
     */
    private function parseDataRowByStructure($worksheet, int $row, array $trimesterColumns, int $trimester): ?array
    {
        // Get student information
        $studentName = trim($worksheet->getCell('B' . $row)->getValue() ?? '');

        // Skip empty rows
        if (empty($studentName)) {
            return null;
        }

        $data = [
            'student_name' => $studentName,
            'student_id' => null, // This Excel doesn't seem to have student IDs
        ];

        // Get marks for the specific trimester
        foreach ($trimesterColumns as $markType => $column) {
            $cell = $worksheet->getCell($column . $row);
            $data[$markType] = $this->parseNumericValue($cell->getValue(), $cell);
        }

        return $data;
    }

    /**
     * Parse numeric value from Excel cell
     *
     * @param mixed $value
     * @param null $cell
     * @return float|null
     */
    private function parseNumericValue(mixed $value, $cell = null): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }

        // Handle Excel date values
        if ($cell && Date::isDateTime($cell)) {
            return null;
        }

        // Convert to string and clean
        $value = $this->formatMark($value);
        if ($value === null) {
            return null;
        }

        $numeric = $value;

        // Validate range (0-20)
        if ($numeric < 0 || $numeric > 20) {
            return null;
        }

        return $numeric;
    }

    /**
     * Clean existing marks
     *
     * @param array $importData
     */
    private function cleanExistingMarks(array $importData): void
    {
        $deleted = Exam::where([
            'classroom_id' => $importData['classroom_id'],
            'subject_id' => $importData['subject_id'],
            'trimester_id' => $importData['trimester_id'],
            'year' => $importData['year']
        ])->delete();

        Log::info("Cleaned {$deleted} existing exam records");
    }

    /**
     * Process marks import with detailed error reporting
     *
     * @param array $excelData
     * @param array $importData
     * @return array
     */
    private function processMarksImportWithDetailedErrors(array $excelData, array $importData): array
    {
        $stats = [
            'total_rows' => count($excelData),
            'successful' => 0,
            'failed' => 0,
            'errors' => [],
            'detailed_errors' => [
                'student_in_different_classroom' => [],
                'student_name_not_found' => [],
                'other_errors' => []
            ]
        ];

        // Get student corrections if any
        $studentCorrections = $importData['student_corrections'] ?? [];

        foreach ($excelData as $index => $row) {
            try {
                $studentResult = $this->findStudentWithDetails($row, $importData['classroom_id'], $studentCorrections);

                if ($studentResult['found']) {
                    $this->createOrUpdateExam($studentResult['student'], $row, $importData);
                    $stats['successful']++;
                } else {
                    // Handle different types of errors
                    $errorMessage = "Row {$row['excel_row']}: {$studentResult['error_message']} - {$row['student_name']}";

                    switch ($studentResult['error_type']) {
                        case 'different_classroom':
                            $stats['detailed_errors']['student_in_different_classroom'][] = [
                                'row_number' => $row['excel_row'],
                                'student_name' => $row['student_name'],
                                'current_classroom' => $studentResult['student_classroom'],
                                'expected_classroom' => $importData['classroom_id'],
                                'student_id' => $studentResult['student_id'],
                                'message' => $errorMessage
                            ];
                            break;
                        case 'name_not_found':
                            $stats['detailed_errors']['student_name_not_found'][] = [
                                'row_number' => $row['excel_row'],
                                'student_name' => $row['student_name'],
                                'suggestions' => $studentResult['similar_names'] ?? [],
                                'message' => $errorMessage
                            ];
                            break;
                        default:
                            $stats['detailed_errors']['other_errors'][] = [
                                'row_number' => $row['excel_row'],
                                'student_name' => $row['student_name'],
                                'message' => $errorMessage
                            ];
                    }

                    $stats['errors'][] = $errorMessage;
                    $stats['failed']++;
                }

            } catch (Exception $e) {
                $errorMessage = "Row {$row['excel_row']}: " . $e->getMessage();
                $stats['errors'][] = $errorMessage;
                $stats['detailed_errors']['other_errors'][] = [
                    'row_number' => $row['excel_row'],
                    'student_name' => $row['student_name'] ?? 'Unknown',
                    'message' => $errorMessage
                ];
                $stats['failed']++;
            }
        }

        return $stats;
    }

    /**
     * Find student with detailed error information and apply corrections
     *
     * @param array $row
     * @param string $expectedClassroomId
     * @param array $studentCorrections
     * @return array
     */
    private function findStudentWithDetails(array $row, string $expectedClassroomId, array $studentCorrections = []): array
    {
        if (empty($row['student_name'])) {
            return [
                'found' => false,
                'error_type' => 'name_not_found',
                'error_message' => 'Empty student name'
            ];
        }

        $studentName = $row['student_name'];

        // Check if there's a correction for this student name
        if (isset($studentCorrections[$studentName])) {
            $correctedStudentId = $studentCorrections[$studentName];
            $student = User::find($correctedStudentId);

            if ($student && $student->classroom_id == $expectedClassroomId) {
                return [
                    'found' => true,
                    'student' => $student
                ];
            }
        }

        // First, try to find exact match in the expected classroom
        $student = User::where('classroom_id', $expectedClassroomId)
            ->where('name', $studentName)
            ->first();

        if ($student) {
            return [
                'found' => true,
                'student' => $student
            ];
        }

        // Try partial match in the expected classroom
        $student = User::where('classroom_id', $expectedClassroomId)
            ->where('name', 'LIKE', '%' . $studentName . '%')
            ->first();

        if ($student) {
            return [
                'found' => true,
                'student' => $student
            ];
        }

        // Check if student exists in a different classroom
        $studentInDifferentClass = User::where('name', $studentName)->first();

        if ($studentInDifferentClass) {
            $classroom = Classroom::find($studentInDifferentClass->classroom_id);
            return [
                'found' => false,
                'error_type' => 'different_classroom',
                'error_message' => 'Student found in different classroom',
                'student_classroom' => $classroom ? $classroom->name : 'Unknown',
                'student_id' => $studentInDifferentClass->id
            ];
        }

        // Try partial match across all classrooms
        $studentInDifferentClass = User::where('name', 'LIKE', '%' . $studentName . '%')->first();

        if ($studentInDifferentClass) {
            $classroom = Classroom::find($studentInDifferentClass->classroom_id);
            return [
                'found' => false,
                'error_type' => 'different_classroom',
                'error_message' => 'Student found in different classroom (partial match)',
                'student_classroom' => $classroom ? $classroom->name : 'Unknown',
                'student_id' => $studentInDifferentClass->id
            ];
        }

        // Find similar names for suggestions
        $similarNames = User::where('name', 'LIKE', '%' . substr($studentName, 0, 5) . '%')
            ->limit(5)
            ->pluck('name')
            ->toArray();

        return [
            'found' => false,
            'error_type' => 'name_not_found',
            'error_message' => 'Student not found',
            'similar_names' => $similarNames
        ];
    }

    /**
     * Create or update exam record
     *
     * @param User $student
     * @param array $row
     * @param array $importData
     */
    private function createOrUpdateExam(User $student, array $row, array $importData): void
    {
        $examData = [
            'student_id' => $student->id,
            'subject_id' => $importData['subject_id'],
            'trimester_id' => $importData['trimester_id'],
            'classroom_id' => $importData['classroom_id'],
            'year' => $importData['year'],
        ];

        // Add marks if they exist
        $markFields = ['ACS1a', 'ACS2a', 'AT', 'MT', 'NE'];
        foreach ($markFields as $field) {
            if (isset($row[$field])) {
                $examData[$field] = $this->formatMark($row[$field]);
            }
        }

        // Calculate MACS if both ACS1a and ACS2a exist and MACS is not provided
        if (isset($examData['ACS1a']) && isset($examData['ACS2a']) && !isset($row['MACS'])) {
            $examData['MACS'] = $this->formatMark(round(($examData['ACS1a'] + $examData['ACS2a']) / 2, 2));
        } elseif (isset($row['MACS'])) {
            $examData['MACS'] = $row['MACS'];
        }

        Exam::updateOrCreate(
            [
                'student_id' => $examData['student_id'],
                'subject_id' => $examData['subject_id'],
                'trimester_id' => $examData['trimester_id'],
                'classroom_id' => $examData['classroom_id'],
                'year' => $examData['year'],
            ],
            $examData
        );
    }

    /**
     * Validate Excel file format
     *
     * @param string $filePath
     * @return array
     */
    public function validateExcelFile(string $filePath): array
    {
        try {
            $spreadsheet = IOFactory::load($filePath);
            $worksheet = $spreadsheet->getActiveSheet();

            // Basic validation
            $highestRow = $worksheet->getHighestRow();
            if ($highestRow < 10) {
                return [
                    'valid' => false,
                    'message' => 'Excel file appears to be empty or too small'
                ];
            }

            // Check for expected structure
            $dataStartRow = $this->findDataStartRow($worksheet);

            if (!$dataStartRow) {
                return [
                    'valid' => false,
                    'message' => 'Could not find student data in Excel file. Please ensure the file follows the expected format.'
                ];
            }

            // Check if we have student names
            $hasStudents = false;
            for ($row = $dataStartRow; $row <= min($dataStartRow + 5, $highestRow); $row++) {
                $studentName = trim($worksheet->getCell('B' . $row)->getValue() ?? '');
                if (!empty($studentName) && strlen($studentName) > 3) {
                    $hasStudents = true;
                    break;
                }
            }

            if (!$hasStudents) {
                return [
                    'valid' => false,
                    'message' => 'No student names found in Excel file'
                ];
            }

            // Get preview of found students
            $preview = [];
            for ($row = $dataStartRow; $row <= min($dataStartRow + 40, $highestRow); $row++) {
                $studentName = trim($worksheet->getCell('B' . $row)->getValue() ?? '');
                if (!empty($studentName)) {
                    $preview[] = $studentName;
                }
            }

            return [
                'valid' => true,
                'message' => 'Excel file format is valid',
                'preview' => $preview,
                'data_start_row' => $dataStartRow,
                'total_rows' => $highestRow
            ];

        } catch (Exception $e) {
            return [
                'valid' => false,
                'message' => 'Could not read Excel file: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Format marks properly before validation and saving
     *
     * @param string $mark The mark to format
     * @return float|null The formatted mark
     */
    private function formatMark($mark)
    {
        if ($mark === '' || $mark === null) {
            return null;
        }

        // Replace comma with dot if present
        $mark = str_replace(',', '.', trim($mark));

        // Remove any trailing dots
        $mark = rtrim($mark, '.');

        return is_numeric($mark) ? (float) $mark : null;
    }
}
