<?php

//namespace App\Services;
//
//use App\Models\PaymentReference;
//use App\Models\User;
//use Carbon\Carbon;
//use Illuminate\Support\Facades\DB;
//use Illuminate\Support\Facades\Log;
//use Exception;
//
//class PaymentReconciliationService
//{
//    const HEADER_TYPE  = '0';
//    const DETAIL_BMEPS = '1';
//    const DETAIL_MEPS  = '2';
//    const TRAILER_TYPE = '9';
//
//    // ======= Diretórios (todos em public/bci/bmeps/...) =======
//    private function baseDir(): string      { return public_path('bci/bmeps'); }
//    private function inDir(): string        { return $this->baseDir() . '/incoming'; }
//    private function processedDir(): string { return $this->baseDir() . '/processed'; }
//    private function errorDir(): string     { return $this->baseDir() . '/failed'; }
//    private function reportsDir(): string   { return $this->baseDir() . '/reports'; }
//
//    // ======= Estado =======
//    private array $processedReferences = [];
//    private array $errors = [];
//    private array $stats = [
//        'total_files'      => 0,
//        'total_records'    => 0,
//        'reconciled'       => 0,
//        'failed'           => 0,
//        'duplicates'       => 0,
//        'not_found'        => 0,
//        'total_amount'     => 0.0,
//        'autoscaled_10x'   => 0,
//        'autoscaled_div10' => 0,
//    ];
//
//    // ======= Entrada principal =======
//    public function processAllPendingFiles(): array
//    {
//        Log::info('Starting reconciliation process (BMEPS/MEPS)');
//
//        try {
//            $files = $this->fetchLocalFiles();
//            if (empty($files)) {
//                Log::warning('No files found in incoming', ['dir' => $this->inDir()]);
//                return $this->stats;
//            }
//
//            Log::info('Found files', ['count' => count($files), 'dir' => $this->inDir(), 'files' => array_map('basename', $files)]);
//
//            foreach ($files as $file) {
//                $this->processFile($file);
//            }
//
//            $this->generateReconciliationReport();
//            Log::info('Reconciliation completed', $this->stats);
//
//        } catch (Exception $e) {
//            Log::error('Global reconciliation failure', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
//        }
//
//        return $this->stats;
//    }
//
//    // ======= FS utils =======
//    private function ensureDirs(): void
//    {
//        foreach ([$this->inDir(), $this->processedDir(), $this->errorDir(), $this->reportsDir()] as $dir) {
//            if (!is_dir($dir)) {
//                @mkdir($dir, 0775, true);
//                Log::info('Ensured directory', ['path' => $dir]);
//            }
//        }
//    }
//
//    private function fetchLocalFiles(): array
//    {
//        $this->ensureDirs();
//        $files = glob($this->inDir() . '/*.txt', GLOB_NOSORT) ?: [];
//        $files = array_merge($files, glob($this->inDir() . '/*.TXT', GLOB_NOSORT) ?: []);
//        return array_unique($files);
//    }
//
//    private function moveFile(string $file, bool $success): void
//    {
//        $destDir = $success ? $this->processedDir() : $this->errorDir();
//        $this->ensureDirs();
//
//        $basename = basename($file);
//        $target   = $destDir . '/' . $basename;
//
//        if (file_exists($target)) {
//            $target = $destDir . '/' . now()->format('Ymd_His') . '_' . $basename;
//        }
//
//        if (@rename($file, $target)) {
//            Log::info('File moved', ['from' => $file, 'to' => $target]);
//        } elseif (@copy($file, $target)) {
//            @unlink($file);
//            Log::info('File copied+removed', ['from' => $file, 'to' => $target]);
//        } else {
//            Log::warning('Could not move file', ['file' => $file, 'to' => $target]);
//        }
//    }
//
//    private function generateReconciliationReport(): string
//    {
//        $report = [
//            'process_date' => now()->toDateTimeString(),
//            'stats'        => $this->stats,
//            'errors'       => $this->errors
//        ];
//
//        $dir = $this->reportsDir();
//        if (!file_exists($dir)) @mkdir($dir, 0775, true);
//
//        $reportPath = $dir . '/rep_' . now()->format('Ymd_His') . '.json';
//        @file_put_contents($reportPath, json_encode($report, JSON_PRETTY_PRINT));
//        Log::info('Reconciliation report generated', ['path' => $reportPath]);
//
//        return $reportPath;
//    }
//
//    // ======= Core por ficheiro =======
//    private function processFile(string $file): void
//    {
//        $this->stats['total_files']++;
//        Log::info("Processing file", ['file' => basename($file)]);
//
//        try {
//            $content = @file_get_contents($file);
//            if ($content === false || $content === '') {
//                throw new Exception("Empty or unreadable file");
//            }
//
//            $parsed = $this->parseFile($content);
//            if (!$this->validateFileStructure($parsed)) {
//                throw new Exception("Invalid file structure");
//            }
//
//            $this->saveFileInfo($file, $parsed);
//
//            DB::beginTransaction();
//
//            foreach ($parsed['details'] as $detail) {
//                $this->processTransaction($detail, $file);
//            }
//
//            $this->validateTrailer($parsed); // só loga divergências
//
//            DB::commit();
//            $this->finalizeLog(basename($file), 'completed');
//            $this->moveFile($file, true);
//
//        } catch (Exception $e) {
//            DB::rollBack();
//
//            Log::error('Failed to process file', ['file' => basename($file), 'error' => $e->getMessage()]);
//            $this->errors[] = ['file' => basename($file), 'error' => $e->getMessage(), 'timestamp' => now()->toDateTimeString()];
//            $this->finalizeLog(basename($file), 'failed');
//            $this->moveFile($file, false);
//        }
//    }
//
//    private function parseFile(string $content): array
//    {
//        $lines  = preg_split('/\r\n|\r|\n/', rtrim($content));
//        $result = ['header' => null, 'details' => [], 'trailer' => null];
//
//        // Detecta MEPS (header contém "MEPS" OU existe linha tipo '2')
//        $isMEPS = false;
//        if (!empty($lines)) {
//            $first = $lines[0];
//            if (stripos($first, 'MEPS') !== false) {
//                $isMEPS = true;
//            } else {
//                foreach ($lines as $l) {
//                    if (isset($l[0]) && $l[0] === self::DETAIL_MEPS) { $isMEPS = true; break; }
//                }
//            }
//        }
//        Log::info('Detected format', ['format' => $isMEPS ? 'MEPS' : 'BMEPS']);
//
//        foreach ($lines as $i => $raw) {
//            $line = rtrim($raw);
//            if ($line === '') continue;
//
//            $type = substr($line, 0, 1);
//            Log::debug("Parsing line", ['index' => $i, 'type' => $type, 'len' => strlen($line)]);
//
//            if ($type === self::HEADER_TYPE) {
//                $result['header'] = $isMEPS ? $this->parseHeaderMEPS($line) : $this->parseHeaderBMEPS($line);
//                continue;
//            }
//            if ($type === self::TRAILER_TYPE) {
//                $result['trailer'] = $isMEPS ? $this->parseTrailerMEPS($line) : $this->parseTrailerBMEPS($line);
//                continue;
//            }
//
//            if ($isMEPS) {
//                if ($type === self::DETAIL_MEPS) {
//                    $d = $this->parseDetailMEPS($line);
//                    if ($d) $result['details'][] = $d;
//                } else {
//                    Log::warning('MEPS unexpected record type', ['type' => $type, 'line' => $line]);
//                }
//            } else {
//                if ($type === self::DETAIL_BMEPS) {
//                    $d = $this->parseDetailBMEPS($line);
//                    if ($d) $result['details'][] = $d;
//                } else {
//                    Log::warning('BMEPS unexpected record type', ['type' => $type, 'line' => $line]);
//                }
//            }
//        }
//
//        // Se header sem data, usa a menor data dos detalhes (quando existir)
//        if (($result['header']['creation_date'] ?? null) === null && !empty($result['details'])) {
//            $minDate = min(array_map(fn($d) => $d['date'] ?? date('Y-m-d'), $result['details']));
//            $result['header']['creation_date'] = $minDate;
//        }
//
//        // --- Auto-correção pelo trailer quando há 1 único detalhe (caso observado 497,76 -> 4 977,60) ---
//        if ($result['trailer'] && count($result['details']) === 1) {
//            $detSum = (float) $result['details'][0]['amount'];
//            $trlSum = (float) $result['trailer']['total_amount'];
//
//            if (abs($detSum - $trlSum) > 0.009) {
//                if (abs(($detSum * 10) - $trlSum) <= 0.009) {
//                    Log::warning('Detail amount auto-corrected by trailer x10', [
//                        'from' => $detSum, 'to' => $detSum * 10, 'trailer' => $trlSum
//                    ]);
//                    $result['details'][0]['amount'] = $detSum * 10;
//                }
//            }
//        }
//
//        Log::info('File parsed', [
//            'has_header'    => !is_null($result['header']),
//            'details_count' => count($result['details']),
//            'has_trailer'   => !is_null($result['trailer'])
//        ]);
//
//        return $result;
//    }
//
//    // ======= Parsers BMEPS =======
//    private function parseHeaderBMEPS(string $line): array
//    {
//        return [
//            'type'          => '0',
//            'format'        => 'BMEPS',
//            'company_id'    => substr($line, 1, 3),
//            'profile_id'    => substr($line, 4, 2),
//            'creation_date' => $this->parseDateToYmd(substr($line, 6, 8)),
//            'file_id'       => substr($line, 14, 5),
//            'raw'           => $line,
//        ];
//    }
//
//    private function parseDetailBMEPS(string $line): ?array
//    {
//        $reference  = $this->normalizeReference(substr($line, 1, 11) ?: substr($line, 72, 11));
//        $amount     = $this->parseAmount(substr($line, 12, 16));
//        $commission = $this->parseAmount(substr($line, 28, 16));
//        $date       = $this->parseDateToYmd(substr($line, 44, 8));
//        $time       = $this->parseTime(substr($line, 52, 6));
//        $txnId      = trim(substr($line, 58, 7));
//        $termId     = trim(substr($line, 65, 10));
//        $location   = rtrim(substr($line, 75, 16) ?: '');
//
//        Log::debug('Parsed BMEPS detail', compact('reference','amount','commission','date','time','txnId','termId'));
//        if (!$reference) return null;
//
//        return [
//            'type'              => '1',
//            'reference'         => $reference,
//            'amount'            => $amount,
//            'commission'        => $commission,
//            'date'              => $date,
//            'time'              => $time,
//            'transaction_id'    => $txnId ?: null,
//            'terminal_id'       => $termId ?: null,
//            'terminal_location' => $location ?: null,
//            'raw'               => $line,
//        ];
//    }
//
//    private function parseTrailerBMEPS(string $line): array
//    {
//        return [
//            'type'             => '9',
//            'total_records'    => (int) substr($line, 1, 7),
//            'total_amount'     => $this->parseAmount(substr($line, 8, 16)),
//            'total_commission' => $this->parseAmount(substr($line, 24, 16)),
//            'raw'              => $line,
//        ];
//    }
//
//    // ======= Parsers MEPS (robustos) =======
//    private function parseHeaderMEPS(string $line): array
//    {
//        // tenta extrair AAAAMMDD; se não houver, deixamos null para ser preenchido pelos detalhes
//        $date = null;
//        if (preg_match('/(20\d{6})/', $line, $m)) {
//            $date = $this->parseDateToYmd($m[1]);
//        }
//
//        return [
//            'type'          => '0',
//            'format'        => 'MEPS',
//            'creation_date' => $date,
//            'raw'           => $line,
//        ];
//    }
//
//    private function parseDetailMEPS(string $line): ?array
//    {
//        // Layout observado (BCI - MEPS):
//        // 2 ... 20YYYYMMDD HHMM[SS] AMOUNT(12) COMMISSION(10|12) [0–3 espaços] 0{10,} [espaços] REF(11) 00
//        $re = '/^2.*?'
//            . '(?P<date>20\d{6})'            // data: 20YYYYMMDD
//            . '(?P<time>\d{4}|\d{6})'        // hora: HHMM ou HHMMSS
//            . '(?P<amount>\d{12})'           // amount: exatamente 12 dígitos (centavos)
//            . '(?P<commission>\d{10,12})'    // comissão: 10–12 dígitos (centavos)
//            . '\s{0,3}'                      // até 3 espaços (no exemplo são 2)
//            . '0{10,}'                       // bloco de zeros (ancoragem)
//            . '\s*'                          // espaços opcionais
//            . '(?P<ref>\d{11})'              // referência: 11 dígitos
//            . '00\s*$'                       // sufixo "00" e fim de linha
//            . '/';
//
//        $trimmed = rtrim($line);
//        if (!preg_match($re, $trimmed, $m)) {
//            Log::warning('MEPS detail not matched (line)', ['line' => $trimmed, 'len' => strlen($trimmed)]);
//            return null;
//        }
//
//        $reference  = $this->normalizeReference($m['ref']);      // "...7100" -> "00001250271"
//        $amount     = ((int)$m['amount'])     / 100.0;           // centavos -> MT
//        $commission = ((int)$m['commission']) / 100.0;
//        $date       = $this->parseDateToYmd($m['date']);         // YYYYMMDD
//
//        $timeRaw    = $m['time'];
//        $time       = (strlen($timeRaw) === 4)
//            ? substr($timeRaw,0,2).':'.substr($timeRaw,2,2).':00'
//            : $this->parseTime($timeRaw);                        // HHMMSS -> HH:MM:SS
//
//        Log::debug('Parsed MEPS detail (fixed widths)', compact('reference','amount','commission','date','time'));
//        if (!$reference) return null;
//
//        return [
//            'type'              => '2',
//            'reference'         => $reference,
//            'amount'            => $amount,
//            'commission'        => $commission,
//            'date'              => $date,
//            'time'              => $time,
//            'transaction_id'    => null,
//            'terminal_id'       => null,
//            'terminal_location' => null,
//            'raw'               => $line,
//        ];
//    }
//
//    private function parseTrailerMEPS(string $line): array
//    {
//        // 9 00000001 0000000000497760 0000000000078400 ...
//        $count     = (int) substr($line, 1, 8);
//        $totalAmt  = (int) substr($line, 9, 16)  / 100.0;
//        $totalComm = (int) substr($line, 25, 16) / 100.0;
//
//        return [
//            'type'             => '9',
//            'total_records'    => $count,
//            'total_amount'     => $totalAmt,
//            'total_commission' => $totalComm,
//            'raw'              => $line,
//        ];
//    }
//
//    // ======= Negócio =======
//    private function processTransaction(array $detail, string $file): void
//    {
//        if (empty($detail['reference'])) {
//            Log::warning('Skipping detail without reference');
//            return;
//        }
//
//        $this->stats['total_records']++;
//
//        $reference = (string) $detail['reference'];
//        $amount    = (float)  $detail['amount'];
//
//        $paymentReference = PaymentReference::where('reference_number', $reference)
//            ->whereIn('status', ['pending', 'paid'])
//            ->latest('updated_at')
//            ->first();
//
//        if (!$paymentReference) {
//            $this->stats['not_found']++;
//            Log::warning('Reference not found', ['reference' => $reference, 'file' => basename($file)]);
//            $this->logUnmatchedPayment($detail);
//            return;
//        }
//
//        if ($paymentReference->status === 'paid') {
//            $this->stats['duplicates']++;
//            Log::notice('Duplicate payment (already paid)', ['reference' => $reference]);
//            return;
//        }
//
//        // valida/normaliza montante
//        $paidAmount = $amount;
//        if (!$this->validateAmount($paymentReference->amount, $paidAmount)) {
//            $this->stats['failed']++;
//            Log::error('Failed to process transaction', [
//                'reference'       => $detail['reference'] ?? null,
//                'amount_parsed'   => $detail['amount'] ?? null,
//                'expected_amount' => $paymentReference->amount,
//                'detail'          => $detail,
//            ]);
//            return;
//        }
//        if (abs($paidAmount - $amount) > 0.009) {
//            if (abs($paidAmount - ($amount * 10)) < 0.009) {
//                $this->stats['autoscaled_10x']++;
//                Log::warning('Paid amount autoscaled x10', ['reference' => $reference, 'from' => $amount, 'to' => $paidAmount]);
//            } elseif (abs($paidAmount - ($amount / 10)) < 0.009) {
//                $this->stats['autoscaled_div10']++;
//                Log::warning('Paid amount autoscaled /10', ['reference' => $reference, 'from' => $amount, 'to' => $paidAmount]);
//            }
//            $amount = $paidAmount;
//        }
//
//        // garantir transaction_id (MEPS normalmente não traz)
//        $this->ensureTransactionId($detail, basename($file));
//
//        DB::transaction(function () use ($paymentReference, $detail, $amount) {
//            $paymentReference->status         = 'paid';
//            $paymentReference->paid_at        = Carbon::parse(($detail['date'] ?? date('Y-m-d')) . ' ' . ($detail['time'] ?? '00:00:00'));
//            $paymentReference->transaction_id = $detail['transaction_id'] ?? null;
//            $paymentReference->terminal_id    = $detail['terminal_id'] ?? null;
//            $paymentReference->save();
//
//            $this->createFeePayment($paymentReference, array_merge($detail, ['amount' => $amount]));
//            $this->markAsProcessed($detail, $amount);
//        });
//
//        $this->stats['reconciled']++;
//        $this->stats['total_amount'] += $amount;
//
//        Log::info('Payment reconciled successfully', [
//            'reference'  => $reference,
//            'student_id' => $paymentReference->student_id ?? null,
//            'amount'     => $amount
//        ]);
//
//        // pós-checagem
//        Log::info('Post-reconcile check', [
//            'reference'       => $detail['reference'],
//            'transaction_id'  => $detail['transaction_id'] ?? null,
//        ]);
//
//        try {
//            $exists = DB::table('payment_reconciliations')
//                ->where('reference', $detail['reference'])
//                ->where('transaction_id', $detail['transaction_id'] ?? '')
//                ->exists();
//            Log::info('payment_reconciliations exists?', ['exists' => $exists ? 'yes' : 'no']);
//        } catch (\Throwable $e) {
//            Log::warning('payment_reconciliations exists? check failed', ['error' => $e->getMessage()]);
//        }
//    }
//
//    // ======= Grava Fee (sem $fillable) + preserva multa =======
//    private function createFeePayment($paymentReference, array $detail): void
//    {
//        $student = User::find($paymentReference->student_id ?? null);
//        if (!$student) {
//            Log::warning('Student not found for payment reference', ['student_id' => $paymentReference->student_id]);
//            return;
//        }
//
//        $expectedBase = (float)($paymentReference->amount ?? 0);
//        $expectedFine = (float)($paymentReference->fine_amount ?? 0);
//        $paid         = (float)$detail['amount'];
//
//        // Distribui pago em base + multa (se pago > base, delta vira multa)
//        [$basePaid, $finePaid] = $this->splitPaidIntoBaseAndFine($expectedBase, $expectedFine, $paid);
//
//        // Datas
//        $payDate = Carbon::parse(($detail['date'] ?? date('Y-m-d')) . ' ' . ($detail['time'] ?? '00:00:00'));
//        $payDay  = (int)$payDate->day;
//        $payMon  = (int)$payDate->month;
//        $payYear = (int)$payDate->year;
//
//        // month textual (mantém compatibilidade com tua tabela)
//        $monthText = $paymentReference->fee_month
//            ? (string)$paymentReference->fee_month
//            : $payDate->format('F'); // “January”, “February”, etc.
//
//        try {
//            DB::table('fee_assign')->insert([
//                'student_id'        => $paymentReference->student_id,
//                'fee_group_id'      => '1',
//
//                'status'            => 'Paid',
//                'pay_type'          => 'reference',
//
//                'transaction_id'    => $detail['transaction_id'] ?? null,
//                'reference_number'  => $detail['reference'] ?? null,
//
//                'payment_date'      => $payDate,
//
//                // colunas legadas
//                'day'               => $payDay,
//                'month'             => $monthText,
//                'year'              => $payYear,
//
//                'pay_day'           => $payDay,
//                'pay_month'         => $payMon,
//                'pay_year'          => $payYear,
//
//                'discount'          => 0.00,
//                'fine'              => $finePaid,    // multa preservada
//                'amount'            => $basePaid,    // base separada
//
//                'created_at'        => now(),
//                'updated_at'        => now(),
//            ]);
//
//            Log::info('Fee payment created (DB::table)', [
//                'student_id'     => $paymentReference->student_id,
//                'reference'      => $detail['reference'] ?? null,
//                'transaction_id' => $detail['transaction_id'] ?? null,
//                'paid_total'     => $paid,
//                'base_paid'      => $basePaid,
//                'fine_paid'      => $finePaid,
//                'pay_date'       => $payDate->toDateTimeString(),
//            ]);
//
//        } catch (\Throwable $e) {
//            Log::error('Fee_assign insert failed', [
//                'reference'      => $detail['reference'] ?? null,
//                'transaction_id' => $detail['transaction_id'] ?? null,
//                'error'          => $e->getMessage(),
//            ]);
//        }
//    }
//
//    // ======= Validadores / Helpers de negócio =======
//    private function validateAmount($expected, &$paid): bool
//    {
//        $expected  = (float)$expected;
//        $paidFloat = (float)$paid;
//        $tol       = (float) config('bmeps.amount_tolerance', 1.00);
//
//        Log::info('Amount check', [
//            'expected'   => $expected,
//            'paid_raw'   => $paidFloat,
//            'tolerance'  => $tol,
//            'diff'       => abs($expected - $paidFloat),
//        ]);
//
//        if (abs($expected - $paidFloat) <= $tol) return true;
//
//        $x10   = $paidFloat * 10;
//        $div10 = $paidFloat / 10;
//
//        Log::info('Amount alt candidates', [
//            'x10'        => $x10,
//            'div10'      => $div10,
//            'diff_x10'   => abs($expected - $x10),
//            'diff_div10' => abs($expected - $div10),
//        ]);
//
//        if (abs($expected - $x10) <= $tol) {
//            Log::warning('Amount auto-normalized x10', ['from' => $paidFloat, 'to' => $x10]);
//            $paid = $x10;
//            return true;
//        }
//        if (abs($expected - $div10) <= $tol) {
//            Log::warning('Amount auto-normalized /10', ['from' => $paidFloat, 'to' => $div10]);
//            $paid = $div10;
//            return true;
//        }
//
//        Log::warning('Amount mismatch exceeds tolerance (after scale tries)', [
//            'expected'   => $expected,
//            'paid'       => $paidFloat,
//            'difference' => abs($expected - $paidFloat),
//            'tolerance'  => $tol
//        ]);
//        return false;
//    }
//
//    private function ensureTransactionId(array &$detail, string $fileName = null): void
//    {
//        if (!empty($detail['transaction_id'])) return;
//
//        $date  = preg_replace('/\D+/', '', $detail['date'] ?? '');
//        $time  = preg_replace('/\D+/', '', $detail['time'] ?? '');
//        $stamp = ($date && $time) ? ($date.$time) : now()->format('YmdHis');
//        $prefix= (isset($detail['type']) && $detail['type'] === '2') ? 'MEPS' : 'BMEPS';
//
//        $detail['transaction_id'] = sprintf(
//            '%s-%s-%s-%s',
//            $prefix,
//            $detail['reference'] ?? 'REF',
//            $stamp,
//            substr(sha1(($fileName ?? '').($detail['reference'] ?? '').$stamp), 0, 6)
//        );
//    }
//
//    private function splitPaidIntoBaseAndFine(float $expectedBase, float $expectedFine, float $paid): array
//    {
//        $tol = 0.01;
//
//        // Pago menor/igual à base -> tudo em base, multa 0
//        if ($paid <= $expectedBase + $tol) {
//            return [round($paid, 2), 0.0];
//        }
//
//        // Pago acima da base -> delta vira multa (mesmo que expectedFine=0)
//        $basePaid = round($expectedBase, 2);
//        $finePaid = round(max(0.0, $paid - $expectedBase), 2);
//
//        // Opcional: limitar multa ao esperado, se houver
//        if ($expectedFine > 0 && $finePaid > $expectedFine + $tol) {
//            $finePaid = round($expectedFine, 2);
//            $basePaid = round($paid - $finePaid, 2);
//        }
//
//        return [$basePaid, $finePaid];
//    }
//
//    private function isAlreadyProcessed(?string $transactionId): bool
//    {
//        if (empty($transactionId)) return false;
//
//        if (in_array($transactionId, $this->processedReferences, true)) {
//            return true;
//        }
//
//        $exists = DB::table('payment_reconciliations')
//            ->where('transaction_id', $transactionId)
//            ->exists();
//
//        return $exists;
//    }
//
//    private function markAsProcessed(array $detail, float $amount): void
//    {
//        try {
//            DB::table('payment_reconciliations')->insert([
//                'transaction_id'    => $detail['transaction_id'] ?? null,
//                'reference'         => $detail['reference'] ?? null,
//                'amount'            => $amount,
//                'payment_date'      => Carbon::parse(($detail['date'] ?? date('Y-m-d')) . ' ' . ($detail['time'] ?? '00:00:00')),
//                'terminal_id'       => $detail['terminal_id'] ?? null,
//                'terminal_location' => $detail['terminal_location'] ?? null,
//                'processed_at'      => now(),
//                'created_at'        => now(),
//                'updated_at'        => now(),
//            ]);
//            Log::info('payment_reconciliations row inserted', [
//                'transaction_id' => $detail['transaction_id'] ?? null,
//                'reference'      => $detail['reference'] ?? null,
//                'amount'         => $amount,
//            ]);
//        } catch (\Throwable $e) {
//            Log::error('payment_reconciliations insert failed', [
//                'error'          => $e->getMessage(),
//                'transaction_id' => $detail['transaction_id'] ?? null,
//                'reference'      => $detail['reference'] ?? null,
//                'amount'         => $amount,
//            ]);
//        }
//    }
//
//    private function logUnmatchedPayment(array $detail): void
//    {
//        try {
//            DB::table('unmatched_payments')->insert([
//                'transaction_id'    => $detail['transaction_id'] ?? null,
//                'reference'         => $detail['reference'] ?? null,
//                'amount'            => (float)($detail['amount'] ?? 0),
//                'payment_date'      => Carbon::parse(($detail['date'] ?? date('Y-m-d')) . ' ' . ($detail['time'] ?? '00:00:00')),
//                'terminal_id'       => $detail['terminal_id'] ?? null,
//                'terminal_location' => $detail['terminal_location'] ?? null,
//                'status'            => 'unmatched',
//                'created_at'        => now()
//            ]);
//        } catch (\Throwable $e) {
//            Log::warning('Could not write into unmatched_payments (optional table).', ['error' => $e->getMessage()]);
//        }
//    }
//
//    private function saveFileInfo(string $filePath, array $parsedData): void
//    {
//        $totalAmount = array_reduce($parsedData['details'], fn($a, $d) => $a + (float)$d['amount'], 0.0);
//
//        // data do ficheiro: header.creation_date OU menor data de detalhe OU hoje
//        $fileDate = $parsedData['header']['creation_date'] ?? (
//            !empty($parsedData['details']) ? min(array_map(fn($d) => $d['date'] ?? date('Y-m-d'), $parsedData['details'])) : date('Y-m-d')
//        );
//
//        try {
//            DB::table('reconciliation_logs')->insert([
//                'file_name'     => basename($filePath),
//                'file_id'       => $parsedData['header']['file_id'] ?? null,
//                'file_date'     => $fileDate,
//                'total_records' => count($parsedData['details']),
//                'total_amount'  => $totalAmount,
//                'status'        => 'processing',
//                'started_at'    => now(),
//                'created_at'    => now()
//            ]);
//        } catch (\Throwable $e) {
//            Log::warning('Could not write into reconciliation_logs (optional table).', ['error' => $e->getMessage()]);
//        }
//    }
//
//    private function finalizeLog(string $fileName, string $status = 'completed'): void
//    {
//        try {
//            DB::table('reconciliation_logs')
//                ->where('file_name', $fileName)
//                ->orderByDesc('started_at')
//                ->limit(1)
//                ->update([
//                    'reconciled'   => $this->stats['reconciled'],
//                    'failed'       => $this->stats['failed'],
//                    'duplicates'   => $this->stats['duplicates'],
//                    'not_found'    => $this->stats['not_found'],
//                    'total_amount' => $this->stats['total_amount'],
//                    'status'       => $status,
//                    'completed_at' => now(),
//                    'updated_at'   => now(),
//                ]);
//        } catch (\Throwable $e) {
//            Log::warning('Could not finalize reconciliation_logs row.', ['file' => $fileName, 'error' => $e->getMessage()]);
//        }
//    }
//
//    // ======= Helpers =======
//    /** Aceita DDMMAAAA OU YYYYMMDD -> Y-m-d */
//    private function parseDateToYmd(string $dateStr): string
//    {
//        $s = preg_replace('/\D+/', '', $dateStr);
//        if (strlen($s) !== 8) return date('Y-m-d');
//
//        $yyyy = (int)substr($s, 0, 4);
//        $mm   = (int)substr($s, 4, 2);
//        $dd   = (int)substr($s, 6, 2);
//
//        if ($yyyy >= 1900 && $yyyy <= 2100) {
//            return sprintf('%04d-%02d-%02d', $yyyy, $mm, $dd);
//        }
//
//        $d = (int)substr($s, 0, 2);
//        $m = (int)substr($s, 2, 2);
//        $y = (int)substr($s, 4, 4);
//        return sprintf('%04d-%02d-%02d', $y, $m, $d);
//    }
//
//    /** hhmmss -> H:i:s */
//    private function parseTime(string $timeStr): string
//    {
//        $s = preg_replace('/\D+/', '', $timeStr);
//        if (strlen($s) !== 6) return '00:00:00';
//        $h = (int)substr($s, 0, 2);
//        $i = (int)substr($s, 2, 2);
//        $u = (int)substr($s, 4, 2);
//        return sprintf('%02d:%02d:%02d', $h, $i, $u);
//    }
//
//    /** inteiro (2 casas) -> float */
//    private function parseAmount(string $amountStr): float
//    {
//        $n = (int)preg_replace('/\D+/', '', $amountStr);
//        return $n / 100.0;
//    }
//
//    /**
//     * Normaliza referência:
//     * - remove sufixo "00" (MEPS observado)
//     * - mantém só dígitos
//     * - garante exatamente 11 dígitos
//     */
//    private function normalizeReference(?string $ref): ?string
//    {
//        if ($ref === null) return null;
//
//        $ref = preg_replace('/\D+/', '', $ref);
//        if (substr($ref, -2) === '00') {
//            $ref = substr($ref, 0, -2);
//        }
//        if (strlen($ref) > 11) {
//            $ref = substr($ref, -11);
//        }
//        if (strlen($ref) < 11) {
//            $ref = str_pad($ref, 11, '0', STR_PAD_LEFT);
//        }
//        return $ref;
//    }
//
//    private function validateFileStructure(array $parsedData): bool
//    {
//        if (!$parsedData['header']) {
//            Log::warning('Missing header');
//        }
//        if (count($parsedData['details']) === 0) {
//            Log::warning('No detail records found');
//            return false;
//        }
//        return true;
//    }
//
//    private function validateTrailer(array $parsedData): bool
//    {
//        if (!$parsedData['trailer']) {
//            Log::warning('No trailer found');
//            return true;
//        }
//
//        $trailer = $parsedData['trailer'];
//        $details = $parsedData['details'];
//
//        $expected = count($details);
//        if ((int)$trailer['total_records'] !== $expected) {
//            Log::warning('Record count mismatch', [
//                'expected_detail_count' => $expected,
//                'trailer_total_records' => $trailer['total_records']
//            ]);
//        }
//
//        $sum = array_reduce($details, fn($a,$d)=>$a + (float)$d['amount'], 0.0);
//        if (abs($sum - (float)$trailer['total_amount']) > 0.01) {
//            Log::warning('Amount total mismatch', [
//                'calculated' => $sum,
//                'trailer'    => $trailer['total_amount']
//            ]);
//        }
//
//        return true;
//    }
//}


namespace App\Services;

use App\Models\PaymentReference;
use App\Models\User;
use App\Services\FeeCalculationService;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class PaymentReconciliationService
{
    const HEADER_TYPE  = '0';
    const DETAIL_BMEPS = '1';
    const DETAIL_MEPS  = '2';
    const TRAILER_TYPE = '9';

    protected $feeCalculationService;

    public function __construct(FeeCalculationService $feeCalculationService)
    {
        $this->feeCalculationService = $feeCalculationService;
    }

    // ======= Diretórios =======
    private function baseDir(): string      { return public_path('bci/bmeps'); }
    private function inDir(): string        { return $this->baseDir() . '/incoming'; }
    private function processedDir(): string { return $this->baseDir() . '/processed'; }
    private function errorDir(): string     { return $this->baseDir() . '/failed'; }
    private function reportsDir(): string   { return $this->baseDir() . '/reports'; }

    // ======= Estado =======
    private array $processedReferences = [];
    private array $errors = [];
    private array $stats = [
        'total_files'           => 0,
        'total_records'         => 0,
        'reconciled'            => 0,
        'failed'                => 0,
        'duplicates'            => 0,
        'not_found'             => 0,
        'total_amount'          => 0.0,
        'autoscaled_10x'        => 0,
        'autoscaled_div10'      => 0,
        'with_calculated_fine'  => 0,
        'total_fines_preserved' => 0.0,
        'with_unified_service'  => 0,
    ];

    // ======= Entrada principal =======
    public function processAllPendingFiles(): array
    {
        Log::info('Starting reconciliation process (BMEPS/MEPS) with unified fee calculation service');

        try {
            $files = $this->fetchLocalFiles();
            if (empty($files)) {
                Log::warning('No files found in incoming', ['dir' => $this->inDir()]);
                return $this->stats;
            }

            Log::info('Found files', ['count' => count($files), 'dir' => $this->inDir(), 'files' => array_map('basename', $files)]);

            foreach ($files as $file) {
                $this->processFile($file);
            }

            $this->generateReconciliationReport();
            Log::info('Reconciliation completed with unified service', $this->stats);

        } catch (Exception $e) {
            Log::error('Global reconciliation failure', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
        }

        return $this->stats;
    }

    // ======= FS utils (mantidos iguais) =======
    private function ensureDirs(): void
    {
        foreach ([$this->inDir(), $this->processedDir(), $this->errorDir(), $this->reportsDir()] as $dir) {
            if (!is_dir($dir)) {
                @mkdir($dir, 0775, true);
                Log::info('Ensured directory', ['path' => $dir]);
            }
        }
    }

    private function fetchLocalFiles(): array
    {
        $this->ensureDirs();
        $files = glob($this->inDir() . '/*.txt', GLOB_NOSORT) ?: [];
        $files = array_merge($files, glob($this->inDir() . '/*.TXT', GLOB_NOSORT) ?: []);
        return array_unique($files);
    }

    private function moveFile(string $file, bool $success): void
    {
        $destDir = $success ? $this->processedDir() : $this->errorDir();
        $this->ensureDirs();

        $basename = basename($file);
        $target   = $destDir . '/' . $basename;

        if (file_exists($target)) {
            $target = $destDir . '/' . now()->format('Ymd_His') . '_' . $basename;
        }

        if (@rename($file, $target)) {
            Log::info('File moved', ['from' => $file, 'to' => $target]);
        } elseif (@copy($file, $target)) {
            @unlink($file);
            Log::info('File copied+removed', ['from' => $file, 'to' => $target]);
        } else {
            Log::warning('Could not move file', ['file' => $file, 'to' => $target]);
        }
    }

    private function generateReconciliationReport(): string
    {
        $report = [
            'process_date' => now()->toDateTimeString(),
            'stats'        => $this->stats,
            'errors'       => $this->errors,
            'service_info' => [
                'unified_fee_calculation_service' => true,
                'preserves_fines_in_history' => true,
                'automatic_fine_calculation' => true
            ]
        ];

        $dir = $this->reportsDir();
        if (!file_exists($dir)) @mkdir($dir, 0775, true);

        $reportPath = $dir . '/rep_' . now()->format('Ymd_His') . '.json';
        @file_put_contents($reportPath, json_encode($report, JSON_PRETTY_PRINT));
        Log::info('Reconciliation report generated', ['path' => $reportPath]);

        return $reportPath;
    }

    // ======= Core por ficheiro =======
    private function processFile(string $file): void
    {
        $this->stats['total_files']++;
        Log::info("Processing file", ['file' => basename($file)]);

        try {
            $content = @file_get_contents($file);
            if ($content === false || $content === '') {
                throw new Exception("Empty or unreadable file");
            }

            $parsed = $this->parseFile($content);
            if (!$this->validateFileStructure($parsed)) {
                throw new Exception("Invalid file structure");
            }

            $this->saveFileInfo($file, $parsed);

            DB::beginTransaction();

            foreach ($parsed['details'] as $detail) {
                $this->processTransaction($detail, $file);
            }

            $this->validateTrailer($parsed);

            DB::commit();
            $this->finalizeLog(basename($file), 'completed');
            $this->moveFile($file, true);

        } catch (Exception $e) {
            DB::rollBack();

            Log::error('Failed to process file', ['file' => basename($file), 'error' => $e->getMessage()]);
            $this->errors[] = ['file' => basename($file), 'error' => $e->getMessage(), 'timestamp' => now()->toDateTimeString()];
            $this->finalizeLog(basename($file), 'failed');
            $this->moveFile($file, false);
        }
    }

    // ======= Negócio USANDO SERVICE UNIFICADO =======
    private function processTransaction(array $detail, string $file): void
    {
        if (empty($detail['reference'])) {
            Log::warning('Skipping detail without reference');
            return;
        }

        $this->stats['total_records']++;

        $reference = (string) $detail['reference'];
        $paidAmount = (float) $detail['amount'];
      
        $paymentReference = PaymentReference::where('reference_number', $reference)
            ->whereIn('status', ['pending', 'paid'])
            ->latest('updated_at')
            ->first();

        if (!$paymentReference) {
            $this->stats['not_found']++;
            Log::warning('Reference not found', ['reference' => $reference, 'file' => basename($file)]);
            $this->logUnmatchedPayment($detail);
            return;
        }

        if ($paymentReference->status === 'paid') {
            $this->stats['duplicates']++;
            Log::notice('Duplicate payment (already paid)', ['reference' => $reference]);
            return;
        }

        // Busca o estudante
        $student = User::find($paymentReference->student_id);
        if (!$student) {
            $this->stats['failed']++;
            Log::error('Student not found for payment reference', [
                'reference' => $reference,
                'student_id' => $paymentReference->student_id
            ]);
            return;
        }

        // ✅ USA O SERVICE UNIFICADO para calcular multas
        $paymentDate = Carbon::parse(($detail['date'] ?? date('Y-m-d')) . ' ' . ($detail['time'] ?? '00:00:00'));
        
        try {
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $student,
                $paymentReference->fee_month,
                $paymentReference->fee_year,
                $paymentDate,
                $paymentReference
            );

            $this->stats['with_unified_service']++;

            Log::info('Fee calculation completed via unified service', [
                'reference' => $reference,
                'student_id' => $student->id,
                'paid_amount' => $paidAmount,
                'calculated_base' => $calculation['base_amount'],
                'calculated_fine' => $calculation['fine_amount'],
                'calculated_total' => $calculation['total_amount'],
                'is_late_payment' => $calculation['is_late_payment']
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to calculate fee using unified service, using fallback', [
                'reference' => $reference,
                'error' => $e->getMessage()
            ]);

            // Fallback para valores da PaymentReference
            $calculation = [
                'base_amount' => $paymentReference->amount ?? $paidAmount,
                'fine_amount' => $paymentReference->fine_amount ?? 0,
                'total_amount' => ($paymentReference->amount ?? $paidAmount) + ($paymentReference->fine_amount ?? 0),
                'is_late_payment' => false
            ];
        }

        // Valida o montante pago com tolerância
        if (!$this->validatePaidAmount($calculation['total_amount'], $paidAmount)) {
            $this->stats['failed']++;
            Log::error('Payment amount validation failed', [
                'reference' => $reference,
                'paid_amount' => $paidAmount,
                'expected_total' => $calculation['total_amount'],
                'base_amount' => $calculation['base_amount'],
                'fine_amount' => $calculation['fine_amount']
            ]);
            return;
        }

        // Garantir transaction_id
        $this->ensureTransactionId($detail, basename($file));

        DB::transaction(function () use ($paymentReference, $detail, $paidAmount, $calculation, $paymentDate, $student) {
            // Atualiza a PaymentReference
            $paymentReference->status = 'paid';
            $paymentReference->paid_at = $paymentDate;
            $paymentReference->transaction_id = $detail['transaction_id'] ?? null;
            $paymentReference->terminal_id = $detail['terminal_id'] ?? null;
            
            // PRESERVA/atualiza valores de multa na PaymentReference para histórico
            if ($calculation['fine_amount'] > ($paymentReference->fine_amount ?? 0)) {
                $paymentReference->fine_amount = $calculation['fine_amount'];
                $this->stats['with_calculated_fine']++;
            }
            
            $paymentReference->save();

            // ✅ USA O SERVICE UNIFICADO para criar o pagamento preservando multas
            $paymentData = [
                'month' => $paymentReference->fee_month,
                'year' => $paymentReference->fee_year,
                'amount'=> (float) $calculation['total_amount'], // líquido (base + multa - desconto)
                'fine'  => (float) $calculation['fine_amount'],  // multa preservada
                'discount' => 0,
                'payment_mode' => 'Reference',
                'paymentMode' => 'Reference',
                'pay_type' => 'reference',
                'transaction_id' => $detail['transaction_id'] ?? null,
                'reference_number' => $detail['reference'] ?? null,
                'payment_date' => $paymentDate,
                'note' => 'Pagamento via reconciliação bancária (BMEPS/MEPS)'
            ];

            $feeAssign = $this->feeCalculationService->createFeePayment(
                $student,
                $paymentData,
                $calculation
            );
            
            $this->markAsProcessed($detail, $paidAmount);

            Log::info('Payment processed via unified service', [
                'reference' => $paymentReference->reference_number,
                'fee_assign_id' => $feeAssign->id,
                'student_id' => $student->id,
                'base_amount' => $feeAssign->amount,
                'fine_preserved' => $feeAssign->fine,
                'transaction_id' => $detail['transaction_id'] ?? null
            ]);
        });

        $this->stats['reconciled']++;
        $this->stats['total_amount'] += $paidAmount;
        $this->stats['total_fines_preserved'] += $calculation['fine_amount'];

        Log::info('Payment reconciled successfully with unified service', [
            'reference' => $reference,
            'student_id' => $student->id,
            'paid_amount' => $paidAmount,
            'base_amount' => $calculation['base_amount'],
            'fine_amount_preserved' => $calculation['fine_amount'],
            'is_late_payment' => $calculation['is_late_payment']
        ]);
    }

    // ======= Validação melhorada =======
    private function validatePaidAmount(float $expectedTotal, float $paidAmount): bool
    {
        $tolerance = (float) config('bmeps.amount_tolerance', 1.00);
        
        Log::info('Paid amount validation', [
            'expected_total' => $expectedTotal,
            'paid_amount' => $paidAmount,
            'tolerance' => $tolerance,
            'difference' => abs($expectedTotal - $paidAmount)
        ]);

        // Verifica se está dentro da tolerância
        if (abs($expectedTotal - $paidAmount) <= $tolerance) {
            return true;
        }

        // Tenta correção automática (x10 ou /10)
        $x10 = $paidAmount * 10;
        $div10 = $paidAmount / 10;

        if (abs($expectedTotal - $x10) <= $tolerance) {
            Log::warning('Paid amount auto-corrected x10', [
                'from' => $paidAmount,
                'to' => $x10,
                'expected' => $expectedTotal
            ]);
            $this->stats['autoscaled_10x']++;
            return true;
        }

        if (abs($expectedTotal - $div10) <= $tolerance) {
            Log::warning('Paid amount auto-corrected /10', [
                'from' => $paidAmount,
                'to' => $div10,
                'expected' => $expectedTotal
            ]);
            $this->stats['autoscaled_div10']++;
            return true;
        }

        Log::warning('Paid amount validation failed - exceeds tolerance', [
            'expected_total' => $expectedTotal,
            'paid_amount' => $paidAmount,
            'difference' => abs($expectedTotal - $paidAmount),
            'tolerance' => $tolerance
        ]);

        return false;
    }

    // ======= Métodos auxiliares mantidos =======
    private function ensureTransactionId(array &$detail, string $fileName = null): void
    {
        if (!empty($detail['transaction_id'])) return;

        $date  = preg_replace('/\D+/', '', $detail['date'] ?? '');
        $time  = preg_replace('/\D+/', '', $detail['time'] ?? '');
        $stamp = ($date && $time) ? ($date.$time) : now()->format('YmdHis');
        $prefix= (isset($detail['type']) && $detail['type'] === '2') ? 'MEPS' : 'BMEPS';

        $detail['transaction_id'] = sprintf(
            '%s-%s-%s-%s',
            $prefix,
            $detail['reference'] ?? 'REF',
            $stamp,
            substr(sha1(($fileName ?? '').($detail['reference'] ?? '').$stamp), 0, 6)
        );
    }

    private function markAsProcessed(array $detail, float $amount): void
    {
        try {
            DB::table('payment_reconciliations')->insert([
                'transaction_id'    => $detail['transaction_id'] ?? null,
                'reference'         => $detail['reference'] ?? null,
                'amount'            => $amount,
                'payment_date'      => Carbon::parse(($detail['date'] ?? date('Y-m-d')) . ' ' . ($detail['time'] ?? '00:00:00')),
                'terminal_id'       => $detail['terminal_id'] ?? null,
                'terminal_location' => $detail['terminal_location'] ?? null,
                'processed_at'      => now(),
                'created_at'        => now(),
                'updated_at'        => now(),
            ]);
        } catch (\Throwable $e) {
            Log::error('payment_reconciliations insert failed', [
                'error'          => $e->getMessage(),
                'transaction_id' => $detail['transaction_id'] ?? null,
                'reference'      => $detail['reference'] ?? null,
                'amount'         => $amount,
            ]);
        }
    }

    private function logUnmatchedPayment(array $detail): void
    {
        try {
            DB::table('unmatched_payments')->insert([
                'transaction_id'    => $detail['transaction_id'] ?? null,
                'reference'         => $detail['reference'] ?? null,
                'amount'            => (float)($detail['amount'] ?? 0),
                'payment_date'      => Carbon::parse(($detail['date'] ?? date('Y-m-d')) . ' ' . ($detail['time'] ?? '00:00:00')),
                'terminal_id'       => $detail['terminal_id'] ?? null,
                'terminal_location' => $detail['terminal_location'] ?? null,
                'status'            => 'unmatched',
                'created_at'        => now()
            ]);
        } catch (\Throwable $e) {
            Log::warning('Could not write into unmatched_payments (optional table).', ['error' => $e->getMessage()]);
        }
    }

    private function saveFileInfo(string $filePath, array $parsedData): void
    {
        $totalAmount = array_reduce($parsedData['details'], fn($a, $d) => $a + (float)$d['amount'], 0.0);

        $fileDate = $parsedData['header']['creation_date'] ?? (
            !empty($parsedData['details']) ? min(array_map(fn($d) => $d['date'] ?? date('Y-m-d'), $parsedData['details'])) : date('Y-m-d')
        );

        try {
            DB::table('reconciliation_logs')->insert([
                'file_name'     => basename($filePath),
                'file_id'       => $parsedData['header']['file_id'] ?? null,
                'file_date'     => $fileDate,
                'total_records' => count($parsedData['details']),
                'total_amount'  => $totalAmount,
                'status'        => 'processing',
                'started_at'    => now(),
                'created_at'    => now()
            ]);
        } catch (\Throwable $e) {
            Log::warning('Could not write into reconciliation_logs (optional table).', ['error' => $e->getMessage()]);
        }
    }

    private function finalizeLog(string $fileName, string $status = 'completed'): void
    {
        try {
            DB::table('reconciliation_logs')
                ->where('file_name', $fileName)
                ->orderByDesc('started_at')
                ->limit(1)
                ->update([
                    'reconciled'   => $this->stats['reconciled'],
                    'failed'       => $this->stats['failed'],
                    'duplicates'   => $this->stats['duplicates'],
                    'not_found'    => $this->stats['not_found'],
                    'total_amount' => $this->stats['total_amount'],
                    'status'       => $status,
                    'completed_at' => now(),
                    'updated_at'   => now(),
                ]);
        } catch (\Throwable $e) {
            Log::warning('Could not finalize reconciliation_logs row.', ['file' => $fileName, 'error' => $e->getMessage()]);
        }
    }

    // ======= Parsers mantidos iguais =======
    private function parseFile(string $content): array
    {
        $lines  = preg_split('/\r\n|\r|\n/', rtrim($content));
        $result = ['header' => null, 'details' => [], 'trailer' => null];

        // Detecta MEPS (header contém "MEPS" OU existe linha tipo '2')
        $isMEPS = false;
        if (!empty($lines)) {
            $first = $lines[0];
            if (stripos($first, 'MEPS') !== false) {
                $isMEPS = true;
            } else {
                foreach ($lines as $l) {
                    if (isset($l[0]) && $l[0] === self::DETAIL_MEPS) { $isMEPS = true; break; }
                }
            }
        }
        Log::info('Detected format', ['format' => $isMEPS ? 'MEPS' : 'BMEPS']);

        foreach ($lines as $i => $raw) {
            $line = rtrim($raw);
            if ($line === '') continue;

            $type = substr($line, 0, 1);

            if ($type === self::HEADER_TYPE) {
                $result['header'] = $isMEPS ? $this->parseHeaderMEPS($line) : $this->parseHeaderBMEPS($line);
                continue;
            }
            if ($type === self::TRAILER_TYPE) {
                $result['trailer'] = $isMEPS ? $this->parseTrailerMEPS($line) : $this->parseTrailerBMEPS($line);
                continue;
            }

            if ($isMEPS) {
                if ($type === self::DETAIL_MEPS) {
                    $d = $this->parseDetailMEPS($line);
                    if ($d) $result['details'][] = $d;
                }
            } else {
                if ($type === self::DETAIL_BMEPS) {
                    $d = $this->parseDetailBMEPS($line);
                    if ($d) $result['details'][] = $d;
                }
            }
        }

        // Se header sem data, usa a menor data dos detalhes
        if (($result['header']['creation_date'] ?? null) === null && !empty($result['details'])) {
            $minDate = min(array_map(fn($d) => $d['date'] ?? date('Y-m-d'), $result['details']));
            $result['header']['creation_date'] = $minDate;
        }

        // Auto-correção pelo trailer quando há 1 único detalhe
        if ($result['trailer'] && count($result['details']) === 1) {
            $detSum = (float) $result['details'][0]['amount'];
            $trlSum = (float) $result['trailer']['total_amount'];

            if (abs($detSum - $trlSum) > 0.009) {
                if (abs(($detSum * 10) - $trlSum) <= 0.009) {
                    Log::warning('Detail amount auto-corrected by trailer x10', [
                        'from' => $detSum, 'to' => $detSum * 10, 'trailer' => $trlSum
                    ]);
                    $result['details'][0]['amount'] = $detSum * 10;
                }
            }
        }

        return $result;
    }

    // ======= Parsers específicos (mantidos iguais) =======
    private function parseHeaderBMEPS(string $line): array
    {
        return [
            'type'          => '0',
            'format'        => 'BMEPS',
            'company_id'    => substr($line, 1, 3),
            'profile_id'    => substr($line, 4, 2),
            'creation_date' => $this->parseDateToYmd(substr($line, 6, 8)),
            'file_id'       => substr($line, 14, 5),
            'raw'           => $line,
        ];
    }

    private function parseDetailBMEPS(string $line): ?array
    {
        $reference  = $this->normalizeReference(substr($line, 1, 11) ?: substr($line, 72, 11));
        $amount     = $this->parseAmount(substr($line, 12, 16));
        $commission = $this->parseAmount(substr($line, 28, 16));
        $date       = $this->parseDateToYmd(substr($line, 44, 8));
        $time       = $this->parseTime(substr($line, 52, 6));
        $txnId      = trim(substr($line, 58, 7));
        $termId     = trim(substr($line, 65, 10));
        $location   = rtrim(substr($line, 75, 16) ?: '');

        if (!$reference) return null;

        return [
            'type'              => '1',
            'reference'         => $reference,
            'amount'            => $amount,
            'commission'        => $commission,
            'date'              => $date,
            'time'              => $time,
            'transaction_id'    => $txnId ?: null,
            'terminal_id'       => $termId ?: null,
            'terminal_location' => $location ?: null,
            'raw'               => $line,
        ];
    }

    private function parseTrailerBMEPS(string $line): array
    {
        return [
            'type'             => '9',
            'total_records'    => (int) substr($line, 1, 7),
            'total_amount'     => $this->parseAmount(substr($line, 8, 16)),
            'total_commission' => $this->parseAmount(substr($line, 24, 16)),
            'raw'              => $line,
        ];
    }

    private function parseHeaderMEPS(string $line): array
    {
        $date = null;
        if (preg_match('/(20\d{6})/', $line, $m)) {
            $date = $this->parseDateToYmd($m[1]);
        }

        return [
            'type'          => '0',
            'format'        => 'MEPS',
            'creation_date' => $date,
            'raw'           => $line,
        ];
    }

            private function parseDetailMEPS(string $line): ?array
            {
                $re = '/^2.*?'
                    . '(?P<date>20\d{6})'
                    . '(?P<time>\d{4}|\d{6})'
                    . '(?P<amount>\d{12})'
                    . '(?P<commission>\d{10,12})'
                    . '\s{0,3}'
                    . '0{10,}'
                    . '\s*'
                    . '(?P<ref>\d{11})'
                    . '00\s*'  // Fixed: added missing concatenation operator
                    . '/';

                $trimmed = rtrim($line);
                if (!preg_match($re, $trimmed, $m)) {
                    return null;
                }

                $reference  = $this->normalizeReference($m['ref']);
                $amount     = ((int)$m['amount']) / 10.0;  // MEPS usa divisão por 10, não por 100
                $commission = ((int)$m['commission']) / 100.0;
                $date       = $this->parseDateToYmd($m['date']);

                $timeRaw = $m['time'];
                $time = (strlen($timeRaw) === 4)
                    ? substr($timeRaw,0,2).':'.substr($timeRaw,2,2).':00'
                    : $this->parseTime($timeRaw);

                if (!$reference) return null;

                return [
                    'type'              => '2',
                    'reference'         => $reference,
                    'amount'            => $amount,
                    'commission'        => $commission,
                    'date'              => $date,
                    'time'              => $time,
                    'transaction_id'    => null,
                    'terminal_id'       => null,
                    'terminal_location' => null,
                    'raw'               => $line,
                ];
            }

    private function parseTrailerMEPS(string $line): array
    {
        $count     = (int) substr($line, 1, 8);
        $totalAmt  = (int) substr($line, 9, 16)  / 100.0;
        $totalComm = (int) substr($line, 25, 16) / 100.0;

        return [
            'type'             => '9',
            'total_records'    => $count,
            'total_amount'     => $totalAmt,
            'total_commission' => $totalComm,
            'raw'              => $line,
        ];
    }

    // ======= Helpers =======
    private function parseDateToYmd(string $dateStr): string
    {
        $s = preg_replace('/\D+/', '', $dateStr);
        if (strlen($s) !== 8) return date('Y-m-d');

        $yyyy = (int)substr($s, 0, 4);
        $mm   = (int)substr($s, 4, 2);
        $dd   = (int)substr($s, 6, 2);

        if ($yyyy >= 1900 && $yyyy <= 2100) {
            return sprintf('%04d-%02d-%02d', $yyyy, $mm, $dd);
        }

        $d = (int)substr($s, 0, 2);
        $m = (int)substr($s, 2, 2);
        $y = (int)substr($s, 4, 4);
        return sprintf('%04d-%02d-%02d', $y, $m, $d);
    }

    private function parseTime(string $timeStr): string
    {
        $s = preg_replace('/\D+/', '', $timeStr);
        if (strlen($s) !== 6) return '00:00:00';
        $h = (int)substr($s, 0, 2);
        $i = (int)substr($s, 2, 2);
        $u = (int)substr($s, 4, 2);
        return sprintf('%02d:%02d:%02d', $h, $i, $u);
    }

    private function parseAmount(string $amountStr): float
    {
        $n = (int)preg_replace('/\D+/', '', $amountStr);
        return $n / 100.0;
    }

    private function normalizeReference(?string $ref): ?string
    {
        if ($ref === null) return null;

        $ref = preg_replace('/\D+/', '', $ref);
        if (substr($ref, -2) === '00') {
            $ref = substr($ref, 0, -2);
        }
        if (strlen($ref) > 11) {
            $ref = substr($ref, -11);
        }
        if (strlen($ref) < 11) {
            $ref = str_pad($ref, 11, '0', STR_PAD_LEFT);
        }
        return $ref;
    }

    private function validateFileStructure(array $parsedData): bool
    {
        if (!$parsedData['header']) {
            Log::warning('Missing header');
        }
        if (count($parsedData['details']) === 0) {
            Log::warning('No detail records found');
            return false;
        }
        return true;
    }

    private function validateTrailer(array $parsedData): bool
    {
        if (!$parsedData['trailer']) {
            Log::warning('No trailer found');
            return true;
        }

        $trailer = $parsedData['trailer'];
        $details = $parsedData['details'];

        $expected = count($details);
        if ((int)$trailer['total_records'] !== $expected) {
            Log::warning('Record count mismatch', [
                'expected_detail_count' => $expected,
                'trailer_total_records' => $trailer['total_records']
            ]);
        }

        $sum = array_reduce($details, fn($a,$d)=>$a + (float)$d['amount'], 0.0);
        if (abs($sum - (float)$trailer['total_amount']) > 0.01) {
            Log::warning('Amount total mismatch', [
                'calculated' => $sum,
                'trailer'    => $trailer['total_amount']
            ]);
        }

        return true;
    }
}
