<?php

namespace App\Http\Livewire\Admin\Fee;

use App\Models\Admin;
use App\Models\FeeGroup;
use App\Models\FeeType;
use App\Models\FeeMaster1;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;

class EditFeeMaster extends Component
{
    use FindGuard, LivewireAlert;

    public $fees_group, $fees_type, $due_date, $amount, $fine_type, $percentage, $fixed_amount;
    public $user, $model;
    public $validatedData;
    public $updateMode = false;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function mount($id)
    {
        $this->model = FeeMaster1::findOrFail($id);
        $this->fees_group = $this->model->fee_group_id; 
        $this->fees_type = $this->model->fee_type_id; 
        // dd($this->fees_group, $this->fees_type);
        $this->due_date = $this->model->due_date;
        $this->amount = $this->model->amount;
        $this->fine_type = $this->model->fine_type;
        $this->percentage = $this->model->percentage;
        $this->fixed_amount = $this->model->fixed_amount;
    }

    // Define validation rules
    protected array $rules = [
        'fees_group' => ['required'], // Ensure it exists in fee_groups table
        'fees_type' => ['required'], // Ensure it exists in fee_types table
        'due_date' => ['required', 'date'],
        'amount' => ['required', 'numeric', 'min:0'],
        'fine_type' => ['nullable', 'in:none,percentage,amount'],
        'percentage' => ['nullable', 'numeric', 'min:0', 'max:100'],
        'fixed_amount' => ['nullable', 'numeric', 'min:0'],
    ];

    // Validate on field update
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    // Update fee master record
    public function updateFeeMaster()
    {
        $this->validatedData = $this->validate();
        // dd($this->validatedData);
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        $this->model->fill([
            'fee_group_id' => $this->fees_group,
            'fee_type_id' => $this->fees_type,
            'due_date' => $this->due_date,
            'amount' => $this->amount,
            'fine_type' => $this->fine_type,
            'percentage' => $this->fine_type === 'percentage' ? $this->percentage : null,
            'fixed_amount' => $this->fine_type === 'amount' ? $this->fixed_amount : null,
        ]);
        // $this->model->fill($this->validatedData);
        // dd($this->model->toArray()); 
        if ($this->model->isClean()) {
            $this->alert('warning', 'Pelo menos um valor deve ser alterado.');
            return redirect()->back();
        }
        $this->model->save();
        Note::createSystemNotification(Admin::class, 'Detalhes', 'Plano de taxas atualizado com sucesso.');
        $this->emit('noteAdded');
        $this->alert('success', 'Plano de taxas atualizado com sucesso.');
        $this->reset();
        return redirect()->route('admin.list.feemaster'); // Redirect to FeeMaster list
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    // Fetch FeeGroups and FeeTypes for the form
    public function render()
    {
        return view('livewire.admin.fee.edit-fee-master', [
            'FeeGroups' => FeeGroup::all(), // Fetch FeeGroups
            'FeeTypes' => FeeType::all(),   // Fetch FeeTypes
        ]);
    }
}
