<?php

namespace App\Http\Livewire\User\Books;

use App\Models\Admin;
use App\Models\Subject;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;
use App\Http\Controllers\SystemController;
use App\Models\Book;
use App\Models\BookStatus;
use App\Models\BookTitle;
use App\Models\ReadingStatus;

class UserShowBookChapterDetail extends Component
{

    use FindGuard, LivewireAlert;

    public $title, $photo, $next, $book_id, $description, $user, $model;
    public $validatedData;
    public $updateMode = false;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function mount($slug)
    {
        $this->model = BookTitle::query()->where('slug', $slug)->first();
        $this->title = $this->model->title;
        $this->book_id = $this->model->book_id;
        $this->description = $this->model->description;
    }

    protected array $rules = [
        'title' => ['required', 'string', 'max:60', 'min:35'],
        'description' => ['required', 'string'],
        'book_id' => ['required', 'string', 'max:60', 'min:20'],
    ];

    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function submit()
    {
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        //get the next read_no
        $this->user = $this->findGuardType()->user();
        $nextstep = BookTitle::query()->where('read_no', $this->model->read_no + 1)->first();
        $lastBiggestId = BookTitle::max('read_no');
        if ($lastBiggestId == $this->model->read_no) {
            $bookStatus = BookStatus::updateOrCreate(
                [

                    'user_id' => $this->user->id,
                    'book_id' => $this->model->book_id,
                ],
                [
                    'is_completed' => true,
                    'is_preferred' => true,
                ]
            );
            $this->alert('warning', 'Você concluiu de ler o livro com sucesso.' . $this->model->book->title);
            return back();
        }
        // Find or create a reading status record for the user and chapter
        $readingStatus = ReadingStatus::Create(
            [
                'user_id' => $this->user->id,
                'book_id' => $this->model->book_id,
                'book_title_id' => $this->nextstep->id,
            ],
        );
        $bookStatus = BookStatus::updateOrCreate(
            [

                'user_id' => $this->user->id,
                'book_id' => $this->model->book_id,
            ],
            [
                'is_completed' => true,
                'is_preferred' => true,
            ]
        );
        Note::createSystemNotification(User::class, 'Detalhes', 'Aberto com sucesso' . $nextstep->title);
        $this->emit('noteAdded');
        $this->alert('success', 'Atualizado com sucesso ' . $nextstep->title);
        return redirect()->route('user.show.chapter.info', $nextstep->slug);
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }
    public function render()
    {
        return view('livewire.user.books.user-show-book-chapter-detail', [
            'books' => Book::query()->orderBy('title')->get(),
        ]);
    }
}
