<?php

namespace App\Console\Commands;

use App\Models\Classroom;
use App\Models\Subject;
use App\Services\ExcelMarksImportService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;

class ImportExcelMarks extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'marks:import-excel
                            {file : The path to the Excel file}
                            {--year= : Academic year}
                            {--trimester= : Trimester (1, 2, or 3)}
                            {--classroom= : Classroom ID or name}
                            {--subject= : Subject ID or name}
                            {--clean : Clean existing marks before import}
                            {--dry-run : Run without making changes}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import exam marks from Excel file';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle(ExcelMarksImportService $service): int
    {
        $filePath = $this->argument('file');

        // Validate file exists
        if (!file_exists($filePath)) {
            $this->error("File not found: {$filePath}");
            return self::FAILURE;
        }

        // Get options
        $year = $this->option('year') ?: date('Y');
        $trimester = $this->option('trimester');
        $classroomIdentifier = $this->option('classroom');
        $subjectIdentifier = $this->option('subject');
        $cleanExisting = $this->option('clean');
        $dryRun = $this->option('dry-run');

        // Interactive mode if missing options
        if (!$trimester) {
            $trimester = $this->choice('Select trimester:', ['1', '2', '3']);
        }

        if (!$classroomIdentifier) {
            $classrooms = Classroom::orderBy('class')->orderBy('name')->get();
            $choices = $classrooms->map(fn($c) => "{$c->id} - {$c->class}ª Classe - {$c->name}")->toArray();
            $selected = $this->choice('Select classroom:', $choices);
            $classroomIdentifier = explode(' - ', $selected)[0];
        }

        if (!$subjectIdentifier) {
            $subjects = Subject::orderBy('name')->get();
            $choices = $subjects->map(fn($s) => "{$s->id} - {$s->name}")->toArray();
            $selected = $this->choice('Select subject:', $choices);
            $subjectIdentifier = explode(' - ', $selected)[0];
        }

        // Resolve classroom and subject
        $classroom = $this->resolveClassroom($classroomIdentifier);
        $subject = $this->resolveSubject($subjectIdentifier);

        if (!$classroom) {
            $this->error("Classroom not found: {$classroomIdentifier}");
            return self::FAILURE;
        }

        if (!$subject) {
            $this->error("Subject not found: {$subjectIdentifier}");
            return self::FAILURE;
        }

        // Validate Excel file
        $this->info('Validating Excel file...');
        $validation = $service->validateExcelFile($filePath);

        if (!$validation['valid']) {
            $this->error("Excel validation failed: {$validation['message']}");
            return self::FAILURE;
        }

        $this->info('✓ Excel file is valid');

        // Display import summary
        $this->info('Import Summary:');
        $this->table(
            ['Parameter', 'Value'],
            [
                ['File', $filePath],
                ['Year', $year],
                ['Trimester', $trimester],
                ['Classroom', "{$classroom->class}ª Classe - {$classroom->name}"],
                ['Subject', $subject->name],
                ['Clean Existing', $cleanExisting ? 'Yes' : 'No'],
                ['Dry Run', $dryRun ? 'Yes' : 'No'],
            ]
        );

        if (!$this->confirm('Do you want to proceed with the import?')) {
            $this->info('Import cancelled');
            return self::SUCCESS;
        }

        // Prepare import data
        $importData = [
            'year' => $year,
            'trimester_id' => $trimester,
            'classroom_id' => $classroom->id,
            'subject_id' => $subject->id,
            'clean_existing' => $cleanExisting && !$dryRun,
        ];

        // Perform import
        $this->info('Starting import...');
        $progressBar = $this->output->createProgressBar();
        $progressBar->start();

        if ($dryRun) {
            $this->info('DRY RUN - No changes will be made');
            // Simulate the import process for dry run
            $result = $this->simulateImport($service, $filePath, $importData);
        } else {
            $result = $service->importMarks($filePath, $importData);
        }

        $progressBar->finish();
        $this->newLine();

        // Display results
        if ($result['success']) {
            $stats = $result['data'];

            $this->info('✓ Import completed successfully');
            $this->table(
                ['Metric', 'Count'],
                [
                    ['Total Rows', $stats['total_rows']],
                    ['Successful', $stats['successful']],
                    ['Failed', $stats['failed']],
                ]
            );

            if (!empty($stats['errors'])) {
                $this->warn('Errors encountered:');
                foreach (array_slice($stats['errors'], 0, 10) as $error) {
                    $this->line("• {$error}");
                }
                if (count($stats['errors']) > 10) {
                    $this->line("• ... and " . (count($stats['errors']) - 10) . " more errors");
                }
            }
        } else {
            $this->error("Import failed: {$result['message']}");
            return self::FAILURE;
        }

        return self::SUCCESS;
    }

    /**
     * Resolve classroom by ID or name
     */
    private function resolveClassroom(string $identifier): ?Classroom
    {
        // Try by ID first
        $classroom = Classroom::find($identifier);
        if ($classroom) {
            return $classroom;
        }

        // Try by name
        return Classroom::where('name', 'like', "%{$identifier}%")->first();
    }

    /**
     * Resolve subject by ID or name
     */
    private function resolveSubject(string $identifier): ?Subject
    {
        // Try by ID first
        $subject = Subject::find($identifier);
        if ($subject) {
            return $subject;
        }

        // Try by name
        return Subject::where('name', 'like', "%{$identifier}%")->first();
    }

    /**
     * Simulate import for dry run
     */
    private function simulateImport(ExcelMarksImportService $service, string $filePath, array $importData): array
    {
        // This would parse the Excel file and show what would happen
        // without actually making database changes
        try {
            $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($filePath);
            $worksheet = $spreadsheet->getActiveSheet();
            $highestRow = $worksheet->getHighestRow();

            return [
                'success' => true,
                'data' => [
                    'total_rows' => $highestRow - 1, // Excluding header
                    'successful' => $highestRow - 1,
                    'failed' => 0,
                    'errors' => []
                ]
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
}
