<?php

namespace Chatify\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Response;
use App\Models\User;
use App\Models\Admin;
use App\Models\ChMessage as Message;
use App\Models\ChFavorite as Favorite;
use Chatify\Facades\ChatifyMessenger as Chatify;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Request as FacadesRequest;
use Illuminate\Support\Str;


class MessagesController extends Controller
{
    protected $perPage = 30;

    /**
     * Authenticate the connection for pusher
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getCurrentUser()
    {
        if (Auth::guard('admin')->check()) {
            $currentUser = Auth::guard('admin')->user();
        } else {
            $currentUser = Auth::user();
        }
        return $currentUser;
    }

    public function pusherAuth(Request $request): mixed
    {
        // Check if the authenticated user is an admin


        return Chatify::pusherAuth(
            $request->user(),
            $this->getCurrentUser(),//maybe wrong order? 

            $request['channel_name'],
            $request['socket_id']
        );
    }


    /**
     * Returning the view of the app with the required data.
     *
     * @param int $id
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function index($id = null): mixed
    {
        $messenger_color = $this->getCurrentUser()->messenger_color;
        return view('Chatify::pages.app', [
            'id' => $id ?? 0,
            'messengerColor' => $messenger_color ? $messenger_color : Chatify::getFallbackColor(),
            'dark_mode' => $this->getCurrentUser()->dark_mode < 1 ? 'light' : 'dark',
        ]);
    }


    /**
     * Fetch data (user, favorite.. etc).
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function idFetchData(Request $request): mixed
    {
        $userId = $request['id'];

        // Check if the user exists in the user table
        $fetchUser = User::find($userId);

        // Check if the user exists in the admin table
        $fetchAdmin = Admin::find($userId);

        if ($fetchUser) {
            $favorite = Chatify::inFavorite($userId);
            $fetch = $fetchUser;
        } elseif ($fetchAdmin) {
            // If the user exists in the admin table, treat them as an admin
            $favorite = null; // No favorite functionality for admins
            $fetch = $fetchAdmin;
        } else {
            return Response::json([
                'error' => 'User not found!',
            ], 404);
        }

        // Get the avatar
        $userAvatar = Chatify::getUserWithAvatar($fetch)->avatar;

        return Response::json([
            'favorite' => $favorite,
            'fetch' => $fetch,
            'user_avatar' => $userAvatar,
        ]);
    }


    /**
     * This method to make a links for the attachments
     * to be downloadable.
     *
     * @param string $fileName
     * @return \Symfony\Component\HttpFoundation\StreamedResponse|void
     */
    public function download($fileName): mixed
    {
        $filePath = config('chatify.attachments.folder') . '/' . $fileName;
        if (Chatify::storage()->exists($filePath)) {
            return Chatify::storage()->download($filePath);
        }
        return abort(404, "Sorry, File does not exist in our server or may have been deleted!");
    }


    /**
     * Send a message to database
     *
     * @param Request $request
     * @return JSON response
     */
    public function send(Request $request)
    {
        // Default variables
        $error = (object) [
            'status' => 0,
            'message' => null
        ];
        $attachment = null;
        $attachment_title = null;

        // If there is an attachment [file]
        if ($request->hasFile('file')) {
            // Allowed extensions
            $allowed_images = Chatify::getAllowedImages();
            $allowed_files = Chatify::getAllowedFiles();
            $allowed = array_merge($allowed_images, $allowed_files);

            $file = $request->file('file');
            // Check file size
            if ($file->getSize() < Chatify::getMaxUploadSize()) {
                if (in_array(strtolower(string: $file->extension()), haystack: $allowed)) {
                    // Get attachment name
                    $attachment_title = $file->getClientOriginalName();
                    // Upload attachment and store the new name
                    $attachment = Str::uuid() . "." . $file->extension();
                    $file->storeAs(config('chatify.attachments.folder'), $attachment, config('chatify.storage_disk_name'));
                } else {
                    $error->status = 1;
                    $error->message = "File extension not allowed!";
                }
            } else {
                $error->status = 1;
                $error->message = "File size you are trying to upload is too large!";
            }
        }

        if (!$error->status) {
            // Determine the type of authenticated user
            if (Str::startsWith(request()->url(), url('admin'))) {
                $from_id = Auth::guard('admin')->id();
            } else {
                $from_id = Auth::id();
            }

            // send to database
            $message = Chatify::newMessage([
                'type' => $request['type'],
                'from_id' => Auth::user()->id,
                'to_channel_id' => $request['channel_id'],
                'body' => trim($request['message']),  // Removed htmlentities
                'attachment' => ($attachment) ? json_encode((object) [
                    'new_name' => $attachment,
                    'old_name' => trim($attachment_title),  // Removed htmlentities
                ]) : null,
            ]);

            // Fetch message to send it with the response
            $messageData = Chatify::parseMessage($message);

            // Send to user using Pusher
            if ($from_id != $request['id']) {
                Chatify::push("private-chatify." . $request['id'], 'messaging', [
                    'from_id' => $from_id,
                    'to_id' => $request['id'],
                    'message' => $messageData
                ]);
            }

             // Send notification after successfully saving the message
             $this->sendNotification($request['channel_id'], $message, $attachment);


        }

        // Send the response
        return response()->json([
            'status' => '200',
            'error' => $error,
            'message' => $messageData ?? [],
            'tempID' => $request['temporaryMsgId'],
        ]);
    }



    protected function sendNotification($channelId, $message, $attachment)
    {
        try {
            // Retrieve user IDs associated with the channel from the ch_channel_user table
            $userIds = DB::table('ch_channel_user')
                ->where('channel_id', $channelId)
                ->pluck('user_id');
    
            if ($userIds->isEmpty()) {
                throw new \Exception("No users found for channel_id: " . $channelId);
            }
    
            $firebaseCredentials = [
                "type" => "service_account",
                "project_id" => "copmoz-3x",
                "private_key_id" => "bb9e045f0e2345c069b299a14636203509899ef8",
                "private_key" => "-----BEGIN PRIVATE KEY-----\nMIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDYfKrqJJyR6/8w\nP0NI75wA0OMzBMq53JD0GX/2G488VD2gE+Wbxt90e/6PUGdrOEwo7C+ilCNKsoX7\nzyy6l6kOt6MgVjO39bc7/qwaKYcA05/48grMrLMUxb6SNAMmKmj3CTQ9dU/zAmqA\n96HleqAkHt22wRSQ8hy4qg44CC1f8faBJxGb68YeMm9TmoYcdUhc7QT13JoMRZjl\nbi5gNDVIFWDyUhXeVpvw13Z3Sfo+EgkiT2Tb2Nr8//rptIabmrhzUsAFTEKnHblh\nY+1nsxBBwImu1NRtr5aQUCNLUZXcbWpuoym6EZtCLe3u6/mL3wI05407q1YqpUOq\nppfDwuHdAgMBAAECggEAMuv0Kfotp/dKgddHpObTm6+G2a/VXa2fnQNIgaPO5+Jq\n0nYLqr43PoeSA7TmdqZ556BSvhu5R29Bq876FvPojLEuWkNci/dsv9xyL/83pb6h\noOWSG729q/OtlWdNnFMaeI0+L/yhrygQ1EArYTacZ5KpFT1KcKO6lMdPa+ekgK4T\nPuwGVD/3VDFG9AWCEm+v3w6bQAhpKbdZiBPpGvTrfFHYov97/Yzv/iC3599VnW+i\no3GweFGaqsOn4PxuyIyYujzgoIc6kBrivUV+ddxZqLufrsQL1uE78SuWjhRNS8/k\n5rUD3Qvaj3BOQzkqCnttclfPZYSmbjMtVyLOMjMcQQKBgQDkE1SnrhY/dZF5QCFg\nIMaf7wkPLLCSStugOeYMGx2oeD0MxaY9L9N1ffRLekngffzANrO/ybE7AlRzgBJB\n2NYFd7fueihmrWuTpMafkBivG+mf+HdTjkxF0aQfYUL+MMW7viBBzGDbwckRSTX5\n5pSqzEHmRvctiLTfHqqgUnDeIQKBgQDy/hzbdteqvgqRok7KoXy3ciij4SQbDOQZ\n19KcmMTFgMiT/xm/3uni924Qw82VVSYQjDbAoSltTl7cpvtkXzSB2LIg00YEhfcE\nj/atYfswDIBYBAGCUZDalOn6s/8TWuTGVfCkQVwHA2Xa3rKqIrI1gHTuu5N4J/P4\niCmL3Vd0PQKBgQC1mirC5Rvz5ZIywHySxSZCfJtzCLuDkLV1zAQ/yWuBFFRQEiS+\n/ZDbMbJOUw32AXs6NZREdr125fEGkoh3A8fOTLgY3A3FS/qncgFxVdRBwfDxHm8t\nCdXzleyfy9sC+STIy6d1nN3WvWNzLx8aX54qrT8fs4vnJr4WFp330Azs4QKBgQCc\nnDj/Hb5sdmWbbW+425HlLfeRf+bkZE+TdG1yrmeMH6+m6zCuYD6AIRbYSUp0J9gL\nrEMRrg1kPLGZJyo5i77svTw5OdIT0j/dueez4cWiNzx0/cf3NRjWOEoBYgdRczCl\nv/gD9XgZsQ/xm7ytOQWAxBUZVN38AEwW77NPSM43xQKBgCgCmFCO57u8Fc6d+4G9\nBy/Vfmy3tZpwGCTGqxkwhFvG+qXZTERJOwRg4jejIM/W6iQlOPmfsis2CWhCZvYL\nSimfvuVeWSVixdnsxVpjVyyT7ci9jMhPXeL7LtMCjNF4hB34qgejqpkwh62Zqg0h\nNuy5ZdoARi0VwY/spf6HK03b\n-----END PRIVATE KEY-----\n",
                "client_email" => "firebase-adminsdk-as4sm@copmoz-3x.iam.gserviceaccount.com",
                "client_id" => "112030584520787629755",
                "auth_uri" => "https://accounts.google.com/o/oauth2/auth",
                "token_uri" => "https://oauth2.googleapis.com/token",
                "auth_provider_x509_cert_url" => "https://www.googleapis.com/oauth2/v1/certs",
                "client_x509_cert_url" => "https://www.googleapis.com/robot/v1/metadata/x509/firebase-adminsdk-as4sm%40copmoz-3x.iam.gserviceaccount.com",
                "universe_domain" => "googleapis.com"
            ];
    
            $factory = (new Factory)->withServiceAccount($firebaseCredentials);
            $messaging = $factory->createMessaging();
            Log::info('Firebase Messaging initialized');
    
            // For each user in the channel, send a notification
            foreach ($userIds as $userId) {
                $user = User::where('id', $userId)->first();
                if (!$user) {
                    Log::error("User not found for ID: $userId");
                    continue;
                }
    
                // Get device tokens for the user
                $tokens = DeviceToken::where('user_id', $user->id)->pluck('token');
                if ($tokens->isEmpty()) {
                    Log::error("No device tokens found for user: " . $user->name);
                    continue;
                }
    
                Log::info('Device tokens retrieved for user: ' . $user->name, ['tokens' => $tokens]);
    
                // Send notification for each device token
                foreach ($tokens as $token) {
                    try {
                        $notification = Notification::fromArray([
                            'token' => $token,
                            'title' => "Nova mensagem de " . Auth::user()->name,
                            'body' => ($attachment) ? "Ficheiro anexado" : $message->body,
                        ]);
    
                        $firebaseMessage = CloudMessage::withTarget('token', $token)
                            ->withNotification($notification)
                            ->withHighestPossiblePriority()
                            ->withDefaultSounds();
    
                        $messaging->send($firebaseMessage);
                        Log::info('Notification sent to token: ' . $token);
    
                    } catch (\Kreait\Firebase\Exception\Messaging\InvalidMessage $e) {
                        Log::error("Invalid token: {$token}. Error: {$e->getMessage()}");
                        DeviceToken::where('token', $token)->delete();  // Remove invalid token
                    } catch (\Exception $e) {
                        Log::error("Error sending notification: " . $e->getMessage());
                    }
                }
            }
    
        } catch (\Exception $e) {
            Log::error('Error sending notifications: ' . $e->getMessage());
        }
    }

                





    /**
     * fetch [user/group] messages from database
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function fetch(Request $request)
    {
        $query = Chatify::fetchMessagesQuery($request['id'])->latest();
        $messages = $query->paginate($request->per_page ?? $this->perPage);
        $totalMessages = $messages->total();
        $lastPage = $messages->lastPage();
        $response = [
            'total' => $totalMessages,
            'last_page' => $lastPage,
            'last_message_id' => collect($messages->items())->last()->id ?? null,
            'messages' => '',
        ];
   
        // if there is no messages yet.
        if ($totalMessages < 1) {
            $response['messages'] = '<p class="message-hint center-el"><span>Diga \'Ola\' para iniciar conversa!</span></p>';
            return Response::json($response);
        }
        if (count($messages->items()) < 1) {
            $response['messages'] = '';
            return Response::json($response);
            
        }
        $allMessages = null;
        foreach ($messages->reverse() as $message) {
            $allMessages .= Chatify::messageCard(
                Chatify::parseMessage($message)
            );
        }
        $response['messages'] = $allMessages;
        return Response::json($response);
    }

    /**
     * Make messages as seen
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function seen(Request $request)
    {
        // make as seen
        $seen = Chatify::makeSeen($request['id']);
        // send the response
        return Response::json([
            'status' => $seen,
        ], 200);
    }

    /**
     * Get contacts list
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getContacts(Request $request)
    {
        // Get all users and admins that received/sent messages from/to [Auth user]
        $contacts = collect();

        // Get users
        $users = Message::join('users', function ($join) {
            $join->on('ch_messages.from_id', '=', 'users.id')
                ->orOn('ch_messages.to_id', '=', 'users.id');
        })
            ->where(function ($q) {
                $q->where('ch_messages.from_id', $this->getCurrentUser()->id)
                    ->orWhere('ch_messages.to_id', $this->getCurrentUser()->id);
            })
            ->where('users.id', '!=', $this->getCurrentUser()->id)
            ->select('users.*', DB::raw('MAX(ch_messages.created_at) max_created_at'))
            ->orderBy('max_created_at', 'desc')
            ->groupBy('users.id')
            ->get();

        $contacts = $contacts->merge($users);

        // Get admins
        $admins = Message::join('admins', function ($join) {
            $join->on('ch_messages.from_id', '=', 'admins.id')
                ->orOn('ch_messages.to_id', '=', 'admins.id');
        })
            ->where(function ($q) {
                $q->where('ch_messages.from_id', $this->getCurrentUser()->id)
                    ->orWhere('ch_messages.to_id', $this->getCurrentUser()->id);
            })
            ->where('admins.id', '!=', $this->getCurrentUser()->id)
            ->select('admins.*', DB::raw('MAX(ch_messages.created_at) max_created_at'))
            ->orderBy('max_created_at', 'desc')
            ->groupBy('admins.id')
            ->get();

        $contacts = $contacts->merge($admins);

        // Paginate
        $contacts = $contacts->unique('id')->sortByDesc('max_created_at')->paginate($request->per_page ?? $this->perPage);

        if ($contacts->count() > 0) {
            $contactsList = '';
            foreach ($contacts as $contact) {
                $contactsList .= Chatify::getContactItem($contact);
            }
        } else {
            $contactsList = '<p class="message-hint center-el"><span>Sua lista de contatos está vazia!</span></p>';
        }

        return Response::json([
            'contacts' => $contactsList,
            'total' => $contacts->total() ?? 0,
            'last_page' => $contacts->lastPage() ?? 1,
        ], 200);
    }


    // /**
    //  * Update user's list item data
    //  *
    //  * @param Request $request
    //  * @return JsonResponse
    //  */
    // public function updateContactItem(Request $request)
    // {
    //     // Get user data
    //     $user = User::where('id', $request['user_id'])->first();
    //     if(!$user){
    //         return Response::json([
    //             'message' => 'User not found!',
    //         ], 401);
    //     }
    //     $contactItem = Chatify::getContactItem($user);

    //     // send the response
    //     return Response::json([
    //         'contactItem' => $contactItem,
    //     ], 200);
    // }

    /**
     * Update user's list item data
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function updateContactItem(Request $request)
    {
        // Get user data (both users and admins)
        $user = User::where('id', $request['user_id'])->first();
        if (!$user) {
            $user = Admin::where('id', $request['user_id'])->first();
            if (!$user) {
                return Response::json([
                    'message' => 'User not found!',
                ], 404);
            }
        }

        // Get contact item
        $contactItem = Chatify::getContactItem($user);

        // Send the response
        return Response::json([
            'contactItem' => $contactItem,
        ], 200);
    }

    /**
     * Put a user in the favorites list
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function favorite(Request $request)
    {
        $userId = $request['user_id'];
        // check action [star/unstar]
        $favoriteStatus = Chatify::inFavorite($userId) ? 0 : 1;
        Chatify::makeInFavorite($userId, $favoriteStatus);

        // send the response
        return Response::json([
            'status' => @$favoriteStatus,
        ], 200);
    }

    /**
     * Get favorites list
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function getFavorites(Request $request)
    {
        $favoritesList = null;
        $favorites = Favorite::where('user_id', $this->getCurrentUser()->id);
        foreach ($favorites->get() as $favorite) {
            // get user data
            $user = User::where('id', $favorite->favorite_id)->first();
            $favoritesList .= view('Chatify::layouts.favorite', [
                'user' => $user,
            ]);
        }
        // send the response
        return Response::json([
            'count' => $favorites->count(),
            'favorites' => $favorites->count() > 0
                ? $favoritesList
                : 0,
        ], 200);
    }

    /**
     * Search in messenger
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function search(Request $request)
    {
        $getRecords = null;
        $input = trim(filter_var($request['input']));
        $records = User::where('id', '!=', $this->getCurrentUser()->id)
            ->where('name', 'LIKE', "%{$input}%")
            ->paginate($request->per_page ?? $this->perPage);

        $recordsteacher = Admin::where('id', '!=', $this->getCurrentUser()->id)
            ->where('name', 'LIKE', "%{$input}%")
            ->paginate($request->per_page ?? $this->perPage);
        foreach ($records->items() as $record) {
            $getRecords .= view('Chatify::layouts.listItem', [
                'get' => 'search_item',
                'user' => Chatify::getUserWithAvatar($record),
            ])->render();
        }

        foreach ($recordsteacher->items() as $record) {
            $getRecords .= view('Chatify::layouts.teacherlistItem', [
                'get' => 'search_item',
                'user' => Chatify::getUserWithAvatar($record),
            ])->render();
        }
        if ($records->total() < 1) {
            $getRecords = '<p class="message-hint center-el"><span>Nada para mostrar</span></p>';
        }
        // send the response
        return Response::json([
            'records' => $getRecords,
            'total' => $records->total(),
            'last_page' => $records->lastPage()
        ], 200);
    }

    /**
     * Get shared photos
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function sharedPhotos(Request $request)
    {
        $shared = Chatify::getSharedPhotos($request['user_id']);
        $sharedPhotos = null;

        // shared with its template
        for ($i = 0; $i < count($shared); $i++) {
            $sharedPhotos .= view('Chatify::layouts.listItem', [
                'get' => 'sharedPhoto',
                'image' => Chatify::getAttachmentUrl($shared[$i]),
            ])->render();
        }
        // send the response
        return Response::json([
            'shared' => count($shared) > 0 ? $sharedPhotos : '<p class="message-hint"><span>Nada partilhado</span></p>',
        ], 200);
    }

    /**
     * Delete conversation
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function deleteConversation(Request $request)
    {
        // delete
        $delete = Chatify::deleteConversation($request['id']);

        // send the response
        return Response::json([
            'deleted' => $delete ? 1 : 0,
        ], 200);
    }

    /**
     * Delete message
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function deleteMessage(Request $request)
    {
        // delete
        $delete = Chatify::deleteMessage($request['id']);

        // send the response
        return Response::json([
            'deleted' => $delete ? 1 : 0,
        ], 200);
    }

    public function updateSettings(Request $request)
    {
        $msg = null;
        $error = $success = 0;

        // dark mode
        if ($request['dark_mode']) {
            $request['dark_mode'] == "dark"
                ? User::where('id', $this->getCurrentUser()->id)->update(['dark_mode' => 1])  // Make Dark
                : User::where('id', $this->getCurrentUser()->id)->update(['dark_mode' => 0]); // Make Light
        }

        // If messenger color selected
        if ($request['messengerColor']) {
            $messenger_color = trim(filter_var($request['messengerColor']));
            User::where('id', $this->getCurrentUser()->id)
                ->update(['messenger_color' => $messenger_color]);
        }
        // if there is a [file]
        if ($request->hasFile('avatar')) {
            // allowed extensions
            $allowed_images = Chatify::getAllowedImages();

            $file = $request->file('avatar');
            // check file size
            if ($file->getSize() < Chatify::getMaxUploadSize()) {
                if (in_array(strtolower($file->extension()), $allowed_images)) {
                    // delete the older one
                    if ($this->getCurrentUser()->avatar != config('chatify.user_avatar.default')) {
                        $avatar = $this->getCurrentUser()->avatar;
                        if (Chatify::storage()->exists($avatar)) {
                            Chatify::storage()->delete($avatar);
                        }
                    }
                    // upload
                    $avatar = Str::uuid() . "." . $file->extension();
                    $update = User::where('id', $this->getCurrentUser()->id)->update(['avatar' => $avatar]);
                    $file->storeAs(config('chatify.user_avatar.folder'), $avatar, config('chatify.storage_disk_name'));
                    $success = $update ? 1 : 0;
                } else {
                    $msg = "File extension not allowed!";
                    $error = 1;
                }
            } else {
                $msg = "File size you are trying to upload is too large!";
                $error = 1;
            }
        }

        // send the response
        return Response::json([
            'status' => $success ? 1 : 0,
            'error' => $error ? 1 : 0,
            'message' => $error ? $msg : 0,
        ], 200);
    }

    /**
     * Set user's active status
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function setActiveStatus(Request $request)
    {
        $activeStatus = $request['status'] > 0 ? 1 : 0;
        $status = User::where('id', $this->getCurrentUser()->id)->update(['active_status' => $activeStatus]);
        return Response::json([
            'status' => $status,
        ], 200);
    }
}
