<?php

namespace App\Http\Livewire\Auth;

use App\Jobs\MailJob;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\URL;
use Illuminate\Validation\Rules\Password;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;

class Register extends Component
{
    use LivewireAlert;

    public $name;
    public $email;
    public $password;
    public $phone_number;
    public $password_confirmation;

    protected function rules(): array
    {
        return [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'unique:users', 'max:255'],
            'phone_number' => ['required', 'string', 'unique:users'],
            'password' => ['required', 'confirmed', 'same:password_confirmation', Password::min(8)
                ->mixedCase()
                ->letters()
                ->numbers()
                ->symbols()
            ],
        ];
    }

    protected $messages = [
        'email.email' => 'O formato do endereço de e-mail não é válido.',
    ];

    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function register()
    {
        $this->validate();
        try {
            $user = User::query()->create([
                'name' => $this->name,
                'phone_number' => $this->phone_number,
                'email' => $this->email,
                'password' => bcrypt($this->password)
            ]);

            dispatch((new MailJob(
                $user->name,
                $user->email,
                'Verificação de conta',
                'Clique no link abaixo para acessar sua conta. O link expirará após 60 minutos.',
                true,
                URL::temporarySignedRoute(
                    'verify', now()->addMinutes(30), ['slug' => $user->slug]
                ),
                '<<< VERIFICAR A CONTA >>>'
            )))->onQueue('emails')->delay(2);

            $this->alert(
                'success',
                'Um código de verificação foi enviado para seu endereço de e-mail.'
            );

            auth()->loginUsingId($user->id);
            return redirect()->route('user.home');
        } catch (Exception $exception) {
            $this->alert(
                'error',
                'Um erro ocorreu. Tente mais tarde'
            );

            return redirect()->back();
        }
    }

    public function render()
    {
        return view('livewire.auth.register')->layout('layouts.auth');
    }
}
