<?php

namespace App\Http\Livewire\User\Books;

use App\Models\Book;
use App\Models\BookStatus;
use App\Models\BookTitle;
use App\Models\User;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithPagination;
use Note\Note;

class ListBooks extends Component
{
    use WithPagination, FindGuard, LivewireAlert;

    public $search, $user, $delete;

    protected $queryString = ['search'];

    protected string $paginationTheme = 'bootstrap';

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public bool $readyToLoad = false;

    public function loadData()
    {
        $this->readyToLoad = true;
    }

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function submit($id)
    {
        $this->delete = Book::query()->firstWhere('id', $id);
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        //check if already enrolled
        $this->user = $this->findGuardType()->user();
        $bookStatus = BookStatus::updateOrCreate(
            [

                'user_id' => $this->user->id,
                'book_id' => $this->delete->id,
            ],
            [
                'is_preferred' => true,
            ]
        );
        Note::createSystemNotification(User::class, 'Livro', 'Marcado como preferido com sucesso' . $this->delete->title);
        $this->emit('noteAdded');
        $this->alert('success', 'Marcado como preferido com sucesso' . $this->delete->title);
        return back();
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function render()
    {
        return view(
            'livewire.user.books.list-books',
            [
                'models' => $this->readyToLoad
                    ? Book::query()
                    ->latest('updated_at')
                    ->where(function ($query) {
                        $query->orWhere('title', 'like', '%' . $this->search . '%')
                            ->orWhere('slug', 'like', '%' . $this->search . '%')
                              ->orWhere('author', 'like', '%' . $this->search . '%')
                                ->orWhere('description', 'like', '%' . $this->search . '%')
                                  ->orWhereRelation('category', 'name','like', '%' . $this->search . '%');
                    })
                    ->paginate(10)
                    : []
            ]
        );
    }
}
